/*
 * I.432.1 cell stream transmitter.  We are very similar to NOKFTX, but with
 * some simplifications as we've done away with the Nokia framing.
 *
 * We continuously transmit ATM cells out of a RAM buffer which holds 8 cells.
 * Each cell is expanded to 64 bytes in the buffer, with an 11 byte gap between
 * HEC and the payload, making the total buffer size 512 bytes.  The HEC value
 * in the RAM buffer should be 0 for normal operation, as it will be XORed with
 * the calculated HEC and with the coset automatically; write other values only
 * to produce deliberate errors.  Optional CRC-32 generation and idle cell
 * payload filling are just like in NOKFTX.
 *
 * SSS scrambling is deferred to a separate logic block.
 */

module xmitter (ram_addr, ram_data, BCLK, QCLK, SDSLout, payload_outgoing_now,
		reset_control, crc32_enable, idlecell_enable, hec_coset);

output [8:0] ram_addr;
input [7:0] ram_data;

input BCLK, QCLK;
output SDSLout, payload_outgoing_now;
reg payload_outgoing_now;

input reset_control, crc32_enable, idlecell_enable;
input [7:0] hec_coset;

/*
 * Note: all following registers describe the *next* octet to be
 * fetched from RAM, *not* the octet currently being shifted out!
 */
reg [2:0] bit_counter;
reg [5:0] byte_pointer;
reg [2:0] cell_no;

reg QCLK_latch;
reg reset_control_sync1, reset_control_sync2;

reg [7:0] shiftreg;
reg [7:0] crc8_accum;
wire [7:0] crc8_nextval;

reg current_cell_idle, current_cell_endpkt;
reg [31:0] crc32_accum;
wire [31:0] crc32_nextval;
reg crc32_outgoing_now;

always @(negedge BCLK)
	QCLK_latch <= QCLK;

always @(posedge BCLK)
	reset_control_sync1 <= reset_control;

always @(posedge BCLK)
	reset_control_sync2 <= reset_control_sync1;

always @(posedge BCLK)
	if (reset_control_sync2 && QCLK_latch)
		bit_counter <= 3'b0;
	else
		bit_counter <= bit_counter + 3'd1;

always @(posedge BCLK)
	if (reset_control_sync2)
		byte_pointer <= 6'b0;
	else if (bit_counter == 3'b111)
	begin
		if (byte_pointer == 6'd4)
			byte_pointer <= 6'd16;
		else
			byte_pointer <= byte_pointer + 6'd1;
	end

always @(posedge BCLK)
	if (reset_control_sync2)
		cell_no <= 3'b0;
	else if (bit_counter == 3'b111 && byte_pointer == 6'b111111)
		cell_no <= cell_no + 3'd1;

assign ram_addr = {cell_no,byte_pointer};

always @(posedge BCLK)
	if (bit_counter == 3'b111)
	begin
		if (byte_pointer == 6'd4)
			shiftreg <= ram_data ^ crc8_nextval ^ hec_coset;
		else if ((|byte_pointer[5:4]) && current_cell_idle &&
				idlecell_enable)
			shiftreg <= 8'h6A;
		else if ((&byte_pointer[5:2]) && current_cell_endpkt &&
				crc32_enable)
			shiftreg <= crc32_nextval[31:24];
		else
			shiftreg <= ram_data;
	end
	else
		shiftreg <= shiftreg << 1;

always @(posedge BCLK)
	if (bit_counter == 3'b111)
	begin
		if ((|byte_pointer[5:4]))
			payload_outgoing_now <= 1'b1;
		else
			payload_outgoing_now <= 1'b0;
	end

always @(posedge BCLK)
	if (bit_counter == 3'b111)
	begin
		if ((&byte_pointer[5:2]) && current_cell_endpkt &&
				crc32_enable)
			crc32_outgoing_now <= 1'b1;
		else
			crc32_outgoing_now <= 1'b0;
	end

always @(posedge BCLK)
	if (bit_counter == 3'b111 && byte_pointer == 6'd3)
	begin
		current_cell_endpkt <= ram_data[1];
		current_cell_idle   <= ram_data[0];
	end

assign SDSLout = shiftreg[7] ^ crc32_outgoing_now;

assign crc8_nextval = (crc8_accum << 1) ^
			(crc8_accum[7] ^ shiftreg[7] ? 8'b00000111 : 8'b0);

always @(posedge BCLK)
	if (byte_pointer == 6'b0 && bit_counter == 3'b111)
		crc8_accum <= 8'b00000000;
	else
		crc8_accum <= crc8_nextval;

assign crc32_nextval = (crc32_accum << 1) ^
			(crc32_accum[31] ^ shiftreg[7] ? 32'h04C11DB7 : 32'b0);

always @(posedge BCLK)
	if (payload_outgoing_now)
		crc32_accum <= crc32_nextval;
	else if (byte_pointer == 6'd2 &&
			(current_cell_endpkt || current_cell_idle))
		crc32_accum <= 32'hFFFFFFFF;

endmodule
