module FPGA (SYSCLK, M68K_addr, M68K_data, URDS, LRDS, UWRS, LWRS, CS_L, IRQ_L,
	BCLK, QCLK, SDSL_RDAT, SDSL_TDAT, DCE_TxD, DCE_RxD,
	CCITT_114, CCITT_115, RxAM, TxAM, dummy_inputs, PB11);

input SYSCLK;
input [12:1] M68K_addr;
inout [15:0] M68K_data;
input URDS, LRDS, UWRS, LWRS;
input CS_L;
output IRQ_L;

input BCLK;
input QCLK;
input SDSL_RDAT;
output SDSL_TDAT;
input DCE_TxD;
output DCE_RxD;

output CCITT_114;
output CCITT_115;

output RxAM;
input TxAM;

input [2:0] dummy_inputs;
output PB11;

/* M68K address space decoding */

wire reg_block_select, rx_block_select, txram_block_select;

assign reg_block_select   = !CS_L && !M68K_addr[10] && !M68K_addr[9];
assign rx_block_select    = !CS_L && !M68K_addr[10] &&  M68K_addr[9];
assign txram_block_select = !CS_L &&  M68K_addr[10] && !M68K_addr[9];

/* Wires between blocks */

wire rx_reset_control, rx_sync_state, rx_data_pending;
wire tx_reset_control, QCLK_control, tx_int_en_ctrl, tx_host_buf_ctrl;
wire tx_buf_sense;
wire effective_QCLK;
wire crc32_enable, idlecell_enable, SSS_enable;
wire [7:0] hec_coset;

assign effective_QCLK = QCLK ^ QCLK_control;

wire [15:0] M68K_reg_read_data, M68K_rx_read_data, M68K_txram_read_data;

/* Rx side wires */

wire [7:0] rx_byte;
wire [5:0] rx_pos;
wire rx_strobe, rx_hec_error;
wire rx_interrupt_strobe;

/* Tx side wires */

wire [8:0] backport_read_addr;
wire [7:0] backport_read_data;

assign tx_buf_sense = backport_read_addr[8];

wire tx_bit_intermediate, tx_bit_payload;

/* Block instantiations */

regblock regblock (SYSCLK, M68K_addr[2:1], M68K_data, UWRS, LWRS,
	reg_block_select, M68K_reg_read_data, SSS_enable, hec_coset,
	rx_reset_control, rx_sync_state, rx_data_pending,
	tx_reset_control, QCLK_control, tx_int_en_ctrl, tx_host_buf_ctrl,
	tx_buf_sense, crc32_enable, idlecell_enable);

rxinterf rxinterf (SYSCLK, M68K_addr[8:1], rx_block_select, UWRS, LWRS,
	M68K_rx_read_data, rx_reset_control,
	rx_byte, rx_pos, rx_strobe, rx_hec_error,
	rx_data_pending, rx_interrupt_strobe);

bitrx bitrx (BCLK, SDSL_RDAT, rx_reset_control, SSS_enable, hec_coset,
	rx_sync_state, rx_byte, rx_pos, rx_strobe, rx_hec_error);

txram16 txram16 (SYSCLK, M68K_addr[8:1], M68K_data, UWRS, LWRS,
	txram_block_select, M68K_txram_read_data,
	backport_read_addr, backport_read_data);

xmitter xmitter (backport_read_addr, backport_read_data, BCLK, effective_QCLK,
	tx_bit_intermediate, tx_bit_payload, tx_reset_control, crc32_enable,
	idlecell_enable, hec_coset);

txsss txsss (SSS_enable, BCLK, tx_bit_intermediate, tx_bit_payload, SDSL_TDAT);

/* Driving the bidirectional M68K data bus */

reg [15:0] M68K_data_out;

always @*
	case (M68K_addr[10:9])
	2'b00:
		M68K_data_out = M68K_reg_read_data;
	2'b01:
		M68K_data_out = M68K_rx_read_data;
	2'b10:
		M68K_data_out = M68K_txram_read_data;
	default:
		M68K_data_out = 16'h0000;
	endcase

assign M68K_data[15:8] = !CS_L && !URDS ? M68K_data_out[15:8] : 8'bzzzzzzzz;
assign M68K_data[7:0]  = !CS_L && !LRDS ? M68K_data_out[7:0]  : 8'bzzzzzzzz;

/* Interrupt output */

assign IRQ_L = !((tx_host_buf_ctrl == tx_buf_sense) && tx_int_en_ctrl);
assign PB11 = !rx_interrupt_strobe;

/* Unused outputs */
assign DCE_RxD = 1'b1;
assign CCITT_114 = 1'b0;
assign CCITT_115 = 1'b0;
assign RxAM = 1'b0;

endmodule
