/*
 * This is the high-level part of the Rx logic.
 *
 * Registers following the Rx cell view:
 *
 * offset 40: ACK virtual register
 * offset 42: overrun counter
 * offset 44: debug register (internal read/write pointers)
 */

module rxinterf (SYSCLK, M68K_addr, block_select, UWRS, LWRS, M68K_data_out,
	reset_control,
	rx_byte, rx_pos, rx_strobe_in, hec_error,
	data_pending, interrupt_strobe);

input SYSCLK;
input [8:1] M68K_addr;
input block_select, UWRS, LWRS;
output [15:0] M68K_data_out;
reg [15:0] M68K_data_out;

input reset_control;
input [7:0] rx_byte;
input [5:0] rx_pos;
input rx_strobe_in, hec_error;

output data_pending, interrupt_strobe;

reg rx_strobe_sync1, rx_strobe_sync2;

always @(posedge SYSCLK)
	rx_strobe_sync1 <= rx_strobe_in;

always @(posedge SYSCLK)
	rx_strobe_sync2 <= rx_strobe_sync1;

wire [9:1] ram_read_addr;
wire [15:0] ram_read_data;
wire [9:0] ram_write_addr;
wire [7:0] ram_write_byte;
wire ram_write_done;

rxram16 ram (SYSCLK, ram_read_addr, block_select && !M68K_addr[6],
	ram_read_data, rx_strobe_sync2, ram_write_done, ram_write_addr,
	ram_write_byte);

assign ram_write_byte = rx_pos == 6'd4 ? {7'b0,hec_error} : rx_byte;

reg [3:0] cell_wr_ptr, cell_rd_ptr;
wire [3:0] cell_wr_next, cell_rd_next;

assign cell_wr_next = cell_wr_ptr + 4'd1;
assign cell_rd_next = cell_rd_ptr + 4'd1;

assign ram_read_addr = {cell_rd_ptr,M68K_addr[5:1]};
assign ram_write_addr = {cell_wr_ptr,rx_pos};

always @(posedge SYSCLK)
	if (reset_control)
		cell_rd_ptr <= 4'b0;
	else if (block_select && (!UWRS || !LWRS) && M68K_addr[6] &&
			!M68K_addr[2] && !M68K_addr[1])
		cell_rd_ptr <= cell_rd_next;

wire final_byte_written, final_byte_written_pulse;
reg final_byte_written_reg;

assign final_byte_written = ram_write_done && rx_pos == 6'b111111;

always @(posedge SYSCLK)
	final_byte_written_reg <= final_byte_written;

assign final_byte_written_pulse = final_byte_written && !final_byte_written_reg;

wire overrun_event;

assign overrun_event = final_byte_written_pulse && cell_wr_next == cell_rd_ptr;

always @(posedge SYSCLK)
	if (reset_control)
		cell_wr_ptr <= 4'b0;
	else if (final_byte_written_pulse)
	begin
		if (overrun_event)
			cell_wr_ptr <= cell_rd_next;
		else
			cell_wr_ptr <= cell_wr_next;
	end

reg [15:0] overrun_counter;

always @(posedge SYSCLK)
	if (block_select && (!UWRS || !LWRS) && M68K_addr[6] &&
			!M68K_addr[2] && M68K_addr[1])
		overrun_counter <= 16'd0;
	else if (overrun_event)
		overrun_counter <= overrun_counter + 16'd1;

assign data_pending = cell_rd_ptr != cell_wr_ptr;
assign interrupt_strobe = final_byte_written_reg;

/* M68K data bus output */

always @*
	casez (M68K_addr[6:1])
	6'b0?????:
		M68K_data_out = ram_read_data;
	6'b1???01:
		M68K_data_out = overrun_counter;
	6'b1???10:
		begin
			M68K_data_out[15:12] = 4'b0;
			M68K_data_out[11:8] = cell_wr_ptr;
			M68K_data_out[7:4] = 4'b0;
			M68K_data_out[3:0] = cell_rd_ptr;
		end
	default:
		M68K_data_out = 16'b0;
	endcase

endmodule
