/*
 * Register layout for the I.432.1 engine:
 *
 * offset 00: FPGA ID word
 * offset 02: flavor control
 * offset 04: Rx CSR
 * offset 06: Tx CSR
 *
 * Flavor control register bit assignments:
 *
 * <8>   SSS enable
 * <7:0> HEC coset
 *
 * Rx CSR bit assignments:
 *
 * <15> Reset/rehunt
 * <1>  Sync state
 * <0>  Data pending
 *
 * Tx CSR bit assignments:
 *
 * <15>	Reset
 * <14>	QCLK invert
 * <9>  CRC-32 enable
 * <8>  idle cell fill enable
 * <7>	Interrupt enable
 * <4>	Buffer last written
 * <3>	Buffer currently transmitted
 */

module regblock (SYSCLK, M68K_addr, M68K_data_in, UWRS, LWRS, block_select,
	M68K_reg_read_data,
	SSS_enable, hec_coset,
	rx_reset_control, rx_sync_state, rx_data_pending,
	tx_reset_control, QCLK_control, tx_int_en_ctrl, tx_host_buf_ctrl,
	tx_buf_sense, crc32_enable, idlecell_enable);

input SYSCLK;
input [2:1] M68K_addr;
input [15:0] M68K_data_in;
input UWRS, LWRS;
input block_select;

output [15:0] M68K_reg_read_data;
reg [15:0] M68K_reg_read_data;

output SSS_enable;
reg SSS_enable;
output [7:0] hec_coset;
reg [7:0] hec_coset;

output rx_reset_control;
reg rx_reset_control;
input rx_sync_state, rx_data_pending;

output tx_reset_control, QCLK_control, tx_int_en_ctrl, tx_host_buf_ctrl;
reg tx_reset_control, QCLK_control, tx_int_en_ctrl, tx_host_buf_ctrl;
input tx_buf_sense;
output crc32_enable, idlecell_enable;
reg crc32_enable, idlecell_enable;

/* Register write logic */

always @(posedge SYSCLK)
	if (block_select && !M68K_addr[2] && M68K_addr[1] && !UWRS)
	begin
		SSS_enable <= M68K_data_in[8];
	end

always @(posedge SYSCLK)
	if (block_select && !M68K_addr[2] && M68K_addr[1] && !LWRS)
	begin
		hec_coset <= M68K_data_in[7:0];
	end

always @(posedge SYSCLK)
	if (block_select && M68K_addr[2] && !M68K_addr[1] && !UWRS)
	begin
		rx_reset_control <= M68K_data_in[15];
	end

always @(posedge SYSCLK)
	if (block_select && M68K_addr[2] && M68K_addr[1] && !UWRS)
	begin
		tx_reset_control <= M68K_data_in[15];
		QCLK_control <= M68K_data_in[14];
		crc32_enable <= M68K_data_in[9];
		idlecell_enable <= M68K_data_in[8];
	end

always @(posedge SYSCLK)
	if (block_select && M68K_addr[2] && M68K_addr[1] && !LWRS)
	begin
		tx_int_en_ctrl <= M68K_data_in[7];
		tx_host_buf_ctrl <= M68K_data_in[4];
	end

/* Register read logic */

always @*
	case (M68K_addr[2:1])
	2'b00:
		M68K_reg_read_data = 16'h4901;
	2'b01:
		begin
			M68K_reg_read_data[15:9] = 7'b0;
			M68K_reg_read_data[8] = SSS_enable;
			M68K_reg_read_data[7:0] = hec_coset;
		end
	2'b10:
		begin
			M68K_reg_read_data[15] = rx_reset_control;
			M68K_reg_read_data[14] = 1'b0;
			M68K_reg_read_data[13] = 1'b0;
			M68K_reg_read_data[12] = 1'b0;
			M68K_reg_read_data[11] = 1'b0;
			M68K_reg_read_data[10] = 1'b0;
			M68K_reg_read_data[9] = 1'b0;
			M68K_reg_read_data[8] = 1'b0;
			M68K_reg_read_data[7] = 1'b0;
			M68K_reg_read_data[6] = 1'b0;
			M68K_reg_read_data[5] = 1'b0;
			M68K_reg_read_data[4] = 1'b0;
			M68K_reg_read_data[3] = 1'b0;
			M68K_reg_read_data[2] = 1'b0;
			M68K_reg_read_data[1] = rx_sync_state;
			M68K_reg_read_data[0] = rx_data_pending;
		end
	2'b11:
		begin
			M68K_reg_read_data[15] = tx_reset_control;
			M68K_reg_read_data[14] = QCLK_control;
			M68K_reg_read_data[13] = 1'b0;
			M68K_reg_read_data[12] = 1'b0;
			M68K_reg_read_data[11] = 1'b0;
			M68K_reg_read_data[10] = 1'b0;
			M68K_reg_read_data[9] = crc32_enable;
			M68K_reg_read_data[8] = idlecell_enable;
			M68K_reg_read_data[7] = tx_int_en_ctrl;
			M68K_reg_read_data[6] = 1'b0;
			M68K_reg_read_data[5] = 1'b0;
			M68K_reg_read_data[4] = tx_host_buf_ctrl;
			M68K_reg_read_data[3] = tx_buf_sense;
			M68K_reg_read_data[2] = 1'b0;
			M68K_reg_read_data[1] = 1'b0;
			M68K_reg_read_data[0] = 1'b0;
		end
	default:
		M68K_reg_read_data = 16'hxxxx;
	endcase

endmodule
