/*
 * This logic module receives the SDSL bit stream and looks for ATM cells.
 *
 * We do everything @(negedge BCLK) rather than @(posedge BCLK) because
 * according to the RS8973 datasheet the timing relationship between RDAT
 * changing state and the BCLK rising edge on which this change occurs
 * is not reliable.
 *
 * Rx side SSS is implemented here as well.
 */

module bitrx (BCLK, RDAT, reset_control, SSS_enable, hec_coset,
		sync_state, rx_byte, rx_pos, rx_strobe, hec_error);

input BCLK, RDAT;
input reset_control, SSS_enable;
input [7:0] hec_coset;

output sync_state, rx_strobe;
reg sync_state, rx_strobe;
output [7:0] rx_byte;
reg [7:0] rx_byte;
output [5:0] rx_pos;
reg [5:0] rx_pos;
output hec_error;

reg reset_control_sync1, reset_control_sync2;

reg [2:0] bit_counter;
reg [7:1] incoming_shift_reg;
wire [7:0] incoming_byte;
reg [5:0] incoming_pos;
reg [7:0] crc8_accum;
wire hec_check_position, retry41;

wire RDAT_SSS;
reg [42:0] SSS_reg;

always @(negedge BCLK)
	reset_control_sync1 <= reset_control;

always @(negedge BCLK)
	reset_control_sync2 <= reset_control_sync1;

always @(negedge BCLK)
	if (reset_control_sync2)
		bit_counter <= 3'b111;
	else if (retry41)
		bit_counter <= 3'b000;
	else
		bit_counter <= bit_counter + 3'd1;

always @(negedge BCLK)
	if (reset_control_sync2)
		incoming_pos <= 6'b111111;
	else if (retry41)
		incoming_pos <= 6'b000000;
	else if (bit_counter == 3'b111)
	begin
		if (incoming_pos == 6'd4)
			incoming_pos <= 6'd16;
		else
			incoming_pos <= incoming_pos + 6'd1;
	end

always @(negedge BCLK)
	if (|incoming_pos[5:4])
		SSS_reg <= {SSS_reg[41:0],RDAT};

assign RDAT_SSS = SSS_enable && (|incoming_pos[5:4]) ?
			RDAT ^ SSS_reg[42] : RDAT;

always @(negedge BCLK)
	incoming_shift_reg <= {incoming_shift_reg[6:1],RDAT_SSS};

assign incoming_byte = {incoming_shift_reg,RDAT_SSS};

always @(negedge BCLK)
	if (bit_counter == 3'b111)
	begin
		rx_byte <= incoming_byte;
		rx_pos  <= incoming_pos;
	end

always @(negedge BCLK)
	if (incoming_pos == 6'b111111 || retry41)
		crc8_accum <= 8'b00000000;
	else if (incoming_pos[5:2] == 4'b0000)
		crc8_accum <= (crc8_accum << 1) ^
			(crc8_accum[7] ^ RDAT ? 8'b00000111 : 8'b0);

/* The following signal is valid *only* when rx_pos == 6'd4! */
assign hec_error = |(rx_byte ^ crc8_accum ^ hec_coset);

always @(negedge BCLK)
	if (reset_control_sync2)
		sync_state <= 1'b0;
	else if (hec_check_position && !hec_error)
		sync_state <= 1'b1;

assign hec_check_position = incoming_pos == 6'd16 && bit_counter == 3'b000;

assign retry41 = hec_check_position && !sync_state && hec_error;

always @(negedge BCLK)
	rx_strobe <= bit_counter[2:1] == 2'b01;

endmodule
