#include "types.h"
#include "stdio.h"
#include "mc68302.h"
#include "cp.h"
#include "rt11ffs.h"
#include "memstruct.h"
#include "xmodem.h"

extern volatile struct mc68302_regs mc68302_regs;
extern volatile struct mc68302_scc_bd scc3_rx_bds[8];
extern volatile struct mc68302_scc_uartparams scc3_params;
extern volatile u_char _getchar_buffer[8];
extern u_short _getchar_headptr;

extern int blocks_in_buffer;
int xmodem_blocks_rcvd;

static struct xmodem_packet rx_buffers[8] __attribute__ ((aligned (2)));
static int cur_rx_buf;

stop_scc3_rx()
{
	mc68302_regs.scc3.scm &= ~SCM_ENR;
	send_cp_command(CR_SCC3, CPCOM_ENTER_HUNT);
}

setup_scc3_for_xmodem()
{
	int i;

	for (i = 0; i < 8; i++) {
		scc3_rx_bds[i].bd_flags = 0xC000;
		scc3_rx_bds[i].bd_bufptr = &rx_buffers[i];;
	}
	scc3_rx_bds[7].bd_flags = 0xE000;
	scc3_params.rbdn = 0;
	cur_rx_buf = 0;
	scc3_params.mrblr = sizeof(struct xmodem_packet);
	scc3_params.max_idl = 960;	/* 1 s */
	mc68302_regs.scc3.scm |= SCM_ENR;
}

restore_scc3()
{
	int i;

	for (i = 0; i < 8; i++) {
		scc3_rx_bds[i].bd_flags = 0x8000;
		scc3_rx_bds[i].bd_bufptr = _getchar_buffer + i;
	}
	scc3_rx_bds[7].bd_flags = 0xA000;
	scc3_params.rbdn = 0;
	_getchar_headptr = 0;
	scc3_params.mrblr = 1;
	scc3_params.max_idl = 1;
	mc68302_regs.scc3.scm |= SCM_ENR;
}

xmodem_upload()
{
	u_short flags;
	struct xmodem_packet *pkt;
	u_char last_blknum, c, *cp, *ep;
	u_char *dptr;
	int pad;

	stop_scc3_rx();
	setup_scc3_for_xmodem();
	printf("\r\nStart XMODEM upload now or press Control-X to cancel\r\n");
	putchar(NAK);
	for (xmodem_blocks_rcvd = 0, dptr = (u_char *)BUFFER_BASE; ; ) {
		do
			flags = scc3_rx_bds[cur_rx_buf].bd_flags;
		while (flags & 0x8000);
		pkt = &rx_buffers[cur_rx_buf];
		cur_rx_buf++;
		if (cur_rx_buf >= 8)
			cur_rx_buf = 0;
		if (flags & 0x0032) {	/* errors */
			putchar(NAK);
			continue;
		}
		if (pkt->soh == CAN) {
			stop_scc3_rx();
			printf("^X abort received\r\n");
			restore_scc3();
			return(-1);
		}
		if (pkt->soh == EOT) {
			putchar(ACK);
			break;
		}
		if (flags & 0x0100) {	/* inter-byte timeout */
			putchar(NAK);
			continue;
		}
		if (pkt->soh != SOH) {
			putchar(NAK);
			continue;
		}
		if (pkt->blknum_1c != ~pkt->blknum) {
			putchar(NAK);
			continue;
		}
		last_blknum = xmodem_blocks_rcvd & 0xFF;
		if (pkt->blknum == last_blknum) {
			putchar(ACK);
			continue;
		} else if (pkt->blknum == last_blknum + 1) {
			if (xmodem_blocks_rcvd >= MAX_XMODEM_BLOCKS) {
				stop_scc3_rx();
				printf("\030\030File size limit exceeded\r\n");
				restore_scc3();
				return(-1);
			}
			for (c = 0, cp = pkt->data, ep = &pkt->chksum; cp < ep;)
				c += *cp++;
			if (c != pkt->chksum) {
				putchar(NAK);
				continue;
			}
			bcopy(pkt->data, dptr, XMODEM_BLOCK_SIZE);
			xmodem_blocks_rcvd++;
			dptr += XMODEM_BLOCK_SIZE;
			putchar(ACK);
			continue;
		} else {
			stop_scc3_rx();
			printf("\030\030Block number out of sequence\r\n");
			restore_scc3();
			return(-1);
		}
	}
	stop_scc3_rx();
	restore_scc3();
	if (!xmodem_blocks_rcvd) {
		printf("Nothing uploaded\r\n");
		return(-1);
	}
	if (xmodem_blocks_rcvd & 3) {
		pad = 4 - (xmodem_blocks_rcvd & 3);
		bzero(dptr, pad * XMODEM_BLOCK_SIZE);
	} else
		pad = 0;
	blocks_in_buffer = (xmodem_blocks_rcvd + pad) / 4;
	printf("Received %u XMODEM block%s (%u RT11FFS block%s)\r\n",
		xmodem_blocks_rcvd, xmodem_blocks_rcvd != 1 ? "s" : "",
		blocks_in_buffer, blocks_in_buffer != 1 ? "s" : "");
	return(0);
}
