#include "../libc/types.h"
#include "../lib8973/typedefs.h"
#include "../lib8973/config.h"
#include "../lib8973/bitpump.h"
#include "../lib8973/util.h"
#include "../lib8973/bpstate.h"
#include "../lib8973/defmeter.h"
#include "../lib8973/timermac.h"
#include "state.h"
#include "privstate.h"
#include "miscmac.h"

/*---------------------------------------------------------*/
/*  Local Defines                                          */
/*---------------------------------------------------------*/

#define LOW_RATE_FFE_INIT      72   /* 576kbps */

#define LONG_LOOP_TH           600

#define NUM_CANCEL_FALSE_PHASES     3
#define NUM_CANCEL_NOISE_PHASES     5

#define MINIMUM_FELM_LOW_TH   100

/* Far End Signal Thresholds */
#define LONG_LOOP_BT8970                        1500
#define MEDIUM_LOOP_BT8970                      3000
#define SHORT_LOOP_BT8970                       7700

/*---------------------------------------------------------*/
/*  Constants                                              */
/*---------------------------------------------------------*/

static BP_U_8BIT BP_CONSTANT phase_delta[4] = {2, 2, 2, 3};
static BP_U_8BIT BP_CONSTANT phase_quality_scale[4] = {7, 6, 3, 2};

BP_U_8BIT BP_CONSTANT dagc_th[4] = {12, 12, 6, 5};

BP_S_16BIT BP_CONSTANT ffe_init[2][4][8] =

/* High Data Rates */
  {
   {{-1119,  1882, -3545, 5606,  -9171, 13685,  -6958,    97 },
    { -917,  1396, -2644,  3916, -6382,  9108,  -3345,  -460 },
    { -451,   500, -1139,  1624, -2988,  4492,  -1212,  -199 },
    { -224,   160,  -525,   678, -1497,  2362,   -387,    34 } },
/* Low Data Rates < 580kbps */
   {{-1028,  980, -1899,  3144, -7473, 15432, -11710,  2855 },
    { -635,   586, -1381,  1867, -3884,  6088,  -2744,    73 },
    { -557,   407, -1214,  1277, -2796,  3535,   -597,  -174 },
    {  -14,     8,   -78,   190,  -710,  2164,    825,   412 } } };

/***********************************************************/
/*    _CalculateOptimalPhase()                             */
/*    Calculate optimal phase, based on eye opening        */
/*    results at 16 sampling phases.                       */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*          status = _CalculateOptimalPhase(no);           */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                24-Oct-1993              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/

void
_CalculateOptimalPhase1(st)
    struct sdcore_state *st;
{
    DECLARE_PST;
    BP_U_16BIT min_value;
    BP_U_8BIT min_index;
    BP_S_8BIT index, j;


    /*
     * Write 3 (n) lowest phase quality values to 0, must mark them
     * as 0xFFFF first.
     */
#ifdef TDEBUG
    printf("Selected as BAD Phase Quality: ");
#endif

    for ( j = 0 ; j < NUM_CANCEL_FALSE_PHASES ; j++ ) {
        min_value = 0xFFFF;
        min_index = 0;
        for (index = 0; index < NPHASES; index++) {
            if (pst->phase_quality[index] < min_value) {
                min_index = index;
                min_value = pst->phase_quality[index];
            }
        }  /* end for index */

	pst->phase_quality[min_index] = 0xFFFF;
#ifdef TDEBUG
        printf("%d(%d) ", (int)min_index, (int)min_value);
#endif
    }  /* end for j */

} /* END _CalculateOptimalPhase1() */

void
_CalculateOptimalPhase2(st)
    struct sdcore_state *st;
{
    DECLARE_PST;
    int index;

    /* Now set 3 lowest to 0 */
    for (index = 0; index < NPHASES; index++) {
	if (pst->phase_quality[index] == 0xFFFF)
		pst->phase_quality[index] = 0;
    }

} /* END _CalculateOptimalPhase2() */

void
_CalculateOptimalPhase3(st)
    struct sdcore_state *st;
{
    DECLARE_PST;
    BP_U_16BIT temp_value;
    BP_U_8BIT scale;
    BP_S_8BIT index, j;

    scale = pst->su_flags.bits.ffe_init_scale;

    /*------------------------------------------------*/
    /* Remove Phases near 0's (possible false phases) */
    /*------------------------------------------------*/
    for (index = 0; index < NPHASES; index++) {

	temp_value = pst->phase_quality[index];

        if (temp_value != 0) {
            for (j = -phase_delta[scale]; j <= phase_delta[scale]; j++) {
		temp_value = pst->phase_quality[(index+j+NPHASES) % NPHASES];
                if (temp_value == 0) {
#ifdef TDEBUG
                    printf("%d(1) ", (int)index);
#endif
		    pst->phase_quality[index] = 1;
                    break;
                } /* END-IF */              
            } /* END-FOR */
        } /* END-IF */
    } /* END-FOR */

#ifdef TDEBUG    
    printf("\r\n");
#endif

} /* END _CalculateOptimalPhase3() */


void
_CalculateOptimalPhase4(st)
    struct sdcore_state *st;
{
    DECLARE_PST;
    BP_U_16BIT max_value;
    BP_U_8BIT max_index;
    BP_S_8BIT index, j;

    /*--------------------------------*/
    /* Remove High Noise (NLM) Phases */
    /*--------------------------------*/
#ifdef TDEBUG
    printf("Selected as High Noise Phases: ");
#endif

    for ( j = 0 ; j < NUM_CANCEL_NOISE_PHASES ; j++ ) {
        max_value = 0;
        max_index = 0;
        for (index = 0; index < NPHASES; index++) {
            if (pst->noise_quality[index] > max_value) {
                max_index = index;
                max_value = pst->noise_quality[index];
            }
        }  /* end for index */

	pst->noise_quality[max_index] = 1;
	pst->phase_quality[max_index] = 1;
#ifdef TDEBUG
        printf("%d(%d) ", (int)max_index, (int)max_value);
#endif
    }  /* end for j */

} /* END _CalculateOptimalPhase4() */

int
_CalculateOptimalPhase(st)
    struct sdcore_state *st;
{
    DECLARE_PST;
    BP_U_16BIT temp_value, max_value;
    BP_U_8BIT opt_phase;
    BP_S_8BIT index, j;
    BP_U_8BIT opt_value, value;

#ifdef TDEBUG
    printf("\r\nPhases = [");
#endif
    
    /*----------------------------------------------*/
    /* Find optimal phase, peak phase quality left  */
    /*----------------------------------------------*/
    max_value = 0;
    opt_phase = 0;
    for (index = 0; index < NPHASES; index++) {
	temp_value = pst->phase_quality[index]; /* Read phase quality */

#ifdef TDEBUG
        printf("%d ", (int)temp_value);
#endif

        if (temp_value > max_value) {
            max_value =  temp_value;
            opt_phase = index;
        } /* END-IF */
    } /* END-FOR */

#ifdef TDEBUG
    printf("]\r\n");
#endif

    /*------------------------------------------*/
    /* If no good (peak) Phase Quality is found */
    /* use the center of island approach        */
    /*------------------------------------------*/
    if (max_value == 0x1) {

#ifdef TDEBUG
	printf("Using center of island approach\r\n");
#endif

        value = opt_value = 0;

        for (j=0;  j<=1; j++) /* Check the phases success twice due to rewind phases */
            {
            for (index=15; index>=0; index--) /* Check sixteen available phases */
                {
		temp_value = pst->phase_quality[index];
                if (temp_value == 0x1) /* Good phase */
                    {
                    value++;
                    }
                else /* Bad phase */
                    {
                    if (value > opt_value)
                        {
                        opt_value = value; /* Save current value */
                        opt_phase = index + 1;
                        } /* END-IF */
                    value = 0; /* Look for next flow of success phases */
                    } /* END-ELSE bad phase */
                } /* END-FOR check sixteen available phases */
            } /* END-FOR check phase success twice */
    
        /*-------------------*/
        /* Set optimal phase */
        /*-------------------*/
        opt_phase = opt_phase + (opt_value >> 1); /* Calculate optimal phase */
        if (opt_phase > 15)
            {
            opt_phase -= 16;
            }
    } /* END-IF  */

#ifdef TDEBUG
    printf("Optimal phase = %d\r\n", (int)opt_phase);
#endif

    return (opt_phase);

} /* END _CalculateOptimalPhase() */



/***********************************************************/
/*    _SetFelmMask()                                       */
/*                                                         */
/*    Description:  This function set the HIGH or LOW      */
/*                  FELM mask off depending on the LOS bit.*/
/*                                                         */
/*                  Both HIGH/LOW masks are ON before      */
/*                  entering this function.                */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     Dean Rasmussen             04-June-1997             */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void
_SetFelmMask(st)
    struct sdcore_state *st;
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;

    if (st->status.bits.los) {
	bp_ptr->irq_source = ~IRQMASK_HIGHFELM;
        BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, high_felm, OFF);
    } else {
	bp_ptr->irq_source = ~IRQMASK_LOWFELM;
        BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, low_felm, OFF);
    } /* END-ELSE */
}



/***********************************************************/
/*    _CalculatePhaseQuality()                             */
/*                                                         */
/*    returns: BP_S_16BIT temp_p (phase quality)           */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                16-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/

BP_S_16BIT
_CalculatePhaseQuality(st)
    struct sdcore_state *st;
{
    DECLARE_PST;
    BP_U_8BIT dagc;
    BP_S_8BIT i, scale;
    BP_S_16BIT temp_p,max_ffe;
    BP_S_32BIT dfe_p, ffe_p;
    u_char data[2];

    scale = pst->su_flags.bits.ffe_init_scale;

    bp_eq_read(DAGC_HIGH, data);
    dagc = (BP_U_8BIT) data[1]; /* High Byte Only */

    max_ffe = 0;
    for (i = 4; i < 8; i++) {
	bp_eq_read(i, data);
        temp_p = (BP_S_16BIT) BYTE2WORD(data[1], data[0]);
        if (abs(temp_p) > max_ffe)
            max_ffe = abs(temp_p);
    }

#ifdef TDEBUG
    printf("F = %d, D = %d ",max_ffe,((int)dagc)<<4);
#endif

    if ( max_ffe < (BP_S_16BIT)10000 && ((BP_S_16BIT)dagc)<<4 > 1800 )
        return 0;



    if ((abs(max_ffe) > 1500) && (dagc > dagc_th[scale])) { /* FFE OK */
        /* ----------- */
        /* FFE Measure */
        /* ----------- */
        ffe_p=0;
        for (i=0; i<8; i+=2) {
	    bp_eq_read(i, data);
            ffe_p += (BP_S_32BIT)((BP_S_16BIT)BYTE2WORD(data[1], data[0]) >> 4);
        }
        for (i=1; i<8; i+=2) {
	    bp_eq_read(i, data);
            ffe_p -= (BP_S_32BIT)((BP_S_16BIT)BYTE2WORD(data[1], data[0]) >> 4);
        }
        if (ffe_p<0)
            ffe_p = -ffe_p;

        /* ----------- */
        /* DFE Measure */
        /* ----------- */
        dfe_p=0;
        for (i=0; i<20; i+=2) {
	    bp_dfe_read(i, data);
            dfe_p += (BP_S_32BIT)((BP_S_16BIT)BYTE2WORD(data[1], data[0]) >> 4);
        }
        for (i=1; i<20; i+=2) {
	    bp_dfe_read(i, data);
            dfe_p -= (BP_S_32BIT)((BP_S_16BIT)BYTE2WORD(data[1], data[0]) >> 4);
        }
        dfe_p -= 1024;      /* consider cursor level */
        if (dfe_p<0)
            dfe_p = -dfe_p;


        /* ------------- */
        /* Total Measure */
        /* ------------- */

        dfe_p <<= 8;
        ffe_p *= dagc;
        ffe_p >>= phase_quality_scale[scale];
        if (ffe_p == 0)
            ffe_p = 1;
        dfe_p /= ffe_p;
        temp_p =  (BP_S_16BIT) dfe_p;
    } else
        temp_p = 0;

    return (temp_p); /* Return phase */

} /* END _CalculatePhaseQuality */


/***********************************************************/
/*    _InitFfe()                                           */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void
_InitFfe(st)
    struct sdcore_state *st;
{
    DECLARE_PST;
    DECLARE_MODE_PTR;
    BP_U_8BIT index;
    u_char data[2];

    /* Freeze FFE */
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, OFF);

    for ( index = 0 ; index < 8; index++ ) {
	data[0] = LOW(ffe_init[pst->su_flags.bits.ffe_data_scale][pst->su_flags.bits.ffe_init_scale][index]);
	data[1] = HIGH(ffe_init[pst->su_flags.bits.ffe_data_scale][pst->su_flags.bits.ffe_init_scale][index]);
	bp_eq_write(index, data);
    }

} /* END _InitFfe() */


/***********************************************************/
/*    _ReOpenOptimalPhase()                                */
/*    Uses same algorithm as optimal phase search          */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables:                                    */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                20-June-1995             */
/*     Dean Rasmussen             01-Oct-1998              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void
_ReOpenOptimalPhase(st)
    struct sdcore_state *st;
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;

    /* ------------------------------ */
    /* Initialize DAGC, DFE, and FFE: */
    /* ------------------------------ */
    _InitDagc();
    _InitDfe();
    _InitFfe(st);

    /* ---------------------------------- */
    /* Adapt DAGC @ HIGH Gain, Use Slicer */
    /* ---------------------------------- */
    _InitDetector();
    _AdaptDagcEqErrorMode();

    SET_WORD(bp_ptr, t3_low, t3_high, 5000);
    RESTART_T3;
    WAIT_FOR_TIMER(TIMERMASK_T3);

    BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF); /* Freeze DAGC coefficient */

    /* ---------- */
    /* Adapt FFE: */
    /* ---------- */

    _AdaptAllFfe();
    SET_WORD(bp_ptr, t3_low, t3_high, 10000);
    RESTART_T3;
    WAIT_FOR_TIMER(TIMERMASK_T3);

    /* -------------- */
    /* Adapt FFE+DFE: */
    /* -------------- */

    _AdaptDfe();
    SET_WORD(bp_ptr, t3_low, t3_high, 10000);
    RESTART_T3;
    WAIT_FOR_TIMER(TIMERMASK_T3);

    _FreezeDfeFfe();

#ifdef TDEBUG
    printf("ReOpen Eye\r\n");
#endif

} /* End _ReOpenOptimalPhase() */


/***********************************************************/
/*    _Aagc()                                              */
/*                                                         */
/*    returns: none                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                     BP_U_8BIT gain;                     */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-May-1995              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void
_Aagc(struct sdcore_state *st, BP_U_8BIT gain)
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;
    DECLARE_PST;
    BP_U_8BIT low, high;

    BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, low_felm, ON); /* Mask low FELM int. */
    BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, high_felm, ON); /* Mask high FELM int. */

    high = HIGH_FELM_ALARM_TH;
    low = LOW_FELM_ALARM_TH;
    switch (gain)
        {
        case AGAIN0DB:
            break;
        case AGAIN3DB:
            high = high + (high >> 1);
            low = low  + (low >> 1);
            break;
        case AGAIN6DB:
            high <<= 1;
            low <<= 1;
            break;
        case AGAIN9DB:
        case AGAIN12DB:
        case AGAIN15DB:
            high = (high << 1) + high;
            low = (low << 1) + low;
            break;
        } /* End switch */

    /* Use BP_WRITE_REG instead of SET_WORD because "high" and "low" are bytes and
     * not words. BP_WRITE_REG also saves ROM and eliminates compiler warnings.
     */
    BP_WRITE_REG(bp_ptr, far_end_high_alarm_th_low, high);
    BP_WRITE_REG(bp_ptr, far_end_high_alarm_th_high, 0x00);
    BP_WRITE_REG(bp_ptr, far_end_low_alarm_th_low, low);
    BP_WRITE_REG(bp_ptr, far_end_low_alarm_th_high, 0x00);

    pst->high_felm_threshold = high;
    pst->low_felm_threshold = low;

    BP_WRITE_BIT(bp_mode_ptr, adc_control, again, gain);

} /* END _Aagc() */



void
_SetLosThreshold(st)
    struct sdcore_state *st;
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;
    DECLARE_PST;
    BP_U_8BIT meter;
    BP_U_16BIT value;

    READ_METER_REG(felm_low, felm_high, value, 0);
    if (value >= MINIMUM_FELM_LOW_TH) {
#ifdef TDEBUG
	printf("Non-normalized FELM: %d\r\n", (int)value);
#endif  
    } else {
#ifdef TDEBUG
	printf("Non-normalized FELM: %d, adjusted to %d\r\n", (int)value,
		MINIMUM_FELM_LOW_TH);
#endif  
        value = MINIMUM_FELM_LOW_TH;
    }

    BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, low_felm, ON); /* Mask low FELM int. */
    BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, high_felm, ON); /* Mask high FELM int. */

    SET_WORD(bp_ptr, far_end_high_alarm_th_low, far_end_high_alarm_th_high, (value>>1));
    SET_WORD(bp_ptr, far_end_low_alarm_th_low, far_end_low_alarm_th_high, (value>>2));

    NORM(meter);
    meter = 6 - meter;

    value >>= meter;
    pst->high_felm_threshold = value >> 1;
    pst->low_felm_threshold = value >> 2;

}  /* END _SetLosThreshold() */


/***********************************************************/
/*    _SetFfeScale()                                       */
/*                                                         */
/*    returns: FFE Scale Init Value:  0 - 3                */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                16-Feb-1996              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/

void
_SetFfeScale(st)
    struct sdcore_state *st;
{
    DECLARE_PTR;
    DECLARE_PST;
    BP_S_8BIT meter;
    BP_S_16BIT value;

    NORM(meter);
    READ_METER_REG(felm_low, felm_high, value, meter);

    if (value < LONG_LOOP_BT8970)
        pst->su_flags.bits.ffe_init_scale = 0;
    else if (value < MEDIUM_LOOP_BT8970)
        pst->su_flags.bits.ffe_init_scale = 1;
    else if (value < SHORT_LOOP_BT8970)
        pst->su_flags.bits.ffe_init_scale = 2;
    else
        pst->su_flags.bits.ffe_init_scale = 3;

    if ( BP_global_state.symbol_rate <= LOW_RATE_FFE_INIT )
        pst->su_flags.bits.ffe_data_scale = 1;
    else
        pst->su_flags.bits.ffe_data_scale = 0;

#ifdef TDEBUG
    printf("FELM = %d\r\n", (int)value);
    printf("FFE Init Scale=%d for %s Data Rates\r\n",
	(int) pst->su_flags.bits.ffe_init_scale,
        pst->su_flags.bits.ffe_data_scale==0?"High":"Low");
#endif


} /* End _SetFfeScale() */

      
/***********************************************************/
/*    _CheckAagc()                                         */
/*                                                         */
/*    returns: 1 - AAGC incremented                        */
/*             0 - AAGC not incremented                    */
/*                                                         */
/*    Input variables: BP_U_8BIT no;                       */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*                                                         */
/* Programmer:                                             */
/*     iris shuker                09-June-1995             */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
int
_CheckAagc(st)
    struct sdcore_state *st;
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;
    BP_U_8BIT gain;
    BP_U_8BIT meter;
    BP_U_16BIT value;

    gain = bp_mode_ptr->adc_control.again;
#ifdef TDEBUG
    printf("current gain = %d dB\r\n", (int)(gain * 3));
#endif

    NORM(meter); /* Calculate meter normalization factor */
    READ_METER_REG(felm_low, felm_high, value, meter);

#ifdef TDEBUG
    printf("FELM = %d\r\n", (int)value);
#endif

    if ( (value < LONG_LOOP_TH) && (gain < MAX_AGAIN) ) {
        gain++;
	_Aagc(st, gain);

#ifdef TDEBUG
        printf("gain incremented to %d dB\r\n", (int)(gain * 3));

        NORM(meter); /* Calculate meter normalization factor */
        READ_METER_REG(slm_low, slm_high, value, meter);

        printf("SLM = %d\r\n", (int)value);
#endif
        return 1;
    } else
        return 0;

} /* End _CheckAagc() */

/*
 * Wrapper around _SetMeterTimer() using the FELM threshold settings
 * in the SDCORE private state structure.
 */
sdcore_SetMeterTimer(st, value)
	struct sdcore_state *st;
	int value;
{
	DECLARE_PST;

	_SetMeterTimer(value, pst->high_felm_threshold,
			pst->low_felm_threshold);
}
