/*************************************************************/
/*    SUR.C                                                  */
/*    Startup control process for HTU-R terminal             */
/*    (C) Copyright 1993 by Rockwell Corporation             */
/*                                                           */
/*    This program is copyrighted by Rockwell Corporation    */
/*                                                           */
/* Description:                                              */
/*    Controlling HTU_R startup process, normal              */
/*    operation, idle mode, test modes etc.                  */
/*                                                           */
/* Notes:                                                    */
/*                                                           */
/* User Modifiable Code:                                     */
/*    None                                                   */
/*                                                           */
/* List of functions included in this module:                */
/*  _HturControlProcess()                                    */
/*                                                           */
/* Programmer:                                               */
/*     Dean Rasmussen             March - 1998               */
/*     Iris Shuker                26-Sept-1993               */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/

/*---------------------------------------------------------*/
/*  Includes                                               */
/*---------------------------------------------------------*/

#include "../libc/types.h"
#include "../lib8973/typedefs.h"
#include "../lib8973/config.h"
#include "../lib8973/bitpump.h"
#include "../lib8973/util.h"
#include "../lib8973/bpstate.h"
#include "../lib8973/defmeter.h"
#include "../lib8973/timermac.h"
#include "state.h"
#include "privstate.h"
#include "stages.h"
#include "miscmac.h"
#include "suutil.h"
#include "../libpreact/autobaud.h"
#include "../libpreact/ifctf_preact.h"
#include "../libpreact/cm_preact.h"

/*---------------------------------------------------------*/
/*  Local Defines                                          */
/*---------------------------------------------------------*/

#define FREQUENCY_CELL_TRY_NO   4
#define FREQ_CELLS              9
#define VCXO_ACQ_INIT_VALUE     -10000    /* frequency acquisition */

#define SUR_WATCH_DOG_INTERVAL   0xFFFF
#define	PREACT_TIMEOUT_5MIN	23438
#define	PREACT_TIMEOUT_1MIN	4688

/*---------------------------------------------------------*/
/*  External Functions                                     */
/*---------------------------------------------------------*/

extern BP_S_16BIT _CalculatePhaseQuality();

/*---------------------------------------------------------*/
/*  Constants                                              */
/*---------------------------------------------------------*/
static BP_S_16BIT BP_CONSTANT vcxo_freq[FREQ_CELLS] =
    {0,-21000,21000,-14000,14000,-7000,7000,-25000,25000};

static const u_short cm_speeds[NO_CM_SPEEDS] =
	{160, 208, 320, 416, 392, 784, 1040, 1568, 2320};

/***********************************************************/
/*    _HturControlProcess()                                */
/*    HTU-R terminal control process.                      */
/*                                                         */
/*    Returns: void.                                       */
/*                                                         */
/*    Input Variables: BP_U_8BIT no                        */
/*                                                         */
/*    Output Variables: none.                              */
/*                                                         */
/*    Example:                                             */
/*           _HturControlProcess(no);                      */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                14-Sept-1993             */
/*                                                         */
/* Revision History:                                       */
/*                                                         */
/***********************************************************/

void
_HturControlProcess(st)
    struct sdcore_state *st;
{
    DECLARE_PST;
    DECLARE_PTR;
    DECLARE_MODE_PTR;
    BP_U_8BIT i;                          /* Local variable */
    BP_U_8BIT prev_aagc;
    BP_S_8BIT temp, meter;                /* Local temporary buffers */
    BP_S_16BIT temp_s16;                  /* Local temporary buffer */
    BP_U_16BIT temp_u16;                  /* Local temporary buffer */
    u_char data[2];
    union autobaud_msg abmsg;

    switch (pst->stage) /* Go to stage */
    {

        case PREACTIVATE_SYSTEM: /* Startup process with preact */

	    st->status.byte = INIT_STATUS_REG;
	    /* Are we doing preact? */
	    if (st->preact_type == PREACT_TYPE_NONE) {
		pst->stage = ACTIVATE_SYSTEM;
		break;
	    }

#ifdef TDEBUG
            printf("STU-R: preparing for pre-activation\r\n");
#endif

	    BP_global_state.symbol_rate = 20;	/* 160 kbps */
	    _BtReset(0);
	    if (st->special_modes & SDCORE_MODE_TXGAIN_OVERRIDE)
		bp_mode_ptr->tx_gain.tx_gain = st->txgain_setting;
	    bp_set_lfsr(0, _HTUR, 0);

	    /* need NORMAL_METER for DC offset */
            _SetMeterTimer(NORMAL_METER, HIGH_FELM_ALARM_TH, LOW_FELM_ALARM_TH);
	    bp_mode_ptr->adc_control.again = AGAIN12DB;

            pst->stage = SU_WAIT_METER_INTERVAL1;
            pst->stage2 = PREACT_DC_CANCEL;
            break;

        case PREACT_DC_CANCEL:
            TIMER_BREAK(TIMERMASK_METER);

            _DcCancel();

	    /* only at this point the actual preact type begins to matter */
	    switch (st->preact_type) {

	    case PREACT_TYPE_CM:
		_SetMeterArbitrary(CM_MULTWAIT_METER);
		SET_WORD(bp_ptr, far_end_high_alarm_th_low,
			 far_end_high_alarm_th_high, CMPULSE_HIGHFELM_TH);
		SET_WORD(bp_ptr, far_end_low_alarm_th_low,
			 far_end_low_alarm_th_high, CMPULSE_LOWFELM_TH);
		pst->stage = SU_WAIT_METER_INTERVAL2;
		pst->stage2 = CM_MULTWAIT0;
		break;

	    case PREACT_TYPE_AUTOBAUD:
	    case PREACT_TYPE_IFCTF:
		abrecv_install(&pst->abrecv);
		printf("Waiting for AutoBaud messages\r\n");
		if (st->special_modes & SDCORE_MODE_AB_CNX_IFCTF)
		    printf("Accepting either Conexant or IFCTF flavor\r\n");
		else
		    printf("Accepting %s flavor only\r\n",
			st->preact_type == PREACT_TYPE_IFCTF ? "IFCTF" :
			"Conexant/Mindspeed");
		abrecv_arm(&pst->abrecv);
		SET_WORD(bp_ptr, sut4_low, sut4_high, PREACT_TIMEOUT_5MIN);
		RESTART_SUT4;
		bp_mode_ptr->mask_low_reg.sut4 = 0;
		pst->stage = STUR_AB_INITIAL;
		break;

	    default:
		printf("Unsupported pre-activation type %02X!\r\n",
			st->preact_type);
		pst->stage = IDLE;
		break;
	    }
	    break;

	case CM_MULTWAIT0:
	    TIMER_BREAK(TIMERMASK_METER);
	    printf("Waiting for CM speed code signal\r\n");
            _SetFelmMask(st);
            SET_WORD(bp_ptr, sut4_low, sut4_high, SUR_WATCH_DOG_INTERVAL);
	    RESTART_SUT4;
	    bp_mode_ptr->mask_low_reg.sut4 = 0;
	    pst->stage = CM_MULTWAIT1;
	    break;

	case CM_MULTWAIT1:
	    if (st->status.bits.los)
		break;
	    SET_WORD(bp_ptr, t3_low, t3_high, 18000);	/* 225 ms */
	    RESTART_T3;
	    pst->stage = CM_MULTWAIT2;
	    break;

	case CM_MULTWAIT2:
	    if (st->status.bits.los) {
		printf("Apparent glitch\r\n");
		pst->stage = CM_MULTWAIT1;
		break;
	    }
	    TIMER_BREAK(TIMERMASK_T3);
	    printf("Got something\r\n");
	    SET_WORD(bp_ptr, t3_low, t3_high, 12000);	/* 150 ms */
	    RESTART_T3;
	    pst->stage = CM_MULTWAIT3;
	    break;

	case CM_MULTWAIT3:
	    if (st->status.bits.los) {
		pst->misc_counter = 0;
		SET_WORD(bp_ptr, t3_low, t3_high, 1904);	/* 23.8 ms */
		RESTART_T3;
		pst->stage = CM_PAUSE1;
		break;
	    }
	    TIMER_BREAK(TIMERMASK_T3);
	    pst->stage = CM_BABBLE0;
	    break;

	case CM_PAUSE1:
	    if (!st->status.bits.los) {
		pst->stage = CM_BABBLE0;
		break;
	    }
	    TIMER_BREAK(TIMERMASK_T3);
	    SET_WORD(bp_ptr, t3_low, t3_high, 12000);	/* 150 ms */
	    RESTART_T3;
	    pst->stage = CM_PAUSE2;
	    break;

	case CM_PAUSE2:
	    if (st->status.bits.los) {
		TIMER_BREAK(TIMERMASK_T3);
		printf("Longstop\r\n");
		pst->stage = CM_MULTWAIT1;
		break;
	    }
	    pst->misc_counter++;
	    SET_WORD(bp_ptr, t3_low, t3_high, 6000);	/* 75 ms */
	    RESTART_T3;
	    pst->stage = CM_PULSE1;
	    break;

	case CM_PULSE1:
	    if (st->status.bits.los) {
		printf("Truncated pulse\r\n");
		pst->stage = CM_MULTWAIT1;
		break;
	    }
	    TIMER_BREAK(TIMERMASK_T3);
	    SET_WORD(bp_ptr, t3_low, t3_high, 24000);	/* 300 ms */
	    RESTART_T3;
	    pst->stage = CM_PULSE2;
	    break;

	case CM_PULSE2:
	    if (st->status.bits.los) {
		SET_WORD(bp_ptr, t3_low, t3_high, 1904);	/* 23.8 ms */
		RESTART_T3;
		pst->stage = CM_PAUSE1;
		break;
	    }
	    TIMER_BREAK(TIMERMASK_T3);
	    /* got it! */
	    i = pst->misc_counter;
	    printf("Counted %u pulse(s)\r\n", i);
	    if (i < 1 || i > NO_CM_SPEEDS) {
		printf("Pulse count doesn't match a known CM speed code\r\n");
		pst->stage = PREACT_RESTART;
		break;
	    }
	    temp_u16 = cm_speeds[i-1];
	    printf("STU-C indicates %u kbps data rate\r\n", temp_u16);
	    if (!is_datarate_supported(temp_u16)) {
		printf("Data rate not supported on this hardware!\r\n");
		pst->stage = PREACT_RESTART;
		break;
	    }
	    /* good to go! */
	    st->user_data_rate = temp_u16 / 8;
	    st->status.bits.preact_done = 1;
	    /*
	     * Well, not so fast.  We need to wait for LOS (end of the
	     * pre-activation signal) before we enter classic HTU-R
	     * startup or it could be thrown off.
	     *
	     * Give it a generous 750 ms.
	     */
            SET_WORD(bp_ptr, sut4_low, sut4_high, 58);
	    RESTART_SUT4;
	    pst->stage = PREACT_WAIT_FOR_LOS;
	    break;

	case CM_BABBLE0:
	    printf("Babble detected, waiting for silence\r\n");
	    pst->stage = CM_BABBLE1;
	    break;

	case CM_BABBLE1:
	    if (st->status.bits.los) {
		SET_WORD(bp_ptr, t3_low, t3_high, 19904);	/* 248.8 ms */
		RESTART_T3;
		pst->stage = CM_BABBLE2;
		break;
	    }
	    /* "goofy signal detector" logic to be implemented here */
	    break;

	case CM_BABBLE2:
	    if (!st->status.bits.los) {
		pst->stage = CM_BABBLE1;
		break;
	    }
	    TIMER_BREAK(TIMERMASK_T3);
	    printf("Return to wait for signal\r\n");
	    pst->stage = CM_MULTWAIT1;
	    break;

	/* AutoBaud and IFCTF pre-activation */

	case STUR_AB_INITIAL:
	    if (pst->abrecv.state != ABRECV_STATE_GOTIT)
		break;
	    abmsg.word = pst->abrecv.data;
	    printf("Got msg: %08X\r\n", abmsg.word);
	    if (abmsg.bytes[0] == IPA_OPC_HELLO && abmsg.bytes[1] == 'C' &&
		ifctf_preact_checkcrc(&abmsg)) {
		if (st->preact_type == PREACT_TYPE_IFCTF ||
		    st->special_modes & SDCORE_MODE_AB_CNX_IFCTF) {
			st->preact_type = PREACT_TYPE_IFCTF;
			st->quat_orient = _SERIAL_SIGNFIRST;
			SET_WORD(bp_ptr, sut4_low, sut4_high,
				 PREACT_TIMEOUT_1MIN);
			RESTART_SUT4;
			pst->stage = STUR_AB_PROC_IFCTF;
			break;
		} else {
		    printf("Configured not to accept IFCTF pre-activation\r\n");
		    abrecv_arm(&pst->abrecv);
		    break;	/* same stage */
		}
	    }
	    if (abmsg.word == 0x7FFF8000) {
		if (st->preact_type == PREACT_TYPE_AUTOBAUD ||
		    st->special_modes & SDCORE_MODE_AB_CNX_IFCTF) {
			st->preact_type = PREACT_TYPE_AUTOBAUD;
			pst->open_attempts = 0;		/* overloading var */
			SET_WORD(bp_ptr, sut4_low, sut4_high,
				 PREACT_TIMEOUT_1MIN);
			RESTART_SUT4;
			pst->stage = STUR_AB_PROC_CNX;
			break;
		} else {
		    printf("Configured not to accept Conexant AutoBaud\r\n");
		    abrecv_arm(&pst->abrecv);
		    break;	/* same stage */
		}
	    }
	    if (abmsg.word == 0xFFFFFFFF)
		printf("Continuous signal detected: perhaps no pre-activation is used?\r\n");
	    else if ((abmsg.word & 0xFFE00000) == 0xF8E00000)
		printf("Seems like Copper Mountain pre-activation signal\r\n");
	    else
		printf("Not valid Hello/Unlock message\r\n");
	    abrecv_arm(&pst->abrecv);
	    break;	/* same stage */

	case STUR_AB_ENGAGED:
	    if (pst->abrecv.state != ABRECV_STATE_GOTIT)
		break;
	    printf("Got msg: %08X\r\n", pst->abrecv.data);
	    switch (st->preact_type) {
	    case PREACT_TYPE_IFCTF:
		pst->stage = STUR_AB_PROC_IFCTF;
		break;
	    case PREACT_TYPE_AUTOBAUD:
		pst->stage = STUR_AB_PROC_CNX;
		break;
	    default:
		printf("BUG: st->preact_type invalid in STUR_AB_ENGAGED!\r\n");
		pst->stage = IDLE;
	    }
	    break;

	case STUR_AB_PROC_IFCTF:
	    abmsg.word = pst->abrecv.data;
	    if (!ifctf_preact_checkcrc(&abmsg)) {
		printf("Bad CRC, ignored\r\n");
		abrecv_arm(&pst->abrecv);
		pst->stage = STUR_AB_ENGAGED;
		break;
	    }
	    if (stur_ab_ifctf_process(st, &abmsg))
		pst->stage = STUR_AB_RESP;
	    else {
		abrecv_arm(&pst->abrecv);
		pst->stage = STUR_AB_ENGAGED;
	    }
	    break;

	case STUR_AB_PROC_CNX:
	    abmsg.word = pst->abrecv.data;
	    if (stur_ab_cnx_process(st, &abmsg))
		pst->stage = STUR_AB_RESP;
	    else {
		abrecv_arm(&pst->abrecv);
		pst->stage = STUR_AB_ENGAGED;
	    }
	    break;

	case STUR_AB_RESP:
	    if (pst->paste.state != PASTE_STATE_DONE)
		break;
	    if (st->status.bits.preact_done)
		pst->stage = ACTIVATE_SYSTEM;
	    else {
		abrecv_arm(&pst->abrecv);
		pst->stage = STUR_AB_ENGAGED;
	    }
	    break;


	case PREACT_RESTART:
	    printf("Restarting pre-activation\r\n");
	    pst->stage = PREACTIVATE_SYSTEM;
	    break;

	case PREACT_WAIT_FOR_LOS:
	    if (!st->status.bits.los)
		break;
	    pst->stage = ACTIVATE_SYSTEM;
	    break;


        case ACTIVATE_SYSTEM: /* Classic HDSL startup process */

            /*------------------------------------------------------------*/
            /* Initialization.                                            */
            /*------------------------------------------------------------*/
#ifdef TDEBUG
            printf("Entering classic HTU-R startup\r\n");
#endif

	    BP_global_state.symbol_rate = st->user_data_rate;
	    _BtReset(0);
	    if (st->special_modes & SDCORE_MODE_TXGAIN_OVERRIDE)
		bp_mode_ptr->tx_gain.tx_gain = st->txgain_setting;
	    bp_mode_ptr->cu_interface_modes.interface_mode = st->quat_orient;
	    bp_set_lfsr(0, _HTUR, 0);
	    bp_set_rxmode(0, st->special_modes & SDCORE_MODE_NORXDESCRAM ?
			SELECT_DETECTOR_OUTPUT : SELECT_DESCRAMBLER_OUTPUT);

	    pst->high_felm_threshold = HIGH_FELM_ALARM_TH;
	    pst->low_felm_threshold = LOW_FELM_ALARM_TH;
            sdcore_SetMeterTimer(st, NORMAL_METER);
            /*Changed ALT_METER to NORMAL_METER to get a more accurate DC Offset*/

            if ( BP_global_state.symbol_rate <= 32 )
		bp_mode_ptr->adc_control.again = AGAIN12DB;

            pst->stage = SU_WAIT_METER_INTERVAL1;
            pst->stage2 = DC_CANCELLATION;
            break;

        case DC_CANCELLATION:
            TIMER_BREAK(TIMERMASK_METER);

            _DcCancel();

            /*
             *  LOS (WAIT_FOR_SIGNAL)meter interval requires the ALT_METER
             *  to be used.
             */
            sdcore_SetMeterTimer(st, ALT_METER);

            pst->stage = GOTO_WAIT_FOR_SIGNAL;
	    break;

	case GOTO_WAIT_FOR_SIGNAL:
	    TIMER_BREAK(TIMERMASK_METER);
            _SetFelmMask(st);

            /*
             * start Watch Dog Timer, 10-15 times longer than typical
             * startup times
	     *
	     * Pre-activation change: if we've done preact, set the timer
	     * to 500 ms instead as we expect S0 to follow the pre-activation
	     * phase very shortly.
             */
	    if (st->preact_type == PREACT_TYPE_NONE)
		temp_u16 = SUR_WATCH_DOG_INTERVAL;
	    else
		temp_u16 = SYMRATE_DECBIN_ADJUST
				(BP_global_state.symbol_rate<<1);
            SET_WORD(bp_ptr, sut4_low, sut4_high, temp_u16);
	    RESTART_SUT4;
	    bp_mode_ptr->mask_low_reg.sut4 = 0;

#ifdef TDEBUG
            PREFIX;
            printf("Wait for signal...\r\n");
#endif
	    pst->stage = WAIT_FOR_SIGNAL;
            break;


            /*------------------------------------------------------------*/
            /* If far-end signal power detected, start measuring startup  */
            /* time intervals. Otherwise - continue waiting for far-end   */
            /* signal.                                                    */
            /*------------------------------------------------------------*/

        case WAIT_FOR_SIGNAL: /* Wait for far end signal */

            if (st->status.bits.los) /* Wait for far end signal */
                break;

            /*------------------------------------------------------------*/
            /* Far-end signal detected, Initialize and run startup timers */
            /*------------------------------------------------------------*/

#ifdef TDEBUG
            PREFIX;
            printf("Signal detected\r\n");
#endif

            sdcore_SetMeterTimer(st, DEFAULT_METER);

            _SetActivationInterval(st);

            pst->stage = SU_WAIT_METER_INTERVAL1;
            pst->stage2 = OPEN_EYE1;

            break;


        /*------------------------------------------------------------*/
        /* Open eye process                                           */
        /*------------------------------------------------------------*/

        case OPEN_EYE1: /* Initialize open eye process */

            TIMER_BREAK(TIMERMASK_METER);

            NORM(meter);
            READ_METER_REG(felm_low, felm_high, temp_s16, meter);

#ifdef TDEBUG
            PREFIX;
            printf("felm = %d\r\n", (int)temp_s16);
#endif
            /*
             * When HTU-C first powers ON, it could cause a glitch
             */
            if ( temp_s16 < 100 )
                {
#ifdef TDEBUG
                PREFIX;
                printf("Detected Glitch - restarting activation\r\n");
#endif

                pst->stage = PREACTIVATE_SYSTEM;
                break;
                }

#ifdef TDEBUG
            PREFIX;
            printf("Open eye\r\n");
            PREFIX;
            printf("Freq cell 0\r\n");
#endif
            
            /* calculate scale */
            _Aagc(st, AGAIN9DB);    /* Need for Set FFE Scale */
            _SetFfeScale(st);

	    pst->freq_cell = 0;
	    pst->open_attempts = 0;

            /*
             * Adjust AAGC based on FFE Init Scale (FELM)
             * For scale=1, already at 9dB
             */
            if ( pst->su_flags.bits.ffe_init_scale == 0 )
                _Aagc(st, AGAIN12DB);
            else if ( pst->su_flags.bits.ffe_init_scale == 2 )
                _Aagc(st, AGAIN6DB);
            else if ( pst->su_flags.bits.ffe_init_scale == 3 )
                _Aagc(st, AGAIN3DB);

            pst->stage = SU_WAIT_METER_INTERVAL1;
            pst->stage2 = OPEN_EYE2;
            break;


        case OPEN_EYE2: /* set frequency cell, Init DFE+FFE, Adapt DAGC */

            BP_WRITE_BIT(bp_mode_ptr, pll_modes, freeze_pll, ON); /* Freeze PLL */

            /* ------------------- */
            /* Set Frequency Cell: */
            /* ------------------- */
            BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, rphs, 0); /* Init receive sampling phase */

            SET_WORD(bp_ptr, vcxo_frequency_low, vcxo_frequency_high, vcxo_freq[pst->freq_cell]);

            /* ------------------------ */
            /* Initialize DAGC+DFE+FFE: */
            /* ------------------------ */
        
            _InitDagc();
            _InitDfe();
            _InitFfe(st);

            /* ---------------------------- */
            /* Adapt DAGC at EQ_Error mode: */
            /* ---------------------------- */

            _InitDetector();
            _AdaptDagcEqErrorMode();

            SET_WORD(bp_ptr, t3_low, t3_high, 17700);
	    RESTART_T3;

            pst->stage = OPEN_EYE3;
            break;

        case OPEN_EYE3: /* Adapt FFE */
            TIMER_BREAK(TIMERMASK_T3);

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF); /* Freeze DAGC coefficient */

            /* ---------- */
            /* Adapt FFE: */
            /* ---------- */

            _AdaptAllFfe();

            SET_WORD(bp_ptr, t3_low, t3_high, 8700);
	    RESTART_T3;

            pst->stage = OPEN_EYE4;
            break;


        case OPEN_EYE4: /* Adapt FFE, DFE */
            TIMER_BREAK(TIMERMASK_T3);

            /* -------------- */
            /* Adapt FFE+DFE: */
            /* -------------- */

            _AdaptDfe();

            SET_WORD(bp_ptr, t3_low, t3_high, 6700);
	    RESTART_T3;

            pst->stage = OPEN_EYE5;
            break;

        case OPEN_EYE5: /* Freeze FFE, DFE */
            TIMER_BREAK(TIMERMASK_T3);

            _FreezeDfeFfe();

            pst->stage = OPEN_EYE6;
            break;

        case OPEN_EYE6: /* Verify eye opening with Slicer */
            if (_IsFirstEyeOpen()) { /* Open eye succeeded */
                pst->stage = ACQUIRE_FREQ1; /* Move to frequency acquisition */

#ifdef TDEBUG
                printf("Freq Cell: %d", (int)pst->freq_cell);
                printf(", Attempts: %d\r\n", (int)pst->open_attempts);
#endif 
            } else /* Open eye failed */
                pst->stage = OPEN_EYE7; /* Try again at this frequency cell */
            break;


        case OPEN_EYE7: /* Frequency cells/open attempts loops control */
            pst->open_attempts++; /* Increment open attempts counter */
            if (pst->open_attempts >= FREQUENCY_CELL_TRY_NO) /* Ran out of attempts, move to next frequency cell */
            {
                pst->open_attempts = 0; /* Init open attempts counter */

                pst->freq_cell++; /* Increment frequency cell counter */

                if (pst->freq_cell >= FREQ_CELLS) /* Checked all frequency cells, SU failed */
                {
#ifdef TDEBUG
                    PREFIX;
                    printf("NO Freq cell, SU failed, Retry SU...\r\n");
#endif
                    pst->stage = SU_WAIT_METER_INTERVAL1;
                    pst->stage2 = OPEN_EYE1;
                    break;
                } /* END-IF all frequencies were checked */
#ifdef TDEBUG
                PREFIX;
                printf("Freq cell %d\r\n", (int) pst->freq_cell);
#endif
            } /* END-IF Ran out of attempts */

            pst->stage = OPEN_EYE2; /* Move to the first stage */
            break;

        /*------------------------------------------------------------*/
        /* Frequency Acquisition                                      */
        /*------------------------------------------------------------*/

        case ACQUIRE_FREQ1: /* Frequency acquisition: highest gain */

#ifdef TDEBUG
            PREFIX;
            printf("Freq Acqu\r\n");
#endif

	    /*
	     * This busticated logic makes no sense to me.
	     * We have just gone through all that effort with the
	     * frequency cells to get the PLL into the ballpark of
	     * the correct frequency (and that part must pass successfully
	     * before startup can proceed further!), but now all that
	     * work gets blown away when the VCXO control voltage is
	     * reset to a constant?!
	     *
	     * Let's try it the way I think ought to be right.
	     *
	     * -MS
	     */
#if 0
            BP_WRITE_BIT(bp_mode_ptr, pll_modes, freeze_pll, ON);
            SET_WORD(bp_ptr, vcxo_frequency_low, vcxo_frequency_high, VCXO_ACQ_INIT_VALUE);
#endif

            /*------------------------------------------------------------*/
            /* Set PLL parameters, activate PLL (highest gain)            */
            /*------------------------------------------------------------*/
            BP_WRITE_BIT(bp_mode_ptr, pll_modes, phase_detector_gain, HIGHER_GAIN); /* Phase detector at high gain */
            BP_WRITE_BIT(bp_mode_ptr, pll_modes, pll_gain, HIGHEST_GAIN); /* Wide PLL */
            BP_WRITE_BIT(bp_mode_ptr, pll_modes, freeze_pll, OFF); /* Adapt PLL */

            SET_WORD(bp_ptr, sut2_low, sut2_high, 536);
	    RESTART_SUT2;

            pst->stage = ACQUIRE_FREQ2;
            break;

        case ACQUIRE_FREQ2: /* Frequency acquisition: higher gain */
            TIMER_BREAK(TIMERMASK_SUT2);

            BP_WRITE_BIT(bp_mode_ptr, pll_modes, pll_gain, HIGHER_GAIN); /* Mid-width PLL */

            SET_WORD(bp_ptr, sut2_low, sut2_high, 115);
	    RESTART_SUT2;

            pst->stage = ACQUIRE_FREQ3;
            break;

        case ACQUIRE_FREQ3: /* Acquire frequency stage */
            TIMER_BREAK(TIMERMASK_SUT2);

            BP_WRITE_BIT(bp_mode_ptr, pll_modes, pll_gain, HIGH_GAIN); /* Narrow PLL */

            SET_WORD(bp_ptr, sut2_low, sut2_high, 90);
	    RESTART_SUT2;

            pst->stage = ACQUIRE_FREQ4;
            break;

        case ACQUIRE_FREQ4: /* Acquire frequency stage */
            TIMER_BREAK(TIMERMASK_SUT2);

            BP_WRITE_BIT(bp_mode_ptr, pll_modes, phase_detector_gain, NORMAL_GAIN); /* Phase detector at normal gain */

            SET_WORD(bp_ptr, sut2_low, sut2_high, 100);
	    RESTART_SUT2;

            pst->stage = ACQUIRE_FREQ5;
            break;

        case ACQUIRE_FREQ5: /* Acquire frequency stage */
            TIMER_BREAK(TIMERMASK_SUT2);

            sdcore_SetMeterTimer(st, PHASE_LOCK_METER);

            pst->stage = SU_WAIT_METER_INTERVAL1;
            pst->stage2 = PHASE_LOCKED;
            break;


        /*------------------------------------------------------------*/
        /* Verify that phase is locked                                */
        /*------------------------------------------------------------*/
        case PHASE_LOCKED: /* Checking phase lock */

#ifdef TDEBUG
            PREFIX;
            printf("Verify phase locking\r\n");
#endif

            if (_IsPhaseLocked()) /* Phase is locked */
                pst->stage = OPTIMAL_PHASE;
            else /* Phase is not locked */
                {

#ifdef TDEBUG
                PREFIX;
                printf("Phase not locked, try again\r\n");
#endif

		/*
		 * This frequency cell cycling logic makes me even more
		 * certain that we shouldn't be blowing away the frequency
		 * that comes from the "cell". -MS
		 */
		temp = pst->freq_cell;
                temp++; /* Increment frequency cell counter */
                if (temp >= FREQ_CELLS)
                    temp = 0;
		pst->freq_cell = temp;

                sdcore_SetMeterTimer(st, DEFAULT_METER);

                pst->stage = SU_WAIT_METER_INTERVAL1;
		pst->stage2 = OPEN_EYE2;
                } /* END-ELSE phase is not locked */
            break;

        case OPTIMAL_PHASE: /* Search for optimal sampling phase */

#ifdef TDEBUG
            PREFIX;
            printf("Search opt phase\r\n");
#endif

	    if (st->htur_su_critsect)
		st->htur_su_critsect(1);

            BP_WRITE_BIT(bp_mode_ptr, pll_modes, freeze_pll, ON); /* Freeze PLL */

            /* -------------------------------------- */
            /* Begin - Search Optimal Phase Algorithm */
            /* -------------------------------------- */

            temp = 0; /* Count no. of phases opened */

            sdcore_SetMeterTimer(st, PHASE_QUALITY_METER);
            BP_WRITE_BIT(bp_mode_ptr, timer_source, meter, OFF);

            for (i = 0; i < NPHASES; i++) { /* Sampling phase loop */
                BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, rphs, i); /* set receive sampling phase */

                /* ------------------------------ */
                /* Initialize DAGC, DFE, and FFE: */
                /* ------------------------------ */
                _InitDagc();
                _InitDfe();
                _InitFfe(st);

                /* ---------------------------------- */
                /* Adapt DAGC @ HIGH Gain, Use Slicer */
                /* ---------------------------------- */
                _InitDetector();
                _AdaptDagcEqErrorMode();

                SET_WORD(bp_ptr, t3_low, t3_high, 5000 /* 15000 */ );   /* 5000 */
		RESTART_T3;
                WAIT_FOR_TIMER(TIMERMASK_T3);

                BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF); /* Freeze DAGC coefficient */

                /* ---------- */
                /* Adapt FFE: */
                /* ---------- */

                _AdaptAllFfe();
                SET_WORD(bp_ptr, t3_low, t3_high, 10000 /* 20000 */);  /* 6800 */
		RESTART_T3;
                WAIT_FOR_TIMER(TIMERMASK_T3);

                /* -------------- */
                /* Adapt FFE+DFE: */
                /* -------------- */

                _AdaptDfe();
                SET_WORD(bp_ptr, t3_low, t3_high, 10000 /* 20000 */); /* 6700 */
		RESTART_T3;
                WAIT_FOR_TIMER(TIMERMASK_T3);

                _FreezeDfeFfe();

#ifdef TDEBUG
                printf("(%d) ", (int)i); 
#endif

                /* -------------------------------- */
                /* Calculate phase quality measure: */
                /* -------------------------------- */
                temp_s16 = _CalculatePhaseQuality(st);

#ifdef TDEBUG
                printf("P=%d ", (int)temp_s16); 
#endif

		pst->phase_quality[i] = temp_s16;
                if (temp_s16 != 0)
                    temp++; /* Increment phase counter */

                /* ---------------------------------------*/
                /* Read NLM (SNR):                        */
                /* Used during the Optimal Phase Search.  */
                /* ---------------------------------------*/

                /*
                 * Need to wait 2 meter intervals to ensure proper reading
                 */
                RESTART_METER;
                WAIT_FOR_TIMER(TIMERMASK_METER);

                RESTART_METER;
                WAIT_FOR_TIMER(TIMERMASK_METER);

                NORM(meter);
                READ_METER_REG(nlm_low, nlm_high, temp_s16, meter);
		pst->noise_quality[i] = temp_s16;

#ifdef TDEBUG
                printf("NLM=%d\r\n", (int)temp_s16);
#endif
                        
            }   /* for Sampling Phase */

            /*
             * Don't need to wait since ReOpenOptimalPhase() has a
             * large amount of symbol delays adapting equalizer
             */
            sdcore_SetMeterTimer(st, DEFAULT_METER);

            if ( temp < NO_OF_OPENED_PHASES ) /* Attenuator recognition */
                {
                if ( pst->su_flags.bits.ffe_init_scale < 3 )
                    {
                    pst->su_flags.bits.ffe_init_scale++;
                    BP_WRITE_BIT(bp_mode_ptr, pll_modes, freeze_pll, OFF); /* Reactivate PLL */
		    if (st->htur_su_critsect)
			st->htur_su_critsect(0);

#ifdef TDEBUG
                    PREFIX;
                    printf("FFE Scale changed to %d\r\n",
				(int)pst->su_flags.bits.ffe_init_scale);

#endif
                    /*
                     * don't change stage number, re-do OPTIMAL_PHASE
                     * with new FFE Init Scale.
                     */
                    break;
                    } /* END-IF */

                } /* END-IF Attenuator recognition */

            /* -------------------------- */
            /* Set Optimal Sampling Phase */
            /* -------------------------- */

            _CalculateOptimalPhase1(st);
            _CalculateOptimalPhase2(st);
            _CalculateOptimalPhase3(st);
            _CalculateOptimalPhase4(st);

            BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, rphs, _CalculateOptimalPhase(st) );

            /* ----------------------- */
            /* Begin Re-Open Process   */
            /* ----------------------- */
            _ReOpenOptimalPhase(st);

            /* ---------------------------------- */
            /* Adapt DFE @ HIGH Gain, Use Slicer: */
            /* ---------------------------------- */

            _AdaptDfe();
            SET_WORD(bp_ptr, t3_low, t3_high, 20000);
	    RESTART_T3;
            WAIT_FOR_TIMER(TIMERMASK_T3);

            /* ---------------------------------------- */
            /* Adapt DFE + FFE @ HIGH Gain, Use Slicer: */
            /* ---------------------------------------- */

            _AdaptAllFfe();
            SET_WORD(bp_ptr, sut2_low, sut2_high, 90);
	    RESTART_SUT2;
            WAIT_FOR_TIMER(TIMERMASK_SUT2);

            /* --------------------------- */
            /* Adapt FFE+DFE @ NORMAL Gain */
            /* --------------------------- */

            BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_gain, NORMAL_GAIN);          /* FFE NORMAL adaptation gain */
            BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_gain, NORMAL_GAIN);          /* DFE NORMAL adaptation gain */


            SET_WORD(bp_ptr, sut2_low, sut2_high, 90);
	    RESTART_SUT2;
            WAIT_FOR_TIMER(TIMERMASK_SUT2);

            _FreezeDfeFfe();

            BP_WRITE_BIT(bp_mode_ptr, pll_modes, pll_gain, HIGHER_GAIN); /* Select narrow width PLL filter */
            BP_WRITE_BIT(bp_mode_ptr, pll_modes, freeze_pll, OFF); /* Reactivate PLL */

	    if (st->htur_su_critsect)
		st->htur_su_critsect(0);
            pst->stage = TRANSMIT_2LEVEL;
            break;

        case TRANSMIT_2LEVEL: /* Transmit RS0 startup sequence */

            /* --------------------------------------------- */
            /* Set DAGC Target value for future readaptation */
            /* --------------------------------------------- */

            NORM(meter); /* Calculate meter normalization factor */
            READ_METER_REG(felm_low, felm_high, temp_s16, 0);

	    bp_eq_read(DAGC_HIGH, data);
            temp = data[1];  /* High Byte only */

            temp >>= 1;
            temp_s16 *= temp;
            temp_s16 = (BP_S_16BIT) (temp_s16 >> (6 - meter));
            temp_s16 = temp_s16 + (temp_s16 >> 1) + (temp_s16 >> 3); /* 62.5% scale */

            SET_WORD(bp_ptr, dagc_target_low, dagc_target_high, temp_s16);

#ifdef TDEBUG
            printf("DAGC Target = %d\r\n", (int)temp_s16);
#endif

#ifdef TDEBUG
            PREFIX;
            printf("TX RS0\r\n");
#endif

            /* Restart Activation Interval */
            _SetActivationInterval(st);

	    if (st->htur_su_critsect)
		st->htur_su_critsect(1);
            BP_WRITE_BIT(bp_mode_ptr, pll_modes, freeze_pll, ON); /* Freeze PLL */
	    bp_set_tx(0, 1, st->special_modes & SDCORE_MODE_EXTS0S1 ?
			UNSCRAMBLED_TWO_LEVEL_DATA : SCRAMBLED_TWO_LEVEL_ONES);

            prev_aagc = bp_mode_ptr->adc_control.again; /* Read AAGC Setting */

            /* --------------------------------------------- */
            /* Determine maximum AAGC setting.               */
            /* --------------------------------------------- */
            temp = MAX_AGAIN;
            do {
                _Aagc(st, temp);

                /*
                 * Need to wait 2 meter intervals to ensure proper reading
                 */

                RESTART_METER;
                WAIT_FOR_TIMER(TIMERMASK_METER);

                RESTART_METER;
                WAIT_FOR_TIMER(TIMERMASK_METER);

                _SetFelmMask(st);

                /* 'meter' has already been normalized */
                READ_METER_REG(slm_low, slm_high, temp_u16, meter);

#ifdef TDEBUG
                PREFIX;
                printf("SLM @ %d dB = %d\r\n", (int)(temp*3), (int)temp_u16);
#endif

                if ( temp_u16 < SIGNAL_LEVEL_TH )
                    break;

                temp--;    /* decrement AAGC to next lower setting */

            } while ( temp >= AGAIN0DB );

            /* -------------------------------- */
            /* Scaling DAGC, FFE:               */
            /* -------------------------------- */

#ifdef TDEBUG
            printf("FFE Coefficients (Before Scaling):\r\n");
            _PrintfFFECoefs();
#endif
	    _FfeReverse();

            temp = bp_mode_ptr->adc_control.again; /* Read AAGC Setting */

	    bp_eq_read(DAGC_HIGH, data);
            temp_s16 = ((BP_S_16BIT)data[1] << 4);  /* High Byte only */

#ifdef TDEBUG
            PREFIX;
            printf("DAGC (Before Scaling) = %d\r\n", (int)temp_s16);
#endif

            /*
             * If AAGC dropped from default setting, need to scale
             * FFE and/or DAGC
             */
            switch (prev_aagc - temp) {
                default:
                    /* don't do anything */
                    break;

                case 1:  /* AAGC Dropped 3dB */
                
                    if (temp_s16 > 1130) {
                        /*
                        * DAGC Too High to compensate for 3dB
                        * Must use FFE Scale
                        */
                    
                        _ScaleFfe125();  /* scale FFE Coeffs by 1.25 */
#ifdef TDEBUG
                        printf("FFE X 1.25\r\n");
#endif
                    }
                    break;

                case 2:  /* AAGC Dropped 6dB */
                    /*
                     * If DAGC < 1200
                     *      Use FFE Scale & DAGC to compensate
                     *
                     * If DAGC > 1200
                     *      Use AAGC, FFE Scale, & DAGC to compensate
                     */
                    if (temp_s16 > 1200) { /* was 1000 */
                        _Aagc(st, (BP_U_8BIT)(temp + 1));  /* increment AAGC by 3dB */

#ifdef TDEBUG
                        printf("AAGC set to %d dB\r\n", (int)(temp+1) * 3 );
#endif
                    }

                    _ScaleFfe125();  /* scale FFE Coeffs by 1.25 */
#ifdef TDEBUG
                    printf("FFE X 1.25\r\n");
#endif
                    break;

                case 3:  
                case 4: /* AAGC Dropped 9dB */
                case 5: /* See this on ETSI Loop 7 */

                    /* Always increase AAGC by 3dB */
                    _Aagc(st, (BP_U_8BIT)(temp + 1));
                    _ScaleFfe125();  /* scale FFE Coeffs by 1.25 */
#ifdef TDEBUG
                    printf("AAGC set to %d dB\r\n", (int)(temp+1) * 3 );
                    printf("FFE X 1.25\r\n");
#endif

		    if (temp_s16 > 1130) {
                        /*
                         * DAGC Too High to compensate
                         * Must use FFE Scale
                         */
                    
                        _ScaleFfe125();  /* scale FFE Coeffs by 1.25 */
#ifdef TDEBUG
                        printf("FFE X 1.25\r\n");
#endif
                    }
                    break;

            }   /* end switch AGAIN */

            /*
             * if DAGC > 1700, then scale FFE. 
             * FFE may be scaled twice
             */
#ifdef TDEBUG
            PREFIX;
            printf("DAGC (Before 2nd Scaling) = %d\r\n", (int)temp_s16);
#endif

            if (temp_s16 > 1700) {
                _ScaleFfe125();  /* scale FFE Coeffs by 1.25 */
#ifdef TDEBUG
                printf("DAGC>1700:  FFE X 1.25\r\n");
#endif
            }

            /*----------------------------------------------------*/
            /* Linear EC adaptation                               */
            /*----------------------------------------------------*/
#ifdef TDEBUG
            PREFIX;
            printf("Adapt EC\r\n");
#endif
            _AdaptEc(0, HIGHEST_GAIN, (BP_U_16BIT)49401); /* Adapt EC at highest gain */
            BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_coefficients, ON); /* Adapt linear EC */
            WAIT_FOR_TIMER(TIMERMASK_T3);

            _AdaptEc(0, HIGHER_GAIN, (BP_U_16BIT)63000); /* Adapt EC at higher gain */
            WAIT_FOR_TIMER(TIMERMASK_T3);

            _AdaptEc(0, HIGH_GAIN, (BP_U_16BIT)63000); /* Adapt EC at high gain */
            WAIT_FOR_TIMER(TIMERMASK_T3);

#ifdef TDEBUG
            PREFIX;
            printf("Before Self:  ");
            _PrintfDAGC();
#endif

            /* ---------------------------- */
            /* Adapt DAGC - Self adaptation */
            /* ---------------------------- */

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, eq_error_adaptation, OFF); /* DAGC at self adaptation mode */
            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_gain, NORMAL_GAIN);
            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, ON); /* Adapt DAGC */

            SET_WORD(bp_ptr, t3_low, t3_high, 15000);
	    RESTART_T3;
            WAIT_FOR_TIMER(TIMERMASK_T3);

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF); /* Freeze DAGC */

#ifdef TDEBUG
            PREFIX;
            printf("After Self:  ");
            _PrintfDAGC();
#endif

            /* -------------------------------- */
            /* Adapt DAGC - Eq Error adaptation */
            /* -------------------------------- */

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, eq_error_adaptation, ON); /* DAGC at eq error adaptation mode */
            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, ON); /* Adapt DAGC */

            SET_WORD(bp_ptr, t3_low, t3_high, 10000);
	    RESTART_T3;
            WAIT_FOR_TIMER(TIMERMASK_T3);

#ifdef TDEBUG
            PREFIX;
            printf("After Readapt:  ");
            _PrintfDAGC();
#endif

            BP_WRITE_BIT(bp_mode_ptr, pll_modes, freeze_pll, OFF); /* Reactivate PLL */
	    if (st->htur_su_critsect)
		st->htur_su_critsect(0);
            pst->stage = ADAPT_EQ1;
            break;

        case ADAPT_EQ1: /* Restart Equalizer adaptation */
#ifdef TDEBUG
            PREFIX;
            printf("Restart EQ:  ");
            _PrintfDAGC();
#endif

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, eq_error_adaptation, ON); /* DAGC at eq error adaptation mode */
            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, ON); /* Adapt DAGC */
            BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_coefficients, ON); /* Adapt DFE */

            SET_WORD(bp_ptr, t3_low, t3_high, 10000);
	    RESTART_T3;

            pst->stage = ADAPT_EQ2;
            break;

        case ADAPT_EQ2: /* Adapt Equalizer */
            TIMER_BREAK(TIMERMASK_T3);

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF); /* Freeze DAGC */
            BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_coefficients, OFF); /* Freeze DFE */

            BP_WRITE_BIT(bp_mode_ptr, detector_modes, two_level, OFF); /* 4-level Slicing */
            sdcore_SetMeterTimer(st, FOUR_LEVEL_METER);

            pst->stage = SU_WAIT_METER_INTERVAL1;
            pst->stage2 = WAIT_FOR_4LEVEL;
#ifdef TDEBUG
            PREFIX;
            printf("Waiting for 4 level...\r\n");
#endif
            break;


        case WAIT_FOR_4LEVEL: /* Wait for far end four level signal, CS1 */
            TIMER_BREAK(TIMERMASK_METER);

            i = BP_READ_REG(bp_ptr, symbol_histogram);

            /* If number of +/-1 symbols exceeds threshold: 4-level signal is received */
            if (i < FOUR_LEVEL_SIGNAL_TH) /* Far end 4-level signal NOT detected */
                break;


            /*------------------------------------------------------------*/
            /* 4-level far-end signal detected                            */
            /*------------------------------------------------------------*/

#ifdef TDEBUG
            PREFIX;
            printf("4 level detected\r\n");
#endif

            /*------------------------------------------------------------*/
            /* Check for Tip/Ring Reversal                                */
            /*------------------------------------------------------------*/
            if (!(st->special_modes & SDCORE_MODE_NOAUTOTR)) {
                READ_METER_REG(ber_meter_low, ber_meter_high, temp_u16, 0);
                printf("BER Meter at positive line polarity = %d\r\n",
			(int)temp_u16);
                if (temp_u16 > AUTOTR_GOODBER_TH) {
                    BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes,
				 negate_symbol, 1);
		    printf("Trying negative line polarity\r\n");
		    pst->stage = SU_WAIT_METER_INTERVAL1;
		    pst->stage2 = TIPRING1;
		    break;
		}
	    }

	    pst->stage = TIPRING3;
	    break;

	case TIPRING1:
	    TIMER_BREAK(TIMERMASK_METER);
            READ_METER_REG(ber_meter_low, ber_meter_high, temp_u16, 0);
            printf("BER Meter = %d\r\n", (int)temp_u16);
            if (temp_u16 <= AUTOTR_GOODBER_TH) {
		st->status.bits.tip_ring_indication = 1;
		pst->stage = TIPRING3;
		break;
	    }
	    printf("Still bad, waiting for good BER with either polarity\r\n");
	    pst->stage = TIPRING2;
	    break;

	case TIPRING2:
	    TIMER_BREAK(TIMERMASK_METER);
            READ_METER_REG(ber_meter_low, ber_meter_high, temp_u16, 0);
            if (temp_u16 <= AUTOTR_GOODBER_TH) {
		printf("BER Meter = %d\r\n", (int)temp_u16);
		st->status.bits.tip_ring_indication =
				bp_mode_ptr->nonlinear_ec_modes.negate_symbol;
		pst->stage = TIPRING3;
		break;
	    }
	    bp_mode_ptr->nonlinear_ec_modes.negate_symbol =
				!bp_mode_ptr->nonlinear_ec_modes.negate_symbol;
	    pst->stage = SU_WAIT_METER_INTERVAL1;
	    pst->stage2 = TIPRING2;
	    break;

	case TIPRING3:
	    printf("Tip/ring polarity appears to be %s\r\n",
		st->status.bits.tip_ring_indication ? "reversed" : "correct");

            sdcore_SetMeterTimer(st, NORMAL_METER);

	    pst->stage = SU_WAIT_METER_INTERVAL1;
	    pst->stage2 = WAIT_FOR_4LEVEL_2;

            break;

        case WAIT_FOR_4LEVEL_2:
            TIMER_BREAK(TIMERMASK_METER);

            _SetFelmMask(st);

            /*------------------------------------------------------------*/
            /* Transmit startup sequence RS1.                             */
            /*------------------------------------------------------------*/

#ifdef TDEBUG
            PREFIX;
            printf("TX RS1\r\n");
#endif

	    bp_set_tx(0, 1, st->special_modes & SDCORE_MODE_EXTS0S1 ?
		      UNSCRAMBLED_FOUR_LEVEL_DATA : SCRAMBLED_FOUR_LEVEL_ONES);

            st->status.bits.four_level_indication = ON;

#ifdef TDEBUG
            _PrintfFFECoefs();
            _PrintfDFECoefs();
#endif

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, ON); /* Adapt DAGC */
            BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_coefficients, ON); /* Adapt DFE */

            SET_WORD(bp_ptr, t3_low, t3_high, 2001);
	    RESTART_T3;

            pst->stage = ADAPT_MODULES1;
            break;

        case ADAPT_MODULES1: /* Adapt NL EC (high gain) */
            TIMER_BREAK(TIMERMASK_T3);

#ifdef TDEBUG
            PREFIX;
            printf("Adapt NL EC & EP\r\n");
#endif
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, zero_output, OFF);
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_gain, HIGH_GAIN); /* NL EC at high adaptation gain */
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_coefficients, ON); /* Adapt NL EC */

            SET_WORD(bp_ptr, t3_low, t3_high, 60000);
	    RESTART_T3;

            pst->stage = ADAPT_MODULES2;
            break;

        case ADAPT_MODULES2: /* Adapt NL EC (low gain) and EP */
            TIMER_BREAK(TIMERMASK_T3);

            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_gain, NORMAL_GAIN); /* NL EC at normal adaptation gain */
	    if (!(st->special_modes & SDCORE_MODE_NLECON)) {
		BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_coefficients, OFF); /* Freeze NL EC */
		BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, zero_output, ON); /* Zero NL EC output */
	    }

            BP_WRITE_BIT(bp_mode_ptr, ep_modes, adapt_gain, NORMAL_GAIN); /* EP at normal adaptation gain */
            BP_WRITE_BIT(bp_mode_ptr, ep_modes, adapt_coefficients, ON); /* Adapt EP */
            
            SET_WORD(bp_ptr, t3_low, t3_high, 60000);
	    RESTART_T3;
            pst->stage = SET_LOS_THRESHOLD;
            break;

        case SET_LOS_THRESHOLD:
            TIMER_BREAK(TIMERMASK_T3);

            _SetLosThreshold(st);

            pst->stage = SU_WAIT_METER_INTERVAL1;
            pst->stage2 = SET_NORMAL_CONDITIONS;
            break;

        case SET_NORMAL_CONDITIONS: /* Set normal conditions */
            TIMER_BREAK(TIMERMASK_METER);

            _SetFelmMask(st);

            BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_gain, NORMAL_GAIN);
            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF); /* Freeze DAGC */

            st->status.bits.normal_operation = ON;

#ifdef TDEBUG
            PREFIX;
            printf("Normal operation\r\n");
#endif

	    /*
	     * Unmask the meter timer interrupt:
	     * it will be used by the temp/env code
	     * as well as anyone wanting to run the BER meter.
	     */
	    BP_global_state.meter_int_count = 0;
	    bp_mode_ptr->mask_low_reg.meter = 0;

            pst->stage = NORMAL_OPERATION;
            break;

        case NORMAL_OPERATION: /* Normal adaptation stage */
	    if (!st->tempenv_num_intervals ||
		BP_global_state.meter_int_count < st->tempenv_num_intervals)
		break;
	    BP_global_state.meter_int_count = 0;
	    tempenv_adapt_dagc_dfe(st, 0);
            break;

        case DEACTIVATE_MODE: /* Deactivate stage */
            break;

        case IDLE: /* Idle mode */
            break;


        case SU_WAIT_METER_INTERVAL1:
            RESTART_METER;
            pst->stage = SU_WAIT_METER_INTERVAL2;
            break;

        case SU_WAIT_METER_INTERVAL2:
            TIMER_BREAK(TIMERMASK_METER);
	    pst->stage = pst->stage2;
            break;

        default:
	    printf("SDCORE: invalid state in _HturControlProcess() ASM!\r\n");
            pst->stage = IDLE;
            break;

        } /* END-SWITCH operational stage */

} /* END _HturControlProcess() */
