/*************************************************************/
/*    SUC.C                                                  */
/*    Startup control process for HTU-C terminal.            */
/*    (C) Copyright 1993 by Rockwell Corporation             */
/*                                                           */
/*    This program is copyrighted by Rockwell Corporation    */
/*                                                           */
/* Description:                                              */
/*    Controlling HTU-C startup process, normal operation,   */
/*    idle mode, test modes etc.                             */
/*                                                           */
/* Notes:                                                    */
/*                                                           */
/* User Modifiable Code:                                     */
/*    None                                                   */
/*                                                           */
/* List of functions included in this module:                */
/*  _HtucControlProcess()                                    */
/*                                                           */
/* Programmer:                                               */
/*     Dean Rasmussen             March - 1998               */
/*     Iris Shuker                26-Sept-1993               */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*************************************************************/

/*---------------------------------------------------------*/
/*  Includes                                               */
/*---------------------------------------------------------*/

#include "../libc/types.h"
#include "../lib8973/typedefs.h"
#include "../lib8973/config.h"
#include "../lib8973/bitpump.h"
#include "../lib8973/util.h"
#include "../lib8973/bpstate.h"
#include "../lib8973/defmeter.h"
#include "../lib8973/timermac.h"
#include "state.h"
#include "privstate.h"
#include "stages.h"
#include "miscmac.h"
#include "suutil.h"
#include "../libpreact/autobaud.h"
#include "../libpreact/ifctf_preact.h"

/*---------------------------------------------------------*/
/*  Local Defines                                          */
/*---------------------------------------------------------*/

/* 5 Seconds */
#define T1MIN_INTERVAL(symbol_rate) \
((((BP_S_16BIT) symbol_rate <<2) + ((BP_S_16BIT) symbol_rate)) << 2)

/* 4 Seconds */
#define T4MAX_INTERVAL(symbol_rate) \
(((BP_S_16BIT) symbol_rate <<2) << 2)

#define MIN_ADJ_GLITCH_FELM_TH      (BP_U_16BIT)1000
#define MIN_NO_ADJ_GLITCH_FELM_TH   (BP_U_16BIT)100

#define ZERO_LEN_ADJUST_FELM  (BP_S_16BIT)512

/*
 * How many times should we retry the pre-activation hello message
 * before redoing the bitpump reset and DC offset cancellation
 */
#define	PREACT_MAX_HELLO_RETRY	50	/* a few minutes */

/*---------------------------------------------------------*/
/*  Externals                                              */
/*---------------------------------------------------------*/

extern BP_U_8BIT BP_CONSTANT dagc_th[4];

extern BP_S_16BIT _CalculatePhaseQuality();

/***********************************************************/
/*    _HtucControlProcess()                                */
/*    HTU-C terminal control process.                      */
/*                                                         */
/*    Returns: void.                                       */
/*                                                         */
/*    Input Variables: BP_U_8BIT no                        */
/*                                                         */
/*    Output Variables: none.                              */
/*                                                         */
/*    Example:                                             */
/*           _HtucControlProcess(no);                      */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                14-Sept-1993             */
/*                                                         */
/* Revision History:                                       */
/*                                                         */
/***********************************************************/

void
_HtucControlProcess(st)
    struct sdcore_state *st;
{
    DECLARE_PST;
    DECLARE_PTR;
    DECLARE_MODE_PTR;
    BP_U_8BIT temp_u;                     /* Local temporary buffer */
    BP_S_8BIT temp, meter;                /* Local temporary buffer */
    BP_S_16BIT temp_s16;                  /* Local temporary buffer */
    BP_U_16BIT temp_u16;                  /* Local temporary buffer */
    u_char data[2];
    union autobaud_msg abmsg;

    switch (pst->stage) /* Go to stage */
    {

	case PREACTIVATE_SYSTEM:

            /* Basic initialization. */
            printf("Activate STU-C\r\n");
	    BP_global_state.symbol_rate = st->user_data_rate;
	    _BtReset(0);
	    if (st->special_modes & SDCORE_MODE_TXGAIN_OVERRIDE)
		bp_mode_ptr->tx_gain.tx_gain = st->txgain_setting;
	    bp_set_lfsr(0, _HTUC, 0);

	    switch (st->preact_type) {

	    case PREACT_TYPE_NONE:
		pst->stage = ACTIVATE_SYSTEM;
		break;

	    case PREACT_TYPE_CM:
		printf("STU-C: sending CM pre-activation signal\r\n");
		if (tx_cm_preact(&pst->paste, st->user_data_rate << 3) < 0) {
		    printf("Error: %u kbps is not a valid CM data rate!\r\n",
			   st->user_data_rate << 3);
		    pst->stage = IDLE;
		    break;
		}
		pst->stage = CM_PREACT_TX;
		break;

	    case PREACT_TYPE_AUTOBAUD:
	    case PREACT_TYPE_IFCTF:
		printf("STU-C: doing interactive pre-activation\r\n");
		/* need NORMAL_METER for DC offset */
		_SetMeterTimer(NORMAL_METER, HIGH_FELM_ALARM_TH,
				LOW_FELM_ALARM_TH);
		bp_mode_ptr->adc_control.again = AGAIN12DB;
		pst->stage = SU_WAIT_METER_INTERVAL1;
		pst->stage2 = PREACT_DC_CANCEL;
		break;

	    default:
		printf("Unsupported pre-activation type %02X!\r\n",
			st->preact_type);
		pst->stage = IDLE;
		break;
	    }
	    break;

	case CM_PREACT_TX:
	    if (pst->paste.state != PASTE_STATE_DONE)
		break;
	    pst->stage = ACTIVATE_SYSTEM;
	    break;

	/* interactive pre-activation: Conexant and IFCTF */
        case PREACT_DC_CANCEL:
            TIMER_BREAK(TIMERMASK_METER);
            _DcCancel();
	    abrecv_install(&pst->abrecv);
	    if (st->preact_fallback)
		pst->misc_counter = st->preact_fallback;
	    else
		pst->misc_counter = PREACT_MAX_HELLO_RETRY;
	    pst->stage = STUC_AB_HELLO;
	    pst->freq_cell = 0;		/* variable overload */
	    break;

	case STUC_AB_HELLO:
	    abrecv_disarm(&pst->abrecv);
	    if (st->preact_type == PREACT_TYPE_IFCTF) {
		printf("Sending IFCTF SDSL Hello message\r\n");
		abmsg.bytes[0] = IPA_OPC_HELLO;
		abmsg.bytes[1] = IPA_TTYPE_C;
		abmsg.bytes[2] = 0;
		ifctf_preact_setcrc(&abmsg);
	    } else {
		printf("Sending Conexant/Mindspeed AutoBaud unlock msg\r\n");
		abmsg.word = 0x7FFF8000;
	    }
	    tx_autobaud_msg(&pst->paste, abmsg.word, pst->freq_cell);
	    pst->freq_cell = 0;
	    pst->stage = STUC_AB_HELLO_T;
	    break;

	case STUC_AB_HELLO_T:
	    if (pst->paste.state != PASTE_STATE_DONE)
		break;
	    abrecv_arm(&pst->abrecv);
	    /* 3 s timeout */
            SET_WORD(bp_ptr, sut2_low, sut2_high,
		SYMRATE_DECBIN_ADJUST(BP_global_state.symbol_rate * 12));
            RESTART_SUT2;
	    pst->stage = STUC_AB_HELLO_W;
	    break;

	case STUC_AB_HELLO_W:
	    if (pst->abrecv.state == ABRECV_STATE_GOTIT) {
		abmsg.word = pst->abrecv.data;
		printf("Got response: %08X\r\n", abmsg.word);
		pst->freq_cell = AB_TURNAROUND_TIME;
		if (st->preact_type == PREACT_TYPE_AUTOBAUD &&
		    abmsg.word != 0x7FFF8000 ||
		    st->preact_type == PREACT_TYPE_IFCTF &&
		    (abmsg.bytes[0] != IPA_OPC_HELLO || abmsg.bytes[1] != 'R')){
		      printf("Response from STU-R is not what's expected\r\n");
			pst->stage = STUC_AB_HELLO_F;
			break;
		}
		if (st->preact_type == PREACT_TYPE_IFCTF) {
		    if (!ifctf_preact_checkcrc(&abmsg)) {
			printf("Bad CRC\r\n");
			pst->stage = STUC_AB_HELLO_F;
			break;
		    }
		    ifctf_show_stur_type(st, abmsg.bytes[2]);
		    pst->stage = STUC_AB_SETRATE;
		} else
		    pst->stage = STUC_AB_HELLO2;
		pst->misc_counter = 3;
		break;
	    }
	    TIMER_BREAK(TIMERMASK_SUT2);
	    pst->stage = STUC_AB_HELLO_F;
	    break;

	case STUC_AB_HELLO_F:
	    if (--pst->misc_counter) {
		pst->stage = STUC_AB_HELLO;
		break;
	    }
	    if (st->preact_fallback) {
		printf("Fallback for dumb STU-R\r\n");
		pst->stage = ACTIVATE_SYSTEM;
	    } else
		pst->stage = PREACT_RESTART;
	    break;

	case STUC_AB_HELLO2:
	    abrecv_disarm(&pst->abrecv);
	    abmsg.bytes[0] = AB_OPC_HTUC_DISC;
	    abmsg.bytes[1] = st->cnxab_htuc_flavor[0];
	    abmsg.bytes[2] = st->cnxab_htuc_flavor[1];
	    abmsg.bytes[3] = abmsg.bytes[0] ^ abmsg.bytes[1] ^ abmsg.bytes[2];
	    printf("Sending flavor ID msg: %08X\r\n", abmsg.word);
	    tx_autobaud_msg(&pst->paste, abmsg.word, pst->freq_cell);
	    pst->freq_cell = 0;
	    pst->stage = STUC_AB_HELLO2_T;
	    break;

	case STUC_AB_HELLO2_T:
	    if (pst->paste.state != PASTE_STATE_DONE)
		break;
	    abrecv_arm(&pst->abrecv);
	    /* 3 s timeout */
            SET_WORD(bp_ptr, sut2_low, sut2_high,
		SYMRATE_DECBIN_ADJUST(BP_global_state.symbol_rate * 12));
            RESTART_SUT2;
	    pst->stage = STUC_AB_HELLO2_W;
	    break;

	case STUC_AB_HELLO2_W:
	    if (pst->abrecv.state == ABRECV_STATE_GOTIT) {
		abmsg.word = pst->abrecv.data;
		pst->freq_cell = AB_TURNAROUND_TIME;
		if (evaluate_ab_response(st, &abmsg) < 0) {
		    pst->stage = STUC_AB_HELLO2_F;
		    break;
		}
		if (abmsg.bytes[0] != AB_OPC_HTUR_DISC) {
		  printf("Response opcode doesn't match expected protocol\r\n");
		    pst->stage = PREACT_RESTART;
		    break;
		}
		st->cnxab_htur_flavor[0] = abmsg.bytes[1];
		st->cnxab_htur_flavor[1] = abmsg.bytes[2];
		pst->stage = STUC_AB_SETRATE;
		pst->misc_counter = 3;
		break;
	    }
	    TIMER_BREAK(TIMERMASK_SUT2);
	    pst->stage = STUC_AB_HELLO2_F;
	    break;

	case STUC_AB_HELLO2_F:
	    if (--pst->misc_counter) {
		pst->stage = STUC_AB_HELLO2;
		break;
	    }
	    pst->stage = PREACT_RESTART;
	    break;

	case STUC_AB_SETRATE:
	    abrecv_disarm(&pst->abrecv);
	    if (st->preact_type == PREACT_TYPE_IFCTF)
		abmsg.bytes[0] = IPA_OPC_STARTB;
	    else
		abmsg.bytes[0] = AB_OPC_SETRATE;
	    abmsg.bytes[1] = st->user_data_rate >> 8;
	    abmsg.bytes[2] = st->user_data_rate & 0xFF;
	    if (st->preact_type == PREACT_TYPE_IFCTF)
		ifctf_preact_setcrc(&abmsg);
	    else
	        abmsg.bytes[3] = abmsg.bytes[0] ^ abmsg.bytes[1] ^
				 abmsg.bytes[2];
	    printf("Sending startup req msg for %u kbps: %08X\r\n",
		   st->user_data_rate << 3, abmsg.word);
	    pst->scratch_long = abmsg.word;
	    tx_autobaud_msg(&pst->paste, abmsg.word, pst->freq_cell);
	    pst->freq_cell = 0;
	    pst->stage = STUC_AB_SETRATE_T;
	    break;

	case STUC_AB_SETRATE_T:
	    if (pst->paste.state != PASTE_STATE_DONE)
		break;
	    abrecv_arm(&pst->abrecv);
	    /* 3 s timeout */
            SET_WORD(bp_ptr, sut2_low, sut2_high,
		SYMRATE_DECBIN_ADJUST(BP_global_state.symbol_rate * 12));
            RESTART_SUT2;
	    pst->stage = STUC_AB_SETRATE_W;
	    break;

	case STUC_AB_SETRATE_W:
	    if (pst->abrecv.state == ABRECV_STATE_GOTIT) {
		abmsg.word = pst->abrecv.data;
		pst->freq_cell = AB_TURNAROUND_TIME;
		if (evaluate_ab_response(st, &abmsg) < 0) {
		    pst->stage = STUC_AB_SETRATE_F;
		    break;
		}
		if (st->preact_type == PREACT_TYPE_IFCTF) {
		    if (evaluate_ifctf_ack(st, &abmsg) < 0)
			pst->stage = PREACT_RESTART;
		    else {
			st->status.bits.preact_done = 1;
			pst->stage = ACTIVATE_SYSTEM;
		    }
		} else {
		    if (abmsg.word == pst->scratch_long) {
			st->status.bits.preact_done = 1;
			pst->stage = ACTIVATE_SYSTEM;
		    } else {
		printf("Didn't get echo of request, don't know what to do\r\n");
			pst->stage = PREACT_RESTART;
		    }
		}
		break;
	    }
	    TIMER_BREAK(TIMERMASK_SUT2);
	    pst->stage = STUC_AB_SETRATE_F;
	    break;

	case STUC_AB_SETRATE_F:
	    if (--pst->misc_counter) {
		pst->stage = STUC_AB_SETRATE;
		break;
	    }
	    pst->stage = PREACT_RESTART;
	    break;

	case PREACT_RESTART:
	    printf("Resetting the pre-activation process\r\n");
	    pst->stage = PREACTIVATE_SYSTEM;
	    break;


        case ACTIVATE_SYSTEM: /* Activate startup process */

	    /*
	     * Because we've already done _BtReset() in PREACTIVATE_SYSTEM,
	     * we only do "light" re-init here.
	     */
            printf("Activate HTU-C\r\n");
	    abrecv_uninstall();
	    _BtInitialize(0);
	    bp_mode_ptr->cu_interface_modes.interface_mode = st->quat_orient;
	    bp_set_lfsr(0, _HTUC, 0);
	    bp_set_rxmode(0, st->special_modes & SDCORE_MODE_NORXDESCRAM ?
			SELECT_DETECTOR_OUTPUT : SELECT_DESCRAMBLER_OUTPUT);

	    pst->high_felm_threshold = HIGH_FELM_ALARM_TH;
	    pst->low_felm_threshold = LOW_FELM_ALARM_TH;
            sdcore_SetMeterTimer(st, NORMAL_METER);
            /*Changed ALT_METER to NORMAL_METER to get a more accurate DC Offset*/

            /* At low data rate, use higher initial AAGC */ 
            if ( BP_global_state.symbol_rate <= 32 )
		bp_mode_ptr->adc_control.again = AGAIN12DB;

            /*
             * Disable signal detection, turning on Tx before adapting EC
             * will cause false detection.
             */
            BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, low_felm, ON);
            BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, high_felm, ON);

            pst->stage = SU_WAIT_METER_INTERVAL1;
	    pst->stage2 = DC_CANCELLATION;
            break;

        case DC_CANCELLATION:
            TIMER_BREAK(TIMERMASK_METER);

            _DcCancel();

            /*
             *  LOS (INIT_STARTUP)meter interval requires the ALT_METER
             *  to be used.
             */
            sdcore_SetMeterTimer(st, ALT_METER);

            pst->stage = SU_WAIT_METER_INTERVAL1;
            pst->stage2 = INIT_STARTUP;
            break;

        case INIT_STARTUP: /* Start CS0 transmission */
            TIMER_BREAK(TIMERMASK_METER);


#ifdef TDEBUG
            printf("TX CS0\r\n");
#endif
            /*------------------------------------------------------------*/
            /* Transmit startup sequence                                  */
            /*------------------------------------------------------------*/
	    bp_set_tx(0, 1, st->special_modes & SDCORE_MODE_EXTS0S1 ?
			UNSCRAMBLED_TWO_LEVEL_DATA : SCRAMBLED_TWO_LEVEL_ONES);

            /*------------------------------------------------------------*/
            /* Initialize and operate startup timers (activation time)    */
            /*------------------------------------------------------------*/
	    _SetActivationInterval(st);


            /*------------------------------------------------------------*/
            /* Adapt EC                                                   */
            /*------------------------------------------------------------*/
#ifdef TDEBUG
            printf("Adapt EC\r\n");
#endif

            BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_coefficients, ON); /* Adapt! */

            _AdaptEc(0, HIGHEST_GAIN, (BP_U_16BIT)39201); /* Adapt Ec at highest gain */
            pst->stage = ADAPT_EC_STAGE1;
            break;

        case ADAPT_EC_STAGE1: /* Linear EC adaptation stage */
            TIMER_BREAK(TIMERMASK_T3);
            _AdaptEc(0, HIGHER_GAIN, (BP_U_16BIT)58804); /* Adapt EC at higher gain */
            pst->stage = ADAPT_EC_STAGE2;
            break;

        case ADAPT_EC_STAGE2: /* Linear EC adaptation stage */
            TIMER_BREAK(TIMERMASK_T3);
            _AdaptEc(0, HIGH_GAIN, (BP_U_16BIT)42202); /* Adapt EC at high gain */
            pst->stage = ADAPT_EC_STAGE3;
            break;

        case ADAPT_EC_STAGE3: /* Linear Ec adaptation stage */
            TIMER_BREAK(TIMERMASK_T3);


            NORM(meter);
            READ_METER_REG(felm_low, felm_high, temp_u16, meter);
#ifdef TDEBUG
            printf("felm = %d\r\n", (int)temp_u16);
#endif

            /*
             * Special 0 length case: hybrid causes high FELM reading.
             * Therefore, just force high threshold to be (FELM + 512)
             *
             * Need to set 'high_felm_th' to non-normalized value since
             * SetMeterTimer() expects prescaled value.
             *
             * ((high_felm_th << meter) << (meter - 1))
             *   ==> Normalized High FELM Th / 2
             *
             * 10/98 - For lower data rates, need a different threshold
             *         
             */
            pst->su_flags.bits.adjusted_felm = 0;

            if (temp_u16 >= (BP_U_16BIT)(pst->high_felm_threshold << 5)) {
		temp_u16 += ZERO_LEN_ADJUST_FELM;
		SET_WORD(bp_ptr, far_end_high_alarm_th_low,
			 far_end_high_alarm_th_high, temp_u16 >> 3);
                pst->high_felm_threshold =
					(temp_u16 + ZERO_LEN_ADJUST_FELM) >> 6;

                pst->su_flags.bits.adjusted_felm = 1;

#ifdef TDEBUG
                PREFIX;
                printf("0 length detected\r\n");
                printf("Adjust FELM High TH =  %d\r\n", temp_u16);
#endif

                pst->stage = SU_WAIT_METER_INTERVAL1;
                pst->stage2 = ADJUST_FELM_HIGH_TH;
                break;

            }
            pst->stage = GOTO_WAIT_FOR_SIGNAL;
            break;

        case ADJUST_FELM_HIGH_TH:
            TIMER_BREAK(TIMERMASK_METER);
            pst->stage = GOTO_WAIT_FOR_SIGNAL;
            break;


        case GOTO_WAIT_FOR_SIGNAL:
            /*
             * Re-enable signal detection.
             */
            _SetFelmMask(st);

            pst->stage = WAIT_FOR_SIGNAL;
#ifdef TDEBUG
            printf("Wait for signal...\r\n");
#endif
            break;


        case WAIT_FOR_SIGNAL: /* Wait for far end signal */

            if (st->status.bits.los) /* Wait for far end signal */
                break;

            /*------------------------------------------------------------*/
            /* Far-end signal detected: Initialize and run T1MIN, T4MAX   */
            /* timers.                                                    */
            /*------------------------------------------------------------*/
#ifdef TDEBUG
            PREFIX;
            printf("Signal detected\r\n");
#endif

            sdcore_SetMeterTimer(st, DEFAULT_METER);

            pst->stage = SU_WAIT_METER_INTERVAL1;
            pst->stage2 = AAGC1;
            break;

        case AAGC1:
            TIMER_BREAK(TIMERMASK_METER);

            _SetFelmMask(st);

	    temp_u16 = BP_global_state.symbol_rate;
            SET_WORD(bp_ptr, sut2_low, sut2_high,
			SYMRATE_DECBIN_ADJUST(T4MAX_INTERVAL(temp_u16)));
            SET_WORD(bp_ptr, sut3_low, sut3_high,
			SYMRATE_DECBIN_ADJUST(T1MIN_INTERVAL(temp_u16)));
            RESTART_SUT2;
            RESTART_SUT3;

            /* Restart Activation Interval when S0 detected */
            _SetActivationInterval(st);

            _SetFfeScale(st);

            _Aagc(st, MAX_AGAIN);

            pst->stage = SU_WAIT_METER_INTERVAL1;
            pst->stage2 = AAGC2;
            break;


        case AAGC2:
            TIMER_BREAK(TIMERMASK_METER);

            _SetFelmMask(st);

            /*-----------------------------------------------------------*/
            /* Read the average absolute level of the input signal.      */
            /*-----------------------------------------------------------*/

            NORM(meter); /* Calculate meter normalization factor */
            READ_METER_REG(slm_low, slm_high, temp_u16, meter);

            temp_u = bp_mode_ptr->adc_control.again; /* Read AAGC Setting */

#ifdef TDEBUG
            PREFIX;
            printf("SLM @ %d dB = %d\r\n", (int)(temp_u*3), (int)temp_u16);
#endif
            if ( temp_u16 < SIGNAL_LEVEL_TH || temp_u == AGAIN0DB ) {
                pst->stage = ADAPT_EC_AFTER_AAGC1;
                break;
            }

            /* decrement AAGC and re-measure SLM */
            temp_u--;
            _Aagc(st, temp_u);

            pst->stage = SU_WAIT_METER_INTERVAL1;
            pst->stage2 = AAGC2;
            break;


        case ADAPT_EC_AFTER_AAGC1: /* Adapt EC highest gain */
            BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_coefficients, ON); /* Restart linear EC adaptation */

#ifdef TDEBUG
            printf("Adapt EC after AAGC\r\n");
#endif

            _AdaptEc(0, HIGHEST_GAIN, (BP_U_16BIT)39123); /* Adapt Ec at highest gain */
            pst->stage = ADAPT_EC_AFTER_AAGC2;
            break;

        case ADAPT_EC_AFTER_AAGC2: /* Adapt EC higher gain */
            TIMER_BREAK(TIMERMASK_T3);
            _AdaptEc(0, HIGHER_GAIN, (BP_U_16BIT)58747); /* Adapt EC at higher gain */
            pst->stage = ADAPT_EC_AFTER_AAGC3;
            break;

        case ADAPT_EC_AFTER_AAGC3: /* Adapt EC high gain */
            TIMER_BREAK(TIMERMASK_T3);
            _AdaptEc(0, HIGH_GAIN, (BP_U_16BIT)39019); /* Adapt EC at high gain */
            pst->stage = ADAPT_EC_AFTER_AAGC4;
            break;

        case ADAPT_EC_AFTER_AAGC4: /* Adapt EC high gain */
            TIMER_BREAK(TIMERMASK_T3);

            NORM(meter);

#ifdef TDEBUG
            READ_METER_REG(slm_low, slm_high, temp_u16, meter);
            PREFIX;
            printf("SLM after AAGC = %d\r\n", (int)temp_u16);
#endif

            READ_METER_REG(felm_low, felm_high, temp_u16, meter);
#ifdef TDEBUG
            PREFIX;
            printf("FELM after AAGC = %d\r\n", (int)temp_u16);
#endif

            /*
             * Look at the FELM to see if the signal is present, could get
             * false glitch if cable disconnected then reconnected.
             *
             * If (FELM < 1000 and Adjusted FELM for Zero Length)
             *
             * Zero Length is determined if the 'high_felm_th' was adjusted.
             */
            if ( (pst->su_flags.bits.adjusted_felm == 1 && temp_u16 < MIN_ADJ_GLITCH_FELM_TH) ||
                 (pst->su_flags.bits.adjusted_felm == 0 && temp_u16 < MIN_NO_ADJ_GLITCH_FELM_TH) )
                {
                /* Reset critical variables */

                /* At low data rate, use higher initial AAGC */ 
                if ( BP_global_state.symbol_rate <= 32 )
		    bp_mode_ptr->adc_control.again = AGAIN12DB;
                else
		    bp_mode_ptr->adc_control.again = AGAIN9DB;


		pst->high_felm_threshold = HIGH_FELM_ALARM_TH;
		pst->low_felm_threshold = LOW_FELM_ALARM_TH;

                sdcore_SetMeterTimer(st, ALT_METER);
                BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, low_felm, ON);
                BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, high_felm, ON);

                st->status.bits.los = 1;

#ifdef TDEBUG
               PREFIX;
               printf("Detected glitch, return to wait for signal.\r\n");
#endif

                BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_coefficients, ON); /* Adapt! */
                _AdaptEc(0, HIGHEST_GAIN, (BP_U_16BIT)39201); /* Adapt Ec at highest gain */

               pst->stage = ADAPT_EC_STAGE1;
               break;
               }


            _CheckAagc(st);

            pst->stage = REMOTE_PHASE_LOCK;
            break;

            /*------------------------------------------------------------*/
            /* Wait for HTU-R phase lock (T4MAX), unless the other side is*/
            /* Bitpump - in which case continue without delay.            */
            /*------------------------------------------------------------*/
        case REMOTE_PHASE_LOCK: /* Check remote side configuration */

	    pst->stage = st->special_modes & SDCORE_MODE_HTUC_EXTRADELAYS ?
			 WAIT_FOR_REMOTE_PHASE_LOCK : START_OPEN_EYE;
            break;

        case WAIT_FOR_REMOTE_PHASE_LOCK: /* Wait for remote phase lock (T4MAX) */
            TIMER_BREAK(TIMERMASK_SUT2);
            pst->stage = START_OPEN_EYE;
            break;

	/*-----------------------------------------------------------------*/
	/* Open the Eye of Ra:                                             */
	/* The eye opening process includes operating the Equalizer        */
	/* (DAGC, DFE, FFE) and Detector modules.                          */
	/* The process is repeated at 16 different sampling phases,        */
	/* the best phase (according to phase quality measure) is selected */
	/*-----------------------------------------------------------------*/

        case START_OPEN_EYE: /* Start HTU-C search optimal phase process */

            BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, rphs, 0); /* Init receive sampling phase */

	    pst->misc_counter = 0;

#ifdef TDEBUG
            PREFIX;
            printf("Search opt phase\r\n");
#endif

            sdcore_SetMeterTimer(st, PHASE_QUALITY_METER);

            pst->stage = SU_WAIT_METER_INTERVAL1;
            pst->stage2 = START_OPEN_EYE2;
            break;

        case START_OPEN_EYE2:
            TIMER_BREAK(TIMERMASK_METER);
            _SetFelmMask(st);
            pst->stage = PHASE_ADAPT_EC1; 
            break;

        case PHASE_ADAPT_EC1: /* EC adaptation at current sampling phase */
            _AdaptEc(0, HIGHEST_GAIN, (BP_U_16BIT)10000); /* Adapt EC at highest gain */
            pst->stage = PHASE_ADAPT_EC2;
            break;

        case PHASE_ADAPT_EC2: /* EC adaptation at current sampling phase */
            TIMER_BREAK(TIMERMASK_T3);
            _AdaptEc(0, HIGHER_GAIN, (BP_U_16BIT)30000); /* Adapt EC at higher gain */
            pst->stage = PHASE_ADAPT_EC3;
            break;

        case PHASE_ADAPT_EC3: /* EC adaptation at current sampling phase */
            TIMER_BREAK(TIMERMASK_T3);
            _AdaptEc(0, HIGH_GAIN, (BP_U_16BIT)30000); /* Adapt EC at high gain */
            pst->stage = PHASE_ADAPT_EC4;
            break;

        case PHASE_ADAPT_EC4: /* EC adaptation at current sampling phase */
            TIMER_BREAK(TIMERMASK_T3);
            pst->stage = OPEN_EYE1;
            break;

        case OPEN_EYE1: /* Open eye process */

            /* ------------------------------ */
            /* Initialize DAGC, DFE, and FFE: */
            /* ------------------------------ */

            _InitDagc();
            _InitDfe();
            _InitFfe(st);

            /* ---------------------------------- */
            /* Adapt DAGC @ HIGH Gain, Use Slicer */
            /* ---------------------------------- */
            _InitDetector();
               
            _AdaptDagcEqErrorMode();

            SET_WORD(bp_ptr, t3_low, t3_high, 5000);
	    RESTART_T3;
            pst->stage = OPEN_EYE2;
            break;

        case OPEN_EYE2: /* Open eye process */
            TIMER_BREAK(TIMERMASK_T3);

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF); /* Freeze DAGC coefficient */

            /* ---------- */
            /* Adapt FFE: */
            /* ---------- */

            _AdaptAllFfe();
            SET_WORD(bp_ptr, t3_low, t3_high, 10000 /* 20000 */);    /* 6800 */
	    RESTART_T3;
            pst->stage = OPEN_EYE3;
            break;

        case OPEN_EYE3:
            TIMER_BREAK(TIMERMASK_T3);
            /* -------------- */
            /* Adapt FFE+DFE: */
            /* -------------- */

            _AdaptDfe();
            SET_WORD(bp_ptr, t3_low, t3_high, 10000 /* 20000 */);    /* 6700 */
	    RESTART_T3;
            pst->stage = OPEN_EYE4;
            break;


        case OPEN_EYE4:
            TIMER_BREAK(TIMERMASK_T3);


            _FreezeDfeFfe();

            /* -------------------------------- */
            /* Calculate phase quality measure: */
            /* -------------------------------- */

            temp = BP_READ_BIT(bp_mode_ptr, receive_phase_select, rphs);
#ifdef TDEBUG
            printf("(%d) ", (int)temp);
#endif

            temp_s16 = _CalculatePhaseQuality(st);
	    pst->phase_quality[temp] = temp_s16;

            if (temp_s16 != 0) 
		pst->misc_counter++;

#ifdef TDEBUG
            printf("P=%d ", (int)pst->phase_quality[temp]);
#endif


            pst->stage = SU_WAIT_METER_INTERVAL1;
            pst->stage2 = OPEN_EYE5;
            break;


        case OPEN_EYE5:
            TIMER_BREAK(TIMERMASK_METER);

            temp = BP_READ_BIT(bp_mode_ptr, receive_phase_select, rphs);

            /* ---------------------------------------*/
            /* Read NLM (SNR):                        */
            /* Used during the Optimal Phase Search.  */
            /* ---------------------------------------*/
            NORM(meter);
            READ_METER_REG(nlm_low, nlm_high, temp_s16, meter);
	    pst->noise_quality[temp] = temp_s16;

#ifdef TDEBUG
            printf("NLM=%d\r\n", (int)temp_s16); 
#endif


            /*------------------------------------------------------------*/
            /* If done (all 16 sampling phases) go to finding best phase, */
            /* otherwise - continue with next phase.                      */
            /*------------------------------------------------------------*/
            if ( temp < 15)  /* Not done yet - continue with next phase */
                {
                BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, rphs, (temp + 1));
                pst->stage = PHASE_ADAPT_EC1;
                } /* END-IF continue to the next phase */
            else /* This was the last phase */
                {
#ifdef TDEBUG
                printf("\r\n");
#endif
                pst->stage = OPEN_EYE6;
                } /* END-ELSE last phase */
            break;

        case OPEN_EYE6:
	    temp_u = pst->misc_counter;
#ifdef TDEBUG
            printf("Number of Opened Phases = %d\r\n", (int)temp_u);
#endif

            if (temp_u < NO_OF_OPENED_PHASES) { /* Attenuator Recognition */

                /* limit to 0 - 3 */
                pst->su_flags.bits.ffe_init_scale = (pst->su_flags.bits.ffe_init_scale + 1) % 4;

#ifdef TDEBUG
                printf("New FFE scale: %d\r\n",
			(int)pst->su_flags.bits.ffe_init_scale);
#endif

                BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, rphs, 0);

		pst->misc_counter = 0;

                pst->stage = PHASE_ADAPT_EC1;
                break;
            } /* end-if */

            /*------------------------------------------------------------*/
            /* Find and set optimal sampling phase.                       */
            /*------------------------------------------------------------*/
            _CalculateOptimalPhase1(st);
            pst->stage = CALC_OPT_PHASE2;
            break;

        case CALC_OPT_PHASE2:
            _CalculateOptimalPhase2(st);
            pst->stage = CALC_OPT_PHASE3;
            break;

        case CALC_OPT_PHASE3:
            _CalculateOptimalPhase3(st);
            pst->stage = CALC_OPT_PHASE4;
            break;

        case CALC_OPT_PHASE4:
            _CalculateOptimalPhase4(st);
            pst->stage = CALC_OPT_PHASE;
            break;

        case CALC_OPT_PHASE:
            BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, rphs, _CalculateOptimalPhase(st) );

            sdcore_SetMeterTimer(st, DEFAULT_METER);

            pst->stage = SU_WAIT_METER_INTERVAL1;
            pst->stage2 = OPEN_EYE7;
            break;

        case OPEN_EYE7:

            /*------------------------------------------------------------*/
            /* Adapt EC and re-open eye at the optimal phase              */
            /*------------------------------------------------------------*/
#ifdef TDEBUG
            PREFIX;
            printf("Adapt EC @ opt phase\r\n");
#endif
            _AdaptEc(0, HIGHEST_GAIN, (BP_U_16BIT)39444); /* Adapt EC at highest gain */
            pst->stage = ADAPT_EC1;

            break;

        case ADAPT_EC1: /* EC adaptation at optimal phase */
            TIMER_BREAK(TIMERMASK_T3);
            _AdaptEc(0, HIGHER_GAIN, (BP_U_16BIT)63000); /* Adapt EC at higher gain */
            pst->stage = ADAPT_EC2;
            break;

        case ADAPT_EC2: /* EC adaptation at optimal phase */
            TIMER_BREAK(TIMERMASK_T3);
            _AdaptEc(0, HIGH_GAIN, (BP_U_16BIT)63000); /* Adapt EC at high gain */
            pst->stage = ADAPT_EC3;
            break;

        case ADAPT_EC3: /* EC adaptation at optimal phase */
            TIMER_BREAK(TIMERMASK_T3);
            pst->stage = OPEN_EYE_AT_OPTIMAL_PHASE1;
            break;



        case OPEN_EYE_AT_OPTIMAL_PHASE1:
            /* ------------------------------ */
            /* Initialize DAGC, DFE, and FFE: */
            /* ------------------------------ */
            _InitDagc();
            _InitDfe();
            _InitFfe(st);

            /* ---------------------------------- */
            /* Adapt DAGC @ HIGH Gain, Use Slicer */
            /* ---------------------------------- */
            _InitDetector();
            _AdaptDagcEqErrorMode();
            SET_WORD(bp_ptr, t3_low, t3_high, 5000);
	    RESTART_T3;

            pst->stage = OPEN_EYE_AT_OPTIMAL_PHASE2;
            break;

        case OPEN_EYE_AT_OPTIMAL_PHASE2:
            TIMER_BREAK(TIMERMASK_T3);

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF);

            /* ---------- */
            /* Adapt FFE: */
            /* ---------- */

            _AdaptAllFfe();
            SET_WORD(bp_ptr, t3_low, t3_high, 10000);
	    RESTART_T3;

            pst->stage = OPEN_EYE_AT_OPTIMAL_PHASE3;
            break;

        case OPEN_EYE_AT_OPTIMAL_PHASE3:
            TIMER_BREAK(TIMERMASK_T3);

            /* -------------- */
            /* Adapt FFE+DFE: */
            /* -------------- */

            _AdaptDfe();
            SET_WORD(bp_ptr, t3_low, t3_high, 10000);
	    RESTART_T3;

            pst->stage = OPEN_EYE_AT_OPTIMAL_PHASE4;
            break;

        case OPEN_EYE_AT_OPTIMAL_PHASE4:
            TIMER_BREAK(TIMERMASK_T3);

            _FreezeDfeFfe();
#ifdef TDEBUG
            printf("ReOpen Eye At Optimal Phase Completed\r\n");
#endif
	    bp_eq_read(DAGC_HIGH, data);

            if (data[1] < dagc_th[pst->su_flags.bits.ffe_init_scale]) {
		/* DAGC NOT OK */

                /* limit to 0 - 3 */
                pst->su_flags.bits.ffe_init_scale = (pst->su_flags.bits.ffe_init_scale + 1) % 4;

#ifdef TDEBUG
                PREFIX;
                printf("Reopen Opt. phase failed!\r\n");
                printf("FFE Scale changed to %d\r\n",
			(int)pst->su_flags.bits.ffe_init_scale);
#endif

                pst->stage = OPEN_EYE_AT_OPTIMAL_PHASE1;
                break;
            }

            _AdaptAllFfe();
            _AdaptDfe();

            SET_WORD(bp_ptr, sut2_low, sut2_high, 90);
	    RESTART_SUT2;
            pst->stage = OPEN_EYE_AT_OPTIMAL_PHASE5;
            break;


        case OPEN_EYE_AT_OPTIMAL_PHASE5: /* Open eye at optimal phase */
            TIMER_BREAK(TIMERMASK_SUT2);

            /* --------------------------- */
            /* Adapt FFE+DFE @ NORMAL Gain */
            /* --------------------------- */

            BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_gain, NORMAL_GAIN);          /* FFE NORMAL adaptation gain */
            BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_gain, NORMAL_GAIN);          /* DFE NORMAL adaptation gain */


            SET_WORD(bp_ptr, sut2_low, sut2_high, 90);
	    RESTART_SUT2;
            pst->stage = OPEN_EYE_AT_OPTIMAL_PHASE6;
            break;

        case OPEN_EYE_AT_OPTIMAL_PHASE6: /* Open eye at optimal phase */
            TIMER_BREAK(TIMERMASK_SUT2);

            _FreezeDfeFfe();

	    bp_eq_read(DAGC_HIGH, data);
            temp_s16 = ((BP_S_16BIT)data[1] << 4);  /* High Byte only */

#ifdef TDEBUG

            PREFIX;
            printf("DAGC (After Reopen Eye) = %d\r\n", (int)temp_s16);

            _PrintfFFECoefs();
#endif

            if (temp_s16 > 1700) {
                /* scale DAGC/FFE Coeffs */
		_ScaleDagc0875();
		_ScaleFfe125();

#ifdef TDEBUG
                PREFIX;
                printf("DAGC>1700:  DAGC*0.875 & FFE*1.25\r\n");
#endif
            }


            /* -------------------------------- */
            /* Adapt DAGC Error Mode adaptation */
            /* -------------------------------- */

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, eq_error_adaptation, ON);
            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_gain, NORMAL_GAIN);
            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, ON); /* Adapt DAGC */

            SET_WORD(bp_ptr, t3_low, t3_high, 20000);
	    RESTART_T3;

            pst->stage = OPEN_EYE_AT_OPTIMAL_PHASE7;
            break;

        case OPEN_EYE_AT_OPTIMAL_PHASE7:
            TIMER_BREAK(TIMERMASK_T3);

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF);

            pst->stage = WAIT_FOR_T1MIN;
            break;


        /*------------------------------------------------------------*/
        /* Wait for T1MIN timeout before transmitting CS1             */
        /*------------------------------------------------------------*/
        case WAIT_FOR_T1MIN: /* Wait for T1 min */

            /*
             * If Bt Other Side or Zip Restart, then don't wait for
             * T1 Min, otherwise wait.
             */
            if (st->special_modes & SDCORE_MODE_HTUC_EXTRADELAYS) {
                TIMER_BREAK(TIMERMASK_SUT3);
	    }

#ifdef TDEBUG
            _PrintfDAGC();
            _PrintfFFECoefs();
#endif

	    _FfeReverse();

            BP_WRITE_BIT(bp_mode_ptr, detector_modes, two_level, OFF); /* 4-level Slicing */

            /*------------------------------------------------------------*/
            /* Transmit startup sequense CS1.                             */
            /*------------------------------------------------------------*/
#ifdef TDEBUG
            PREFIX;
            printf("TX CS1\r\n");
#endif

	    bp_set_tx(0, 1, st->special_modes & SDCORE_MODE_EXTS0S1 ?
		      UNSCRAMBLED_FOUR_LEVEL_DATA : SCRAMBLED_FOUR_LEVEL_ONES);

            st->status.bits.four_level_indication = ON;

#ifdef TDEBUG
            PREFIX;
            printf("Wait for 4level...\r\n");
#endif

            sdcore_SetMeterTimer(st, FOUR_LEVEL_METER);

            pst->stage = SU_WAIT_METER_INTERVAL1;
            pst->stage2 = WAIT_FOR_4LEVEL;
            break;

        case WAIT_FOR_4LEVEL: /* Wait for 4 level far-end signal */
            TIMER_BREAK(TIMERMASK_METER);

            temp = BP_READ_REG(bp_ptr, symbol_histogram);

            /* If number of +/-1 symbols exceeds threshold: 4-level signal is received */
            if (temp < FOUR_LEVEL_SIGNAL_TH) /* Far end 4-level signal not detected */
                break;

            /*------------------------------------------------------------*/
            /* 4-level far-end signal detected                            */
            /*------------------------------------------------------------*/

#ifdef TDEBUG
            PREFIX;
            printf("4-level detected\r\n");
#endif

            /*------------------------------------------------------------*/
            /* Check for Tip/Ring Reversal                                */
            /*------------------------------------------------------------*/
            if (!(st->special_modes & SDCORE_MODE_NOAUTOTR)) {
                READ_METER_REG(ber_meter_low, ber_meter_high, temp_u16, 0);
                printf("BER Meter at positive line polarity = %d\r\n",
			(int)temp_u16);
                if (temp_u16 > AUTOTR_GOODBER_TH) {
                    BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes,
				 negate_symbol, 1);
		    printf("Trying negative line polarity\r\n");
		    pst->stage = SU_WAIT_METER_INTERVAL1;
		    pst->stage2 = TIPRING1;
		    break;
		}
	    }

	    pst->stage = TIPRING3;
	    break;

	case TIPRING1:
	    TIMER_BREAK(TIMERMASK_METER);
            READ_METER_REG(ber_meter_low, ber_meter_high, temp_u16, 0);
            printf("BER Meter = %d\r\n", (int)temp_u16);
            if (temp_u16 <= AUTOTR_GOODBER_TH) {
		st->status.bits.tip_ring_indication = 1;
		pst->stage = TIPRING3;
		break;
	    }
	    printf("Still bad, waiting for good BER with either polarity\r\n");
	    pst->stage = TIPRING2;
	    break;

	case TIPRING2:
	    TIMER_BREAK(TIMERMASK_METER);
            READ_METER_REG(ber_meter_low, ber_meter_high, temp_u16, 0);
            if (temp_u16 <= AUTOTR_GOODBER_TH) {
		printf("BER Meter = %d\r\n", (int)temp_u16);
		st->status.bits.tip_ring_indication =
				bp_mode_ptr->nonlinear_ec_modes.negate_symbol;
		pst->stage = TIPRING3;
		break;
	    }
	    bp_mode_ptr->nonlinear_ec_modes.negate_symbol =
				!bp_mode_ptr->nonlinear_ec_modes.negate_symbol;
	    pst->stage = SU_WAIT_METER_INTERVAL1;
	    pst->stage2 = TIPRING2;
	    break;

	case TIPRING3:
	    printf("Tip/ring polarity appears to be %s\r\n",
		st->status.bits.tip_ring_indication ? "reversed" : "correct");

            _ShiftFfe();

            SET_WORD(bp_ptr, t3_low, t3_high, 2200);
	    RESTART_T3;
            pst->stage = WAIT_FOR_4LEVEL_2;
            break;

        case WAIT_FOR_4LEVEL_2:
            TIMER_BREAK(TIMERMASK_T3);

            sdcore_SetMeterTimer(st, NORMAL_METER);

            pst->stage = SU_WAIT_METER_INTERVAL1;
            pst->stage2 = WAIT_FOR_4LEVEL3;

            break;

        case WAIT_FOR_4LEVEL3:
            TIMER_BREAK(TIMERMASK_METER);

            _SetFelmMask(st);

            /* --------------------- */
            /* Set DAGC Target value */
            /* --------------------- */

            NORM(meter); /* Calculate meter normalization factor */
            READ_METER_REG(felm_low, felm_high, temp_s16, 0);

	    bp_eq_read(DAGC_HIGH, data);
            temp = data[1];  /* High Byte only */

            temp >>= 1;
            temp_s16 *= temp;
            temp_s16 = (BP_S_16BIT) (temp_s16 >> (6 - meter));
            temp_s16 = temp_s16 + (temp_s16 >> 1) + (temp_s16 >> 4); /* 58.333% scale */

            SET_WORD(bp_ptr, dagc_target_low, dagc_target_high, temp_s16);

#ifdef TDEBUG
            printf("DAGC Target = %d\r\n", (int)temp_s16);

            _PrintfDAGC();
            _PrintfFFECoefs();
            _PrintfDFECoefs();
#endif

            /* --------------------------------------- */
            /* DAGC self adaptation mode @ Normal gain */
            /* --------------------------------------- */

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, eq_error_adaptation, OFF);                
            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_gain, NORMAL_GAIN);

            /* ------------------ */
            /* FFE  @ Normal gain */
            /* ------------------ */

            BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_gain, NORMAL_GAIN);

            /* ---------------------- */
            /* Adapt DAGC + FFE + DFE */
            /* ---------------------- */

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, ON);
            BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, ON);
            BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_coefficients, ON);
        
            SET_WORD(bp_ptr, t3_low, t3_high, 10000);
	    RESTART_T3;
            pst->stage = ADAPT_MODULES1;
            break;

        case ADAPT_MODULES1: /* NL EC (high gain) adaptation */
            TIMER_BREAK(TIMERMASK_T3);

#ifdef TDEBUG
            printf("Adapt NLEC & EP\r\n");
#endif
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, zero_output, OFF);
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_gain, HIGH_GAIN); /* NL EC high adaptation gain */
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_coefficients, ON); /* Adapt NL EC */

            SET_WORD(bp_ptr, t3_low, t3_high, 59111);
	    RESTART_T3;
            pst->stage = ADAPT_MODULES2;
            break;

        case ADAPT_MODULES2: /* NL EC (low gain) & EP adaptation */
            TIMER_BREAK(TIMERMASK_T3);

            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_gain, NORMAL_GAIN); /* NL EC normal adaptation gain */
	    if (!(st->special_modes & SDCORE_MODE_NLECON)) {
		BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_coefficients, OFF); /* Freeze NL EC */
		BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, zero_output, ON); /* Zero NL EC output */
	    }

            BP_WRITE_BIT(bp_mode_ptr, ep_modes, adapt_gain, NORMAL_GAIN); /* EP normal adaptation gain */
            BP_WRITE_BIT(bp_mode_ptr, ep_modes, adapt_coefficients, ON); /* Adapt EP */

            SET_WORD(bp_ptr, t3_low, t3_high, 8000);
	    RESTART_T3;
            pst->stage = SET_LOS_THRESHOLD;
            break;

        case SET_LOS_THRESHOLD:
            TIMER_BREAK(TIMERMASK_T3);

            _SetLosThreshold(st);

            pst->stage = SU_WAIT_METER_INTERVAL1;
            pst->stage2 = SET_NORMAL_CONDITIONS;
            break;

        case SET_NORMAL_CONDITIONS: /* Set normal conditions */
            TIMER_BREAK(TIMERMASK_METER);

            _SetFelmMask(st);

            BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_gain, NORMAL_GAIN);

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF); /* Freeze DAGC */
            BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_coefficients, OFF); /* Freeze DFE */
            BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, OFF); /* Freeze FFE */


#ifdef TDEBUG
            printf("Final ");
            _PrintfDAGC();
            _PrintfFFECoefs();
#endif

            st->status.bits.normal_operation = ON;

#ifdef TDEBUG
            PREFIX;
            printf("Normal operation\r\n");
#endif

	    /*
	     * Unmask the meter timer interrupt:
	     * it will be used by the temp/env code
	     * as well as anyone wanting to run the BER meter.
	     */
	    BP_global_state.meter_int_count = 0;
	    bp_mode_ptr->mask_low_reg.meter = 0;

            pst->stage = NORMAL_OPERATION;
            break;

        case NORMAL_OPERATION: /* Normal adaptation stage */
	    if (!st->tempenv_num_intervals ||
		BP_global_state.meter_int_count < st->tempenv_num_intervals)
		break;
	    BP_global_state.meter_int_count = 0;
	    tempenv_adapt_dagc_dfe(st, 1);
	    tempenv_adapt_ffe(st);
            break;


        case DEACTIVATE_MODE: /* Deactivate stage */
            break;

        case IDLE: /* Idle mode */
            break;


        case SU_WAIT_METER_INTERVAL1:
            RESTART_METER;
            pst->stage = SU_WAIT_METER_INTERVAL2;
            break;

    
        case SU_WAIT_METER_INTERVAL2:
            TIMER_BREAK(TIMERMASK_METER);
	    pst->stage = pst->stage2;
            break;

        default:
	    printf("SDCORE: invalid state in _HtucControlProcess() ASM!\r\n");
            pst->stage = IDLE;
            break;

        } /* END-SWITCH operational stage */

} /* END _HtucControlProcess() */
