#include "../libc/types.h"
#include "../include/mc68302.h"
#include "../include/gpio.h"
#include "state.h"
#include "privstate.h"
#include "../libpreact/autobaud.h"
#include "../libpreact/ifctf_preact.h"

extern volatile struct mc68302_regs mc68302_regs;

static void
ifctf_stur_hello(st)
	struct sdcore_state *st;
{
	DECLARE_PST;
	union autobaud_msg rmsg;

	rmsg.bytes[0] = IPA_OPC_HELLO;
	rmsg.bytes[1] = IPA_TTYPE_R;
	rmsg.bytes[2] = mc68302_regs.pbdat & PORTB_DTR ? IPA_RTYPE_DSU :
				IPA_RTYPE_DSU_DTEDOWN;
	ifctf_preact_setcrc(&rmsg);
	printf("Responding with %08X\r\n", rmsg.word);
	tx_autobaud_msg(&pst->paste, rmsg.word, AB_TURNAROUND_TIME);
}

static void
ifctf_ack(st)
	struct sdcore_state *st;
{
	DECLARE_PST;
	union autobaud_msg rmsg;

	rmsg.bytes[0] = IPA_OPC_ACK;
	rmsg.bytes[1] = 0;
	rmsg.bytes[2] = 0;
	ifctf_preact_setcrc(&rmsg);
	printf("Responding with good ACK\r\n");
	tx_autobaud_msg(&pst->paste, rmsg.word, AB_TURNAROUND_TIME);
}

static void
ifctf_nak(st, errcode)
	struct sdcore_state *st;
{
	DECLARE_PST;
	union autobaud_msg rmsg;

	rmsg.bytes[0] = IPA_OPC_NAK;
	rmsg.bytes[1] = 0;
	rmsg.bytes[2] = errcode;
	ifctf_preact_setcrc(&rmsg);
	printf("Responding with NAK: %08X\r\n", rmsg.word);
	tx_autobaud_msg(&pst->paste, rmsg.word, AB_TURNAROUND_TIME);
}

stur_ab_ifctf_process(st, cmsg)
	struct sdcore_state *st;
	union autobaud_msg *cmsg;
{
	u_short datarate;

	switch (cmsg->bytes[0]) {
	case IPA_OPC_HELLO:
		if (cmsg->bytes[1] == IPA_TTYPE_R)
			goto bogus_stur_msg;
		if (cmsg->bytes[1] != IPA_TTYPE_C)
			goto unknown;
		printf("IFCTF STU-C Hello message, repeater hop count = %d%s\r\n",
			cmsg->bytes[2] & IPA_HOPCOUNT_MASK,
			cmsg->bytes[2] & IPA_HOPCOUNT_UNCERTAIN ?
			" (uncertain)" : "");
		ifctf_stur_hello(st);
		return(1);

	case IPA_OPC_STARTB:
		datarate = (u_short)cmsg->bytes[1] << 8 | cmsg->bytes[2];
		printf("Startup Request at %u kbps\r\n", datarate << 3);
		if (is_datarate_supported(datarate << 3)) {
			st->user_data_rate = datarate;
			st->status.bits.preact_done = 1;
			ifctf_ack(st);
		} else {
			printf("Data rate not supported on this hardware!\r\n");
			ifctf_nak(st, IPA_NAK_HWLIMIT);
		}
		return(1);

	case IPA_OPC_DEFECT:
		printf("Segment Defect message\r\n");
		ifctf_ack(st);
		return(1);

	case IPA_OPC_ACK:
	case IPA_OPC_NAK:
	bogus_stur_msg:
		printf("Serious configuration error: STU-R message received!\r\n");
		return(0);

	default:
	unknown:
		printf("Unknown message opcode\r\n");
		ifctf_nak(st, IPA_NAK_UNKNOWN);
		return(1);
	}
}

static void
ab_flavor_resp(st)
	struct sdcore_state *st;
{
	DECLARE_PST;
	union autobaud_msg rmsg;

	rmsg.bytes[0] = AB_OPC_HTUR_DISC;
	rmsg.bytes[1] = st->cnxab_htur_flavor[0];
	rmsg.bytes[2] = st->cnxab_htur_flavor[1];
	printf("Responding with flavor ID %02X %02X\r\n",
		rmsg.bytes[1], rmsg.bytes[2]);
	tx_autobaud_msg_csum(&pst->paste, &rmsg, AB_TURNAROUND_TIME);
}

stur_ab_cnx_process(st, cmsg)
	struct sdcore_state *st;
	union autobaud_msg *cmsg;
{
	DECLARE_PST;
	u_short datarate;

	if (cmsg->bytes[0] ^ cmsg->bytes[1] ^ cmsg->bytes[2] ^ cmsg->bytes[3]) {
		printf("Bad checksum\r\n");
nak:		printf("Responding with NAK\r\n");
		tx_autobaud_msg(&pst->paste, 0xF00000F0, AB_TURNAROUND_TIME);
		return(1);
	}

	switch (cmsg->bytes[0]) {
	case AB_OPC_HTUC_DISC:
		st->cnxab_htuc_flavor[0] = cmsg->bytes[1];
		st->cnxab_htuc_flavor[1] = cmsg->bytes[2];
		if (st->cnxab_response(st) < 0)
			return(0);
		pst->open_attempts = 1;		/* overloading the var */
		ab_flavor_resp(st);
		return(1);

	case AB_OPC_HTUR_DISC:
		printf("Serious configuration error: HTU-R message received!\r\n");
		return(0);

	case AB_OPC_PROBE:
		printf("HTU-C requests line probe, don't know how to respond\r\n");
		return(0);

	case AB_OPC_SETRATE:
		datarate = (u_short)cmsg->bytes[1] << 8 | cmsg->bytes[2];
		printf("Startup Request at %u kbps\r\n", datarate << 3);
		if (!is_datarate_supported(datarate << 3)) {
			printf("Data rate not supported on this hardware!\r\n");
			goto nak;
		}
		if (!pst->open_attempts) {
			printf("Protocol violation: startup req with no flavor ID\r\n");
			return(0);
		}
		st->user_data_rate = datarate;
		st->status.bits.preact_done = 1;
		printf("Responding with echo of request\r\n");
		tx_autobaud_msg(&pst->paste, cmsg->word, AB_TURNAROUND_TIME);
		return(1);

	case AB_OPC_UNLOCK:
		printf("Conexant/Mindspeed unlock msg, responding with key 7FFF8000\r\n");
		tx_autobaud_msg(&pst->paste, 0x7FFF8000, AB_TURNAROUND_TIME);
		return(1);

	default:
		printf("Unknown AutoBaud opcode\r\n");
		goto nak;
	}
}
