#include "types.h"
#include "stdio.h"
#include "mc68302.h"
#include "cp.h"
#include "intr.h"
#include "sccsave.h"
#include "param.h"

extern caddr_t m68000_vectors[];
extern volatile struct mc68302_regs mc68302_regs;
extern volatile struct mc68302_scc_bd scc2_rx_bds[8];
extern volatile struct mc68302_scc_hdlcparams scc2_params;
extern struct scc_saved_initstate scc2_saved_initstate;

extern caddr_t buffers;
extern int scc_running;
extern caddr_t saved_timer1_vec;

static int rx_ring_headptr;

setup_hdlcdump()
{
	caddr_t bufptr;
	int i;

	if (mc68302_regs.scc2.scm & (SCM_ENR|SCM_ENT) || scc_running) {
		printf("SCC2 is already running\r\n");
		return(0);
	}

	mc68302_regs.scc2.scm = SCM_SOFT_CTS_CD | SCM_MODE_HDLC;
	for (i = 0, bufptr = buffers; i < 8; i++) {
		scc2_rx_bds[i].bd_flags = 0xC000;
		scc2_rx_bds[i].bd_bufptr = bufptr;
		bufptr += BUFSIZE;
	}
	scc2_rx_bds[7].bd_flags = 0xE000;
	rx_ring_headptr = 0;
	scc2_params.rfcr = 0x50;
	scc2_params.tfcr = 0x50;
	scc2_params.mrblr = BUFSIZE;
	scc2_params.c_mask_l = 0xF0B8;
	scc2_params.disfc = 0;
	scc2_params.crcec = 0;
	scc2_params.abtsc = 0;
	scc2_params.nmarc = 0;
	scc2_params.retrc = 0;
	scc2_params.mflr = BUFSIZE;
	scc2_params.hmask = 0;
	mc68302_regs.scc2.scm |= SCM_ENR;
	scc_running = 1;
	printf("SCC2 HDLC receiver armed\r\n");
	return(0);
}

setup_capture()
{
	if (mc68302_regs.scc2.scm & (SCM_ENR|SCM_ENT) || scc_running) {
		printf("SCC2 is already running\r\n");
		return(0);
	}

	mc68302_regs.scc2.scm = SCM_SOFT_CTS_CD | SCM_BISYNC_EXSYN
				| SCM_BISYNC_NTSYN | SCM_BISYNC_REVD
				| SCM_MODE_BISYNC;
	scc2_rx_bds[0].bd_flags = 0xC000;
	scc2_rx_bds[0].bd_bufptr = buffers;
	scc2_rx_bds[1].bd_flags = 0;
	scc2_rx_bds[2].bd_flags = 0;
	scc2_rx_bds[3].bd_flags = 0;
	scc2_rx_bds[4].bd_flags = 0;
	scc2_rx_bds[5].bd_flags = 0;
	scc2_rx_bds[6].bd_flags = 0;
	scc2_rx_bds[7].bd_flags = 0x2000;
	scc2_params.rfcr = 0x50;
	scc2_params.tfcr = 0x50;
	scc2_params.mrblr = ONESHOT;
	scc2_params.rbdn = 0;
	mc68302_regs.scc2.scm |= SCM_ENR;
	scc_running = 2;
	printf("Bit stream capture requested\r\n");
	return(0);
}

scc_stop()
{
	mc68302_regs.imr &= ~INTMASK_SCC2;
	m68000_vectors[INTVEC_TIMER1] = saved_timer1_vec;
	mc68302_regs.scc2.scm &= ~(SCM_ENR|SCM_ENT);
	scc_running = 0;
	scc2_params.rx_int_state = scc2_saved_initstate.rx_int_state;
	scc2_params.rbdn = scc2_saved_initstate.rbdn;
	scc2_params.tx_int_state = scc2_saved_initstate.tx_int_state;
	scc2_params.tbdn = scc2_saved_initstate.tbdn;
	return(0);
}

hdlc_rx_proc()
{
	volatile struct mc68302_scc_bd *bd;
	u_short flags, len;
	int i, rem16;
	u_char *cp;

	bd = scc2_rx_bds + rx_ring_headptr;
	flags = bd->bd_flags;
	if (flags & 0x8000)
		return;
	len = bd->bd_len;
	printf("Rx frame, length=%d, flags:", len);
	if (flags & 0x0800)
		printf(" L");
	if (flags & 0x0400)
		printf(" F");
	if (flags & 0x0020)
		printf(" LG");
	if (flags & 0x0010)
		printf(" NO");
	if (flags & 0x0008)
		printf(" AB");
	if (flags & 0x0004)
		printf(" CR");
	if (flags & 0x0002)
		printf(" OV");
	if (flags & 0x0001)
		printf(" CD");
	printf("\r\n");
	if (len > scc2_params.mrblr)
		len = scc2_params.mrblr;
	for (cp = bd->bd_bufptr, i = 0; i < len; cp++, i++) {
		rem16 = i & 0xF;
		if (!rem16)
			printf("%04X:", i);
		printf("%s %02X", rem16 & 3 ? "" : " ", (u_long) *cp);
		if (rem16 == 15 || i == (len - 1))
			printf("\r\n");
	}
	printf("\r\n");
	bd->bd_flags = (flags & 0x7000) | 0x8000;
	rx_ring_headptr++;
	if (rx_ring_headptr >= 8)
		rx_ring_headptr = 0;
}

capture_proc()
{
	volatile struct mc68302_scc_bd *bd;
	u_short flags, len;
	int i, rem16;
	u_char *cp;

	bd = scc2_rx_bds;
	flags = bd->bd_flags;
	if (flags & 0x8000)
		return;
	len = bd->bd_len;
	printf("Capture status = %04X, length = %d\r\n", flags, len);
	if (len > scc2_params.mrblr)
		len = scc2_params.mrblr;
	for (cp = bd->bd_bufptr, i = 0; i < len; cp++, i++) {
		rem16 = i & 0xF;
		if (!rem16)
			printf("%04X:", i);
		printf("%s %02X", rem16 & 3 ? "" : " ", (u_long) *cp);
		if (rem16 == 15 || i == (len - 1))
			printf("\r\n");
	}
	printf("\r\n");
	scc_stop();
}
