#include "types.h"
#include "stdio.h"
#include "ctype.h"
#include "strings.h"
#include "console.h"
#include "../sdcore/api.h"
#include "../sdcore/state.h"
#include "../lib8973/typedefs.h"
#include "../lib8973/bitpump.h"

extern struct sdcore_api *sdcore_api;
extern struct sdcore_state sdcore_state;

extern int show_meters_raw;

cmd_set_ipl(cmd)
	char *cmd;
{
	struct param params[1];
	u_long ipl;

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return(0);
	if (parse_hexnum(params[0].text, params[0].len, &ipl))
		return(0);
	if (ipl & 0xFFFFFFF8) {
		error("Value too large");
		return(0);
	}
	splx(ipl);
	return(0);
}

cmd_set_bclk(cmd)
	char *cmd;
{
	struct param params[1];
	u_long val;

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return(0);
	if (parse_decnum(params[0].text, params[0].len, &val))
		return(0);
	if (!sdcore_api->datarate_issup(val)) {
		error("Data rate %d not supported", val);
		return(0);
	}
	sdcore_api->set_bclk(&sdcore_state, val >> 3);
	return(0);
}

cmd_set_datarate(cmd)
	char *cmd;
{
	struct param params[1];
	u_long val;

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return(0);
	if (parse_decnum(params[0].text, params[0].len, &val))
		return(0);
	if (!sdcore_api->datarate_issup(val)) {
		error("Data rate %d not supported", val);
		return(0);
	}
	sdcore_state.user_data_rate = val >> 3;
	return(0);
}

static const struct keyword terminal_types[] = {
	{"CO", 1, _HTUC},
	{"HTUC", 4, _HTUC},
	{"HTUR", 4, _HTUR},
	{"RT", 1, _HTUR},
	{"STUC", 4, _HTUC},
	{"STUR", 4, _HTUR},
	{NULL, 0, -1}};

cmd_set_terminal(cmd)
	char *cmd;
{
	struct param params[1];
	int code, ambig = 0;

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return(0);
	code = find_keyword(params[0].text, params[0].len, terminal_types,
				&ambig);
	if (code != -1)
		sdcore_state.terminal_type = code;
	else {
		if (ambig)
			error("Ambiguous mode keyword");
		else
			error("Invalid mode keyword");
	}
	return(0);
}

static const struct keyword preact_types[] = {
	{"AB", 1, PREACT_TYPE_AUTOBAUD},
	{"AUTOBAUD", 1, PREACT_TYPE_AUTOBAUD},
	{"CM", 1, PREACT_TYPE_CM},
	{"IFCTF", 1, PREACT_TYPE_IFCTF},
	{"NONE", 1, PREACT_TYPE_NONE},
	{NULL, 0, -1}};

cmd_set_preact(cmd)
	char *cmd;
{
	struct param params[1];
	int code, ambig = 0;

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return(0);
	code = find_keyword(params[0].text, params[0].len, preact_types,
				&ambig);
	if (code != -1)
		sdcore_state.preact_type = code;
	else {
		if (ambig)
			error("Ambiguous mode keyword");
		else
			error("Invalid mode keyword");
	}
	return(0);
}

static const struct keyword quat_orientations[] = {
	{"MAGFIRST", 1, _SERIAL_MAGFIRST},
	{"SIGNFIRST", 1, _SERIAL_SIGNFIRST},
	{NULL, 0, -1}};

cmd_set_quat(cmd)
	char *cmd;
{
	struct param params[1];
	int code, ambig = 0;

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return(0);
	code = find_keyword(params[0].text, params[0].len, quat_orientations,
				&ambig);
	if (code != -1)
		sdcore_state.quat_orient = code;
	else {
		if (ambig)
			error("Ambiguous mode keyword");
		else
			error("Invalid mode keyword");
	}
	return(0);
}

static const struct keyword mdisplay_modes[] = {
	{"NORMALIZED", 1, 0},
	{"RAW", 1, 1},
	{NULL, 0, -1}};

cmd_set_mdisplay(cmd)
	char *cmd;
{
	struct param params[1];
	int code, ambig = 0;

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return(0);
	code = find_keyword(params[0].text, params[0].len, mdisplay_modes,
				&ambig);
	if (code != -1)
		show_meters_raw = code;
	else {
		if (ambig)
			error("Ambiguous mode keyword");
		else
			error("Invalid mode keyword");
	}
	return(0);
}

cmd_set_specialmodes(cmd)
	char *cmd;
{
	struct param params[1];
	u_long val;

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return(0);
	if (parse_hexnum(params[0].text, params[0].len, &val))
		return(0);
	if (val & 0xFFFF0000) {
		error("Value too large");
		return(0);
	}
	sdcore_state.special_modes = val;
	return(0);
}

cmd_set_txgain(cmd)
	char *cmd;
{
	struct param params[1];
	int val;

	if (parse_cmd_plusminus(cmd, 1, 1, params))
		return(0);
	if (parse_signed_decnum(params[0].text, params[0].len, &val))
		return(0);
	sdcore_state.txgain_setting = val;
	sdcore_state.special_modes |= SDCORE_MODE_TXGAIN_OVERRIDE;
	return(0);
}

cmd_set_abflavor(cmd)
	char *cmd;
{
	struct param params[2];
	u_long b1, b2;

	if (parse_cmd(cmd, 2, 2, params, NULL, NULL, NULL))
		return(0);
	if (parse_hexnum(params[0].text, params[0].len, &b1))
		return(0);
	if (b1 & 0xFFFFFF00) {
		error("Value too large");
		return(0);
	}
	if (parse_hexnum(params[1].text, params[1].len, &b2))
		return(0);
	if (b2 & 0xFFFFFF00) {
		error("Value too large");
		return(0);
	}
	sdcore_state.cnxab_htuc_flavor[0] = b1;
	sdcore_state.cnxab_htuc_flavor[1] = b2;
	return(0);
}

cmd_set_fallback(cmd)
	char *cmd;
{
	struct param params[1];
	u_long val;

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return(0);
	if (parse_decnum(params[0].text, params[0].len, &val))
		return(0);
	if (val > 255) {
		error("Value too large");
		return(0);
	}
	sdcore_state.preact_fallback = val;
	return(0);
}

cmd_set_tempenv(cmd)
	char *cmd;
{
	struct param params[1];
	u_long val;

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return(0);
	if (parse_decnum(params[0].text, params[0].len, &val))
		return(0);
	if (val > 255) {
		error("Value too large");
		return(0);
	}
	sdcore_state.tempenv_num_intervals = val;
	return(0);
}

static const struct keyword set_subcmds[] = {
	{"ABFLAVOR", 1, (long) &cmd_set_abflavor},
	{"BCLK", 1, (long) &cmd_set_bclk},
	{"DATARATE", 1, (long) &cmd_set_datarate},
	{"FALLBACK", 1, (long) &cmd_set_fallback},
	{"IPL", 1, (long) &cmd_set_ipl},
	{"MDISPLAY", 1, (long) &cmd_set_mdisplay},
	{"PREACTIVATION", 1, (long) &cmd_set_preact},
	{"QUAT", 1, (long) &cmd_set_quat},
	{"SPECIALMODES", 1, (long) &cmd_set_specialmodes},
	{"TEMPENV", 3, (long) &cmd_set_tempenv},
	{"TERMINAL", 3, (long) &cmd_set_terminal},
	{"TXGAIN", 2, (long) &cmd_set_txgain},
	{NULL, 0, -1}};

cmd_set(cmd)
	char *cmd;
{
	char *cp = cmd, *subcmd;
	int count;
	void (*subcmdfunc)();
	int ambigsubcmd;

	while (*cp == ' ')
		cp++;
	if (!issym(*cp)) {
		error("Parameter required");
		return(0);
	}
	for (subcmd = cp, count = 0; issym(*cp); cp++)
		count++;
	ambigsubcmd = 0;
	subcmdfunc = (void (*)())
			find_keyword(subcmd, count, set_subcmds, &ambigsubcmd);
	if ((long) subcmdfunc != -1)
		(*subcmdfunc)(cp);
	else {
		if (ambigsubcmd)
			error("Ambiguous parameter");
		else
			error("Illegal parameter");
	}
	return(0);
}
