#include "types.h"
#include "stdio.h"
#include "ctype.h"
#include "strings.h"
#include "console.h"
#include "mc68302.h"
#include "cp.h"
#include "intr.h"
#include "sccsave.h"

extern volatile struct mc68302_regs mc68302_regs;
extern volatile struct mc68302_scc_bd scc1_rx_bds[8];
extern volatile struct mc68302_scc_bd scc1_tx_bds[8];
extern volatile struct mc68302_scc_hdlcparams scc1_params;
extern struct scc_saved_initstate scc1_saved_initstate;

extern int hdlc_running;

static int rx_ring_headptr;

cmd_scc_hdlc(cmd)
	char *cmd;
{
	struct param params[2];
	u_long bufstart, maxframe;
	caddr_t bufptr;
	int i;

	if (parse_cmd(cmd, 2, 2, params, NULL, NULL, NULL))
		return(0);
	if (parse_hexnum(params[0].text, params[0].len, &bufstart))
		return(0);
	if (bufstart & 1) {
		error("The buffer address must be even");
		return(0);
	}
	if (parse_hexnum(params[1].text, params[1].len, &maxframe))
		return(0);
	if (maxframe & 0xFFFF0000) {
		error("Value too large");
		return(0);
	}
	if (maxframe & 1) {
		error("The maximum frame length must be even");
		return(0);
	}
	if (mc68302_regs.scc1.scm & (SCM_ENR|SCM_ENT) || hdlc_running) {
		error("SCC1 is already running");
		return(0);
	}

	mc68302_regs.scc1.scm = SCM_SOFT_CTS_CD | SCM_MODE_HDLC;
	for (i = 0, bufptr = (caddr_t)bufstart; i < 8; i++) {
		scc1_rx_bds[i].bd_flags = 0xC000;
		scc1_rx_bds[i].bd_bufptr = bufptr;
		bufptr += maxframe;
	}
	scc1_rx_bds[7].bd_flags = 0xE000;
	rx_ring_headptr = 0;
	for (i = 0; i < 8; i++) {
		scc1_tx_bds[i].bd_flags = 0;
	}
	scc1_tx_bds[7].bd_flags = 0x2000;
	scc1_params.rfcr = 0x50;
	scc1_params.tfcr = 0x50;
	scc1_params.mrblr = maxframe;
	scc1_params.c_mask_l = 0xF0B8;
	scc1_params.disfc = 0;
	scc1_params.crcec = 0;
	scc1_params.abtsc = 0;
	scc1_params.nmarc = 0;
	scc1_params.retrc = 0;
	scc1_params.mflr = maxframe;
	scc1_params.hmask = 0;
	mc68302_regs.scc1.scm |= SCM_ENR | SCM_ENT;
	hdlc_running = 1;
	return(0);
}

static const struct keyword capture_quals[] =
	{{"MSB", 1, 1},
	 {NULL, 0, -1}};

cmd_scc_capture(cmd)
	char *cmd;
{
	struct param params[2];
	struct qual quals[1];
	u_long bufstart, bufsize;

	quals[0].id = 0;
	if (parse_cmd(cmd, 2, 2, params, NULL, capture_quals, quals))
		return(0);
	if (quals[0].id && quals[0].value) {
		error("No value qualifier");
		return(0);
	}
	if (parse_hexnum(params[0].text, params[0].len, &bufstart))
		return(0);
	if (bufstart & 1) {
		error("The buffer address must be even");
		return(0);
	}
	if (parse_hexnum(params[1].text, params[1].len, &bufsize))
		return(0);
	if (bufsize & 0xFFFF0000) {
		error("Value too large");
		return(0);
	}
	if (bufsize & 1) {
		error("The buffer size must be even");
		return(0);
	}
	if (mc68302_regs.scc1.scm & (SCM_ENR|SCM_ENT) || hdlc_running) {
		error("SCC1 is already running");
		return(0);
	}

	mc68302_regs.scc1.scm = SCM_SOFT_CTS_CD | SCM_BISYNC_EXSYN
				| SCM_BISYNC_NTSYN | SCM_MODE_BISYNC;
	if (quals[0].id)
		mc68302_regs.scc1.scm |= SCM_BISYNC_REVD;
	scc1_rx_bds[0].bd_flags = 0xC000;
	scc1_rx_bds[0].bd_bufptr = (caddr_t) bufstart;
	scc1_rx_bds[1].bd_flags = 0;
	scc1_rx_bds[2].bd_flags = 0;
	scc1_rx_bds[3].bd_flags = 0;
	scc1_rx_bds[4].bd_flags = 0;
	scc1_rx_bds[5].bd_flags = 0;
	scc1_rx_bds[6].bd_flags = 0;
	scc1_rx_bds[7].bd_flags = 0x2000;
	scc1_params.rfcr = 0x50;
	scc1_params.tfcr = 0x50;
	scc1_params.mrblr = bufsize;
	scc1_params.rbdn = 0;
	mc68302_regs.scc1.scm |= SCM_ENR;
	return(0);
}

scc_stop()
{
	mc68302_regs.imr &= ~INTMASK_SCC1;
	mc68302_regs.scc1.scm &= ~(SCM_ENR|SCM_ENT);
	hdlc_running = 0;
	scc1_params.rx_int_state = scc1_saved_initstate.rx_int_state;
	scc1_params.rbdn = scc1_saved_initstate.rbdn;
	scc1_params.tx_int_state = scc1_saved_initstate.tx_int_state;
	scc1_params.tbdn = scc1_saved_initstate.tbdn;
	return(0);
}

static const struct keyword scc_subcmds[] = {
	{"CAPTURE", 1, (long) &cmd_scc_capture},
	{"HDLC", 1, (long) &cmd_scc_hdlc},
	{"STOP", 1, (long) &scc_stop},
	{NULL, 0, -1}};

cmd_scc(cmd)
	char *cmd;
{
	char *cp = cmd, *subcmd;
	int count;
	void (*subcmdfunc)();
	int ambigsubcmd;

	while (*cp == ' ')
		cp++;
	if (!issym(*cp)) {
		error("Subcommand required");
		return(0);
	}
	for (subcmd = cp, count = 0; issym(*cp); cp++)
		count++;
	ambigsubcmd = 0;
	subcmdfunc = (void (*)())
			find_keyword(subcmd, count, scc_subcmds, &ambigsubcmd);
	if ((long) subcmdfunc != -1)
		(*subcmdfunc)(cp);
	else {
		if (ambigsubcmd)
			error("Ambiguous subcommand");
		else
			error("Illegal subcommand");
	}
	return(0);
}

hdlc_rx_proc()
{
	volatile struct mc68302_scc_bd *bd;
	u_short flags, len;
	int i, rem16;
	u_char *cp;

	bd = scc1_rx_bds + rx_ring_headptr;
	flags = bd->bd_flags;
	if (flags & 0x8000)
		return;
	len = bd->bd_len;
	printf("Rx frame, length=%d, flags:", len);
	if (flags & 0x0800)
		printf(" L");
	if (flags & 0x0400)
		printf(" F");
	if (flags & 0x0020)
		printf(" LG");
	if (flags & 0x0010)
		printf(" NO");
	if (flags & 0x0008)
		printf(" AB");
	if (flags & 0x0004)
		printf(" CR");
	if (flags & 0x0002)
		printf(" OV");
	if (flags & 0x0001)
		printf(" CD");
	printf("\r\n");
	if (len > scc1_params.mrblr)
		len = scc1_params.mrblr;
	for (cp = bd->bd_bufptr, i = 0; i < len; cp++, i++) {
		rem16 = i & 0xF;
		if (!rem16)
			printf("%04X:", i);
		printf("%s %02X", rem16 & 3 ? "" : " ", (u_long) *cp);
		if (rem16 == 15 || i == (len - 1))
			printf("\r\n");
	}
	printf("\r\n");
	bd->bd_flags = (flags & 0x7000) | 0x8000;
	rx_ring_headptr++;
	if (rx_ring_headptr >= 8)
		rx_ring_headptr = 0;
}
