/*
 * This is the new EOC message dump code which understands message
 * boundaries more reliably, checks RFC 1662 transparency and FCS,
 * and is expected to be well-behaved enough to be always enabled.
 */

#include "types.h"
#include "globals.h"
#include "stdio.h"

/* The following are tunable and may be bumped up if needed */
#define	EOCMSG_MAXSIZE		64
#define	EOCDUMP_CIRCBUF_SIZE	16

struct eoc_msg {
	u_short	len;
	u_char	data[EOCMSG_MAXSIZE];
};

/* interrupt-time EOC receiver states */
#define	EOCSTATE_WAITFOR7E	0x00
#define	EOCSTATE_WAITFORMSG	0x01
#define	EOCSTATE_RECEIVING	0x02

int eoc_intrx_state, eoc_intrx_count;
struct eoc_msg eocdump_circbuf[EOCDUMP_CIRCBUF_SIZE];
int eocdump_read_ptr, eocdump_write_ptr;
int eoc_dumper_active;

eocdump_init()
{
	eoc_intrx_state = EOCSTATE_WAITFOR7E;
	eocdump_read_ptr = eocdump_write_ptr = 0;
	eoc_dumper_active = 0;
}

eocdump_input(inbyte)
{
	if (inbyte == 0x7E) {
		if (eoc_intrx_state == EOCSTATE_RECEIVING) {
			eocdump_circbuf[eocdump_write_ptr++].len =
				eoc_intrx_count;
			if (eocdump_write_ptr >= EOCDUMP_CIRCBUF_SIZE)
				eocdump_write_ptr = 0;
		}
		eoc_intrx_state = EOCSTATE_WAITFORMSG;
		return;
	}
	switch (eoc_intrx_state) {
	case EOCSTATE_WAITFORMSG:
		eoc_intrx_state = EOCSTATE_RECEIVING;
		eoc_intrx_count = 0;
		/* FALL THRU */
	case EOCSTATE_RECEIVING:
		if (eoc_intrx_count >= EOCMSG_MAXSIZE) {
			eocdump_circbuf[eocdump_write_ptr++].len = 0xFFFF;
			if (eocdump_write_ptr >= EOCDUMP_CIRCBUF_SIZE)
				eocdump_write_ptr = 0;
			eoc_intrx_state = EOCSTATE_WAITFOR7E;
			return;
		}
		eocdump_circbuf[eocdump_write_ptr].data[eoc_intrx_count++] =
			inbyte;
	}
}

#define	DUMPER_MSGTYPE_NORMAL		0x00
#define	DUMPER_MSGTYPE_MALFORMED	0x01
#define	DUMPER_MSGTYPE_BABBLE		0x02

eocdump_process()
{
	struct eoc_msg msgin;
	u_char decmsg[EOCMSG_MAXSIZE], *dp;
	int type, len, i;

	if (!eoc_dumper_active) {
		if (eoc_intrx_state != EOCSTATE_WAITFOR7E) {
			printf("EOC appears active\r\n");
			eoc_dumper_active = 1;
		} else
			return;
	}
	if (eocdump_read_ptr == eocdump_write_ptr)
		return;
	bcopy(&eocdump_circbuf[eocdump_read_ptr], &msgin,
		sizeof(struct eoc_msg));
	eocdump_read_ptr++;
	if (eocdump_read_ptr >= EOCDUMP_CIRCBUF_SIZE)
		eocdump_read_ptr = 0;
	len = msgin.len;
	if (len > EOCMSG_MAXSIZE) {
		type = DUMPER_MSGTYPE_BABBLE;
		len = EOCMSG_MAXSIZE;
		printf("EOC babble:");
		dp = msgin.data;
	} else if (rfc1662_decode(msgin.data, decmsg, &len) < 0) {
		type = DUMPER_MSGTYPE_MALFORMED;
		printf("EOC message (malformed octet transparency):");
		dp = msgin.data;
	} else {
		type = DUMPER_MSGTYPE_NORMAL;
		printf("EOC message:");
		dp = decmsg;
	}
	for (i = 0; i < len; i++)
		printf("%s%02X", i&15 ? " " : "\r\n    ", dp[i]);
	if (type == DUMPER_MSGTYPE_NORMAL) {
		i = rfc1662_fcs16_good(decmsg, len);
		printf(" (FCS %s)", i ? "good" : "bad");
	}
	printf("\r\n");
	if (type == DUMPER_MSGTYPE_NORMAL && i)
		stats.eoc_good_msgs_rcvd++;
	else
		stats.eoc_garbage_rcvd++;
}

rfc1662_decode(src, dest, lenp)
	u_char *src, *dest;
	int *lenp;
{
	u_char *cp, *dp, *end;
	int c, len;

	cp = src;
	dp = dest;
	end = src + *lenp;
	for (len = 0; cp < end; len++) {
		c = *cp++;
		if (c != 0x7D) {
			*dp++ = c;
			continue;
		}
		if (cp >= end)
			return(-1);
		c = *cp++;
		switch (c) {
		case 0x5D:
			*dp++ = 0x7D;
			break;
		case 0x5E:
			*dp++ = 0x7E;
			break;
		default:
			return(-1);
		}
	}
	*lenp = len;
	return(0);
}
