/*
 * Logical ATM layer, Rx direction
 */

#include "types.h"
#include "param.h"
#include "globals.h"
#include "atm.h"
#include "sdsl_asm.h"

extern u_long crc32();
extern struct buffer *get_buffer_from_queue();

/* This is the main ATM WAN Rx entry */
atm_cell_rx(cell, badhec)
	struct atmcell *cell;
{
	u_long cellhdr_addr;
	struct vc *vc;
	int i;

	if (cell->cellhdr[0] & 0xF0 && !badhec)
		stats.atm_rx_nonzero_gfc++;
	bcopy(cell->cellhdr, &cellhdr_addr, 4);
	if (!(cellhdr_addr & 0x0FFFFFFE))
		return;
	cellhdr_addr &= 0x0FFFFFF8;
	for (vc = l2conv_global.vc_table, i = 0; i < l2conv_global.nvcs;
	     vc++, i++)
		if (cellhdr_addr == *(u_long *)vc->cellhdr_normal) {
			atm_cell_for_vc(cell, vc);
			stats.atm_rx_good_cells++;
			return;
		}
	if (!badhec)
		stats.atm_rx_wrong_cells++;
}

atm_cell_for_vc(cell, vc)
	struct atmcell *cell;
	struct vc *vc;
{
	int s;
	struct buffer *buf;
	struct aal5_trailer *tr;
	int len;

	switch (vc->aal5_rx_state) {
	case AAL5RX_STATE_IDLE:
		buf = l2conv_api->get_buffer();
		buf->buf_vc = vc;
		vc->aal5_rx_buf = buf;
		vc->aal5_rx_state = AAL5RX_STATE_INPROGRESS;
		vc->aal5_rx_ptr = buf->buf_dataspc + 4;
		buf->buf_dataoff = 4;
		vc->aal5_rx_count = 0;
		vc->aal5_rx_crcaccum = AAL5_CRC32_INIT;
		vc->atm_rx_packets++;
		break;
	case AAL5RX_STATE_INPROGRESS:
		buf = vc->aal5_rx_buf;
		break;
	}
	if (vc->aal5_rx_state == AAL5RX_STATE_INPROGRESS)
		atmrx_work_cell_in(cell, vc);
	if (!(cell->cellhdr[3] & 0x02))
		return;
	if (vc->aal5_rx_state == AAL5RX_STATE_DISCARD) {
		vc->aal5_rx_state = AAL5RX_STATE_IDLE;
		return;
	}
	if (vc->aal5_rx_state != AAL5RX_STATE_INPROGRESS) {
		printf("Fatal internal error: invalid AAL5 Rx state\r\n");
		asm("trap #0");
	}
	vc->aal5_rx_state = AAL5RX_STATE_IDLE;
	if (vc->aal5_rx_crcaccum != AAL5_CRC32_GOOD) {
		l2conv_api->free_buffer(buf);
		vc->atm_rx_badcrc32++;
		return;
	}
	tr = (struct aal5_trailer *)(vc->aal5_rx_ptr - 8);
	len = tr->cpcs_length;
	if (!len) {
		l2conv_api->free_buffer(buf);
		vc->atm_rx_aborts++;
		return;
	}
	if (len > vc->aal5_rx_count*48-8) {
		l2conv_api->free_buffer(buf);
		vc->atm_rx_badlen++;
		return;
	}
	buf->buf_datalen = len;
	stats.a2h_rx_packets++;
	if (hdlc_tx_outstanding >= HDLC_TXQ_MAXLEN) {
		l2conv_api->free_buffer(buf);
		stats.a2h_dropped_congest++;
		return;
	}
	if (l2conv_api->atm2hdlc(buf) < 0) {
		l2conv_api->free_buffer(buf);
		stats.a2h_dropped_convfail++;
		return;
	}
	hdlc_post_tx(buf);
	stats.a2h_tx_packets++;
}

atmrx_work_cell_in(cell, vc)
	struct atmcell *cell;
	struct vc *vc;
{
	if (vc->aal5_rx_count >= AAL5_MAXCELLS) {
		vc->aal5_rx_state = AAL5RX_STATE_DISCARD;
		l2conv_api->free_buffer(vc->aal5_rx_buf);
		vc->atm_rx_giants++;
		return;
	}
	bcopy(cell->payload, vc->aal5_rx_ptr, 48);
	vc->aal5_rx_ptr += 48;
	vc->aal5_rx_count++;
	vc->aal5_rx_crcaccum = crc32(vc->aal5_rx_crcaccum, cell->payload, 48);
}
