/*
 * The Activation State Machine for Flavor N1
 * is virtually identical to the one we had on the Hack-o-Rocket
 */

#include "types.h"
#include "stdio.h"
#include "mc68302.h"
#include "gpio.h"
#include "sdsl_asm.h"
#include "globals.h"
#include "param.h"
#include "nokia.h"
#include "nokia_impl.h"

extern volatile struct mc68302_regs mc68302_regs;
extern volatile long timebase;

char * const sdsl_asm_states[8] = {
	"INACTIVE_STATE",
	"WAIT_FOR_DTR",
	"ACTIVATING_STATE",
	"FRAME_ACQU_STATE",
	"ACTIVE_STATE",
	"DEACTIVATED_STATE",
	"WAIT_FOR_LOST",
	"WAIT_FOR_LOS"
};

sdsl_asm()
{
	int dtr;

	switch (sdsl_asm_state) {

	case INACTIVE_STATE:
		nokia_prepare();
		dtr = opcore_api->dtrcheck(1, 1);
		if (dtr) {
			printf("Attempting SDSL startup\r\n");
			sdcore_api->activate(&sdcore_state);
			set_sdsl_state(ACTIVATING_STATE);
			if ((mc68302_regs.padat & PORTA_LSLED_MASK) !=
			    PORTA_LSLED_RED)
				mc68302_regs.padat =
					mc68302_regs.padat & ~PORTA_LSLED_MASK
					| PORTA_LSLED_ORANGE;
		} else {
			sdcore_api->powerdown(&sdcore_state);
			printf("Waiting for DTR\r\n");
			set_sdsl_state(WAIT_FOR_DTR);
			mc68302_regs.padat &= ~PORTA_LSLED_MASK;
		}
		return;

	case WAIT_FOR_DTR:
		dtr = opcore_api->dtrcheck(1, 0);
		if (dtr) {
			printf("Attempting SDSL startup\r\n");
			sdcore_api->activate(&sdcore_state);
			set_sdsl_state(ACTIVATING_STATE);
			mc68302_regs.padat =
					mc68302_regs.padat & ~PORTA_LSLED_MASK
					| PORTA_LSLED_ORANGE;
		}
		return;

	case ACTIVATING_STATE:
		dtr = opcore_api->dtrcheck(0, 1);
		if (!dtr) {
			set_sdsl_state(DEACTIVATED_STATE);
			return;
		}
		if (sdcore_state.status.bits.normal_operation)
			sdsl_startup_success();
		else if (sdcore_state.status.bits.activation_interval) {
		  printf("Timer expired: SDSL startup failed, will retry\r\n");
		  set_sdsl_state(DEACTIVATED_STATE);
		}
		return;

	case FRAME_ACQU_STATE:
		dtr = opcore_api->dtrcheck(0, 1);
		if (!dtr) {
			set_sdsl_state(DEACTIVATED_STATE);
			return;
		}
		if (nokia_tx_error) {
			nokia_handle_tx_error();
			return;
		}
		switch (nokia_rx_state) {
		case NOKIARX_STATE_SYNC:
			/* yay! */
			printf("Achieved successful frame lock!\r\n");
			set_sdsl_state(ACTIVE_STATE);
			mc68302_regs.pbdat |= PORTB_CD | PORTB_CTS;
			mc68302_regs.padat =
					mc68302_regs.padat & ~PORTA_LSLED_MASK
					| PORTA_LSLED_GREEN;
			stats.sdsl_activations++;
			return;
		case NOKIARX_STATE_LOSW:
			/*
			 * We must have gotten to SYNC and then to LOSW
			 * before this ground level routine got to run!
			 */
			printf("SDSL: False frame lock\r\n");
			set_sdsl_state(DEACTIVATED_STATE);
			return;
		case NOKIARX_STATE_OVERRUN:
			nokia_handle_rx_overrun();
			return;
		}
		if (timebase - sdsl_asm_state_time >= frame_acqu_time) {
			printf("No frame lock achieved\r\n");
			set_sdsl_state(DEACTIVATED_STATE);
		}
		return;

	case ACTIVE_STATE:
		dtr = opcore_api->dtrcheck(0, 1);
		if (!dtr) {
			set_sdsl_state(DEACTIVATED_STATE);
			return;
		}
		/* Framer health is our link up criterion */
		if (nokia_tx_error) {
			nokia_handle_tx_error();
			return;
		}
		switch (nokia_rx_state) {
		case NOKIARX_STATE_LOSW:
			printf("SDSL: lost Rx frame sync word (LOSW)\r\n");
			set_sdsl_state(DEACTIVATED_STATE);
			stats.sdsl_losw_shutdowns++;
			return;
		case NOKIARX_STATE_OVERRUN:
			nokia_handle_rx_overrun();
			return;
		}
		return;

	case DEACTIVATED_STATE:
		printf("Tearing SDSL link down\r\n");
		sdcore_api->deactivate(&sdcore_state);
		mc68302_regs.pbdat &= ~(PORTB_CD | PORTB_CTS);
		mc68302_regs.padat = mc68302_regs.padat & ~PORTA_LSLED_MASK
					| PORTA_LSLED_RED;
		switch (sdsl_asm_prev_state) {
		case FRAME_ACQU_STATE:
		case ACTIVE_STATE:
			sdsl_shutdown();
		}
		if (sdcore_state.terminal_type) {
			printf("Now wait for LOS\r\n");
			set_sdsl_state(WAIT_FOR_LOS);
		} else {
			printf("Now wait for LOST\r\n");
			set_sdsl_state(WAIT_FOR_LOST);
		}
		return;

	case WAIT_FOR_LOST:
		if (sdcore_state.status.bits.lost) {
			set_sdsl_state(INACTIVE_STATE);
			return;
		}
		if (los_lost_timeout && timebase - sdsl_asm_state_time >=
		    los_lost_timeout) {
		    printf("Never got LOST, proceeding to retry anyway\r\n");
			set_sdsl_state(INACTIVE_STATE);
		}
		return;

	case WAIT_FOR_LOS:
		if (sdcore_state.status.bits.los) {
			set_sdsl_state(INACTIVE_STATE);
			return;
		}
		if (los_lost_timeout && timebase - sdsl_asm_state_time >=
		    los_lost_timeout) {
			printf("Never got LOS, proceeding to retry anyway\r\n");
			set_sdsl_state(INACTIVE_STATE);
		}
		return;

	default:
		printf("Fatal internal error: invalid state in SDSL ASM\r\n");
		asm("trap #0");
	}
}

set_sdsl_state(newstate)
	int newstate;
{
	sdsl_asm_prev_state = sdsl_asm_state;
	sdsl_asm_state = newstate;
	sdsl_asm_state_time = timebase;
}
