#include "types.h"
#include "stdio.h"
#include "ctype.h"
#include "strings.h"
#include "mon302.h"
#include "mc68302.h"
#include "cp.h"
#include "gpio.h"
#include "../lib8973/typedefs.h"
#include "../lib8973/bitpump.h"
#include "../lib8973/bpstate.h"

extern volatile struct mc68302_regs mc68302_regs;
extern volatile long timebase;

extern u_short uart_baudrate_to_scon();

cmd_set_bclk(cmd)
	char *cmd;
{
	DECLARE_MODE_PTR;
	struct param params[1];
	u_long val;

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return(0);
	if (parse_decnum(params[0].text, params[0].len, &val))
		return(0);
	if (!is_datarate_supported(val)) {
		error("Invalid BCLK setting");
		return(0);
	}
	BP_global_state.symbol_rate = val >> 3;
	if (bp_mode_ptr->global_modes.mode)
		set_bitpump_clock();
	else
		_BtReset(0);
	return(0);
}

cmd_set_bitpump(cmd)
	char *cmd;
{
	DECLARE_MODE_PTR;
	struct param params[1];

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return(0);
	if (params[0].len == 2 && !strncasecmp(params[0].text, "UP", 2))
		_BtReset(0);
	else if (params[0].len == 4 && !strncasecmp(params[0].text, "DOWN", 4))
		bp_power_down();
	else
		error("Invalid setting");
	return(0);
}

cmd_set_console(cmd)
	char *cmd;
{
	struct param params[1];
	u_long baudrate;
	u_short newscon;
	long time;

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return(0);
	if (parse_decnum(params[0].text, params[0].len, &baudrate))
		return(0);
	switch (baudrate) {
	case 1200:
	case 2400:
	case 4800:
	case 9600:
	case 19200:
	case 38400:
	case 57600:
	case 115200:
		newscon = uart_baudrate_to_scon(baudrate);
		break;
	default:
		error("Baud rate %u not supported", baudrate);
		return(0);
	}
	/* spare the unwitting user the dead loop */
	if (current_ipl() >= 4) {
		error("SET CONSOLE doesn't work at IPL 4 or higher");
		return(0);
	}

	/* do it */
	time = timebase;
	while (timebase - time < 10)
		;
	send_cp_command(CR_SCC3, CPCOM_STOP_TX);
	mc68302_regs.scc3.scm &= ~(SCM_ENR | SCM_ENT);
	send_cp_command(CR_SCC3, CPCOM_ENTER_HUNT);
	mc68302_regs.scc3.scon = newscon;
	send_cp_command(CR_SCC3, CPCOM_RESTART_TX);
	mc68302_regs.scc3.scm |= SCM_ENR | SCM_ENT;
	time = timebase;
	while (timebase - time < 10)
		;
	return(0);
}

cmd_set_ipl(cmd, excstate)
	char *cmd;
	u_long *excstate;
{
	struct param params[1];
	u_long ipl;

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return(0);
	if (parse_hexnum(params[0].text, params[0].len, &ipl))
		return(0);
	if (ipl & 0xFFFFFFF8) {
		error("Value too large");
		return(0);
	}
	splx(ipl);
	return(0);
}

static const struct keyword lsled_settings[] = {
	{"OFF", 2, (long) PORTA_LSLED_OFF},
	{"ORANGE", 2, (long) PORTA_LSLED_ORANGE},
	{"RED", 1, (long) PORTA_LSLED_RED},
	{"GREEN", 1, (long) PORTA_LSLED_GREEN},
	{NULL, 0, -1}};

cmd_set_lsled(cmd)
	char *cmd;
{
	struct param params[1];
	int code, ambig = 0;

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return(0);
	code = find_keyword(params[0].text, params[0].len, lsled_settings,
				&ambig);
	if (code != -1)
		mc68302_regs.padat = mc68302_regs.padat & ~PORTA_LSLED_MASK |
					code;
	else {
		if (ambig)
			error("Ambiguous LED setting");
		else
			error("Invalid LED setting");
	}
	return(0);
}

cmd_set_mctl(cmd, mask)
	char *cmd;
	u_short mask;
{
	struct param params[1];

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return(0);
	if (params[0].len == 2 && !strncasecmp(params[0].text, "ON", 2))
		mc68302_regs.pbdat |= mask;
	else if (params[0].len == 3 && !strncasecmp(params[0].text, "OFF", 3))
		mc68302_regs.pbdat &= ~mask;
	else
		error("Invalid setting");
	return(0);
}

cmd_set_dsr(cmd)
	char *cmd;
{
	cmd_set_mctl(cmd, PORTB_DSR);
	return(0);
}

cmd_set_cd(cmd)
	char *cmd;
{
	cmd_set_mctl(cmd, PORTB_CD);
	return(0);
}

cmd_set_cts(cmd)
	char *cmd;
{
	cmd_set_mctl(cmd, PORTB_CTS);
	return(0);
}

static const struct keyword set_subcmds[] = {
	{"BCLK", 2, (long) &cmd_set_bclk},
	{"BITPUMP", 2, (long) &cmd_set_bitpump},
	{"CD", 2, (long) &cmd_set_cd},
	{"CONSOLE", 2, (long) &cmd_set_console},
	{"CTS", 2, (long) &cmd_set_cts},
	{"DSR", 1, (long) &cmd_set_dsr},
	{"IPL", 1, (long) &cmd_set_ipl},
	{"LSLED", 1, (long) &cmd_set_lsled},
	{NULL, 0, -1}};

cmd_set(cmd, excstate)
	char *cmd;
	u_long *excstate;
{
	char *cp = cmd, *subcmd;
	int count;
	void (*subcmdfunc)();
	int ambigsubcmd;

	while (*cp == ' ')
		cp++;
	if (!issym(*cp)) {
		error("Parameter required");
		return(0);
	}
	for (subcmd = cp, count = 0; issym(*cp); cp++)
		count++;
	ambigsubcmd = 0;
	subcmdfunc = (void (*)())
			find_keyword(subcmd, count, set_subcmds, &ambigsubcmd);
	if ((long) subcmdfunc != -1)
		(*subcmdfunc)(cp, excstate);
	else {
		if (ambigsubcmd)
			error("Ambiguous parameter");
		else
			error("Illegal parameter");
	}
	return(0);
}
