#include "types.h"
#include "stdio.h"
#include "ctype.h"
#include "strings.h"
#include "mon302.h"
#include "excstate.h"
#include "mc68302.h"

extern volatile struct mc68302_regs mc68302_regs;

extern u_long read_usp();

extern u_long last_memacc_addr;
extern u_short last_memacc_size;

cmd_call(cmd, excstate, retmode)
	char *cmd;
	u_long *excstate;
	int *retmode;
{
	struct param params[1];
	u_long addr;
	void (*fptr)();

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return(0);
	if (parse_hexnum(params[0].text, params[0].len, &addr))
		return(0);
	if (addr & 1) {
		error("Odd address");
		return(1);
	}
	fptr = (void (*)()) addr;
	(*fptr)();
	return(0);
}

cmd_continue(cmd, excstate, retmode)
	char *cmd;
	u_long *excstate;
	int *retmode;
{
	if (parse_cmd(cmd, 0, 0, NULL, NULL, NULL, NULL))
		return(0);
	*retmode = 0;
	return(1);
}

cmd_deposit(cmd, excstate)
	char *cmd;
	u_long *excstate;
{
	struct param params[256], *datum;
	struct datactl datactl;
	char *addrtext;
	int addrlen;
	u_long addr;
	char *qp = NULL;

	bzero(params, sizeof(params));
	bzero(&datactl, sizeof(datactl));
	if (parse_cmd(cmd, 2, 256, params, &datactl, NULL, NULL))
		return(0);
	addrtext = params[0].text;
	addrlen = params[0].len;
	if (parse_addr(addrtext, addrlen, &datactl, &addr))
		return(0);
	if (params[2].text && datactl.count) {
		error("Qualifier conflict");
		return(0);
	}
	for (datum = &params[1]; datum->text; datum++) {
		u_long num;
		int stat;

		if (check_intr())
			return(0);
		switch (datactl.datasize) {
		case DATASIZE_BYTE:
			if (qp) {
				if (*qp == '\"') {
					qp = NULL;
					continue;
				}
				stat = deposit_byte(datactl.addrspc, addr, *qp);
				qp++;
				datum--;
				break;
			}
			if (datum->text[0] == '\"') {
				if (datactl.count) {
					error("Qualifier conflict");
					return;
				}
				qp = datum->text + 1;
				datum--;
				continue;
			} else if (parse_hexnum(datum->text, datum->len, &num))
				return(0);
			if (num & 0xFFFFFF00) {
				error("Value too large");
				return(0);
			}
			stat = deposit_byte(datactl.addrspc, addr, num);
			break;
		case DATASIZE_WORD:
			if (parse_hexnum(datum->text, datum->len, &num))
				return(0);
			if (num & 0xFFFF0000) {
				error("Value too large");
				return(0);
			}
			stat = deposit_word(datactl.addrspc, addr, num);
			break;
		case DATASIZE_LONG:
			if (parse_hexnum(datum->text, datum->len, &num))
				return(0);
			stat = deposit_long(datactl.addrspc, addr, num,
					excstate);
			break;
		}
		if (stat)
			break;
		if (datactl.addrspc == ADDRSPC_MEM) {
			last_memacc_addr = addr;
			last_memacc_size = datactl.datasize;
		}
		addr += datactl.step;
		if (datactl.count) {
			datum--;
			datactl.count--;
		}
	}
	return(0);
}

deposit_byte(addrspc, addr, val)
	u_long addr;
	u_char val;
{
	*(u_char *)addr = val;
	return(0);
}

deposit_word(addrspc, addr, val)
	u_long addr;
	u_short val;
{
	if (addr & 1) {
		error("Odd address");
		return(1);
	}
	*(u_short *)addr = val;
	return(0);
}

deposit_long(addrspc, addr, val, excstate)
	u_long addr;
	u_long val;
	u_long *excstate;
{
	switch (addrspc) {
	case ADDRSPC_MEM:
		if (addr & 1) {
			error("Odd address");
			return(1);
		}
		*(u_long *)addr = val;
		return(0);
	case ADDRSPC_XSTATE:
		if (addr > EXCSTATE_PC) {
			error("Illegal address");
			return(1);
		}
		excstate[addr] = val;
		return(0);
	case ADDRSPC_USP:
		write_usp(val);
		return(0);
	}
	return(1);
}

cmd_examine(cmd, excstate)
	char *cmd;
	u_long *excstate;
{
	struct param params[1];
	struct datactl datactl;
	char *addrtext;
	int addrlen;
	u_long addr;

	bzero(params, sizeof(params));
	bzero(&datactl, sizeof(datactl));
	if (parse_cmd(cmd, 0, 1, params, &datactl, NULL, NULL))
		return(0);
	addrtext = params[0].text;
	addrlen = params[0].len;
	/* If no address is specified, + is assumed. */
	if (addrtext == NULL) {
		addrtext = "+";
		addrlen = 1;
	}
	if (parse_addr(addrtext, addrlen, &datactl, &addr))
		return(0);
	do {
		int stat;
		u_char bval;
		u_short wval;
		u_long lval;

		if (check_intr())
			return(0);
		switch (datactl.datasize) {
		case DATASIZE_BYTE:
			stat = examine_byte(datactl.addrspc, addr, &bval);
			break;
		case DATASIZE_WORD:
			stat = examine_word(datactl.addrspc, addr, &wval);
			break;
		case DATASIZE_LONG:
			stat = examine_long(datactl.addrspc, addr, &lval,
					excstate);
			break;
		}
		if (stat)
			break;
		switch (datactl.datasize) {
		case DATASIZE_BYTE:
			printf(" %c %08X %02X\r\n", datactl.addrspc, addr,
				(u_long) bval);
			break;
		case DATASIZE_WORD:
			printf(" %c %08X %04X\r\n", datactl.addrspc, addr,
				(u_long) wval);
			break;
		case DATASIZE_LONG:
			printf(" %c %08X %08X\r\n", datactl.addrspc, addr,
				lval);
			break;
		}
		if (datactl.addrspc == ADDRSPC_MEM) {
			last_memacc_addr = addr;
			last_memacc_size = datactl.datasize;
		}
		addr += datactl.step;
	} while (datactl.count--);
	return(0);
}

examine_byte(addrspc, addr, valptr)
	u_long addr;
	u_char *valptr;
{
	*valptr = *(u_char *)addr;
	return(0);
}

examine_word(addrspc, addr, valptr)
	u_long addr;
	u_short *valptr;
{
	if (addr & 1) {
		error("Odd address");
		return(1);
	}
	*valptr = *(u_short *)addr;
	return(0);
}

examine_long(addrspc, addr, valptr, excstate)
	u_long addr;
	u_long *valptr;
	u_long *excstate;
{
	u_long val;

	switch (addrspc) {
	case ADDRSPC_MEM:
		if (addr & 1) {
			error("Odd address");
			return(1);
		}
		val = *(u_long *)addr;
		break;
	case ADDRSPC_XSTATE:
		if (addr > EXCSTATE_PC) {
			error("Illegal address");
			return(1);
		}
		val = excstate[addr];
		break;
	case ADDRSPC_USP:
		val = read_usp();
		break;
	}
	*valptr = val;
	return(0);
}

const struct keyword hexdump_quals[] =
	{{"ASCII", 1, 1},
	 {NULL, 0, -1}};

cmd_hexdump(cmd)
	char *cmd;
{
	struct param params[2];
	struct qual quals[1];
	u_long start, count;
	u_char *cp, *end;
	u_char buf[16];
	register int cc, i, c;

	quals[0].id = 0;
	if (parse_cmd(cmd, 2, 2, params, NULL, hexdump_quals, quals))
		return(0);
	if (quals[0].id && quals[0].value) {
		error("No value qualifier");
		return(0);
	}
	if (parse_hexnum(params[0].text, params[0].len, &start))
		return(0);
	if (parse_hexnum(params[1].text, params[1].len, &count))
		return(0);
	end = (u_char *) start + count;
	start &= 0xFFFFFFF0;
	cp = (u_char *) start;

	for (; cp < end; cp += cc) {
		cc = end - cp;
		if (cc > 16)
			cc = 16;
		if (check_intr())
			return(0);
		bcopy(cp, buf, cc);
		printf("%08X:", cp);
		for (i = 0; i < 16; i++) {
			if ((i & 3) == 0)
				putchar(' ');
			if (i < cc)
				printf(" %02X", buf[i]);
			else
				printf("   ");
		}
		if (quals[0].id) {
			printf("  ");
			for (i = 0; i < cc; i++) {
				c = buf[i];
				if (!isprint(c))
					c = '.';
				putchar(c);
			}
		}
		printf("\r\n");
	}
	return(0);
}

cmd_reinit(cmd, excstate, retmode)
	char *cmd;
	u_long *excstate;
	int *retmode;
{
	if (parse_cmd(cmd, 0, 0, NULL, NULL, NULL, NULL))
		return(0);
	spl7();
	asm("reset");
	sysinit();
	return(0);
}

cmd_restack(cmd, excstate, retmode)
	char *cmd;
	u_long *excstate;
	int *retmode;
{
	if (parse_cmd(cmd, 0, 0, NULL, NULL, NULL, NULL))
		return(0);
	*retmode = 1;
	return(1);
}

cmd_start(cmd, excstate, retmode)
	char *cmd;
	u_long *excstate;
	int *retmode;
{
	struct param params[1];

	bzero(params, sizeof(params));
	if (parse_cmd(cmd, 0, 1, params, NULL, NULL, NULL))
		return(0);
	if (params[0].text) {
		u_long addr;

		if (parse_hexnum(params[0].text, params[0].len, &addr))
			return(0);
		if (addr & 1) {
			error("Odd address");
			return(1);
		}
		excstate[EXCSTATE_PC] = addr;
	}
	*retmode = 0;
	return(1);
}
