#include "../libc/types.h"
#include "typedefs.h"
#include "config.h"
#include "bitpump.h"
#include "util.h"
#include "timermac.h"

/*---------------------------------------------------------*/
/*  Local Defines                                          */
/*---------------------------------------------------------*/

#define AVERAGE_TH             44
#define ABS_AVERAGE_TH         44
#define PHASE_LOCK_TRY_NO      4
#define LOG2_TRY_NO            2

#define MAX_PLL_VALUE                 (BP_S_16BIT)32760

/***********************************************************/
/*    _IsPhaseLocked                                       */
/*    Check if VCXO phase is locked to incoming signal     */
/*                                                         */
/*    returns: true/false                                  */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           status = _IsPhaseLocked(no);                  */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                29-Oct-1993              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
_IsPhaseLocked()
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;
    BP_S_8BIT temp;
    BP_S_16BIT value, average, abs_average;

    /* sometimes the VCXO goes to the limits. this indicates error */
    READ_METER_REG(vcxo_frequency_low, vcxo_frequency_high, value, 0);

#ifdef TDEBUG
    printf("VCXO = %d\r\n", (int)value);
#endif

    if ( abs(value) > MAX_PLL_VALUE )
        return(0);          /* Phase is not locked */

    average = abs_average = 0;

    for ( temp = 0; temp < PHASE_LOCK_TRY_NO; temp++)
        {

        WAIT_FOR_TIMER(TIMERMASK_METER);

        /*----------------------------------------------------*/
        /* Read phase detector meter value                    */
        /*----------------------------------------------------*/
        READ_METER_REG(pdm_low, pdm_high, value, 0);

        /*----------------------------------------------------*/
        /* Accumulate and normalize PDM and |PDM| values       */
        /*----------------------------------------------------*/
    
        average += value;
        abs_average += abs(value);

#ifdef TDEBUG
        printf("PDM(%d) = %d ; ", (int)temp, (int)value);
#endif

        } /* END-FOR */

#ifdef TDEBUG
        printf("\r\n");
#endif

    /*----------------------------------------------------*/
    /* Normalize accumulated results                       */
    /*----------------------------------------------------*/

    average >>= LOG2_TRY_NO; /* Divide by log 2 of try no */
    abs_average >>= LOG2_TRY_NO; /* Divide by log 2 of try no */
    temp = BP_READ_BIT(bp_mode_ptr, pll_modes, phase_detector_gain); /* PD Gain Value */
    switch (temp) /* Divide by phase detector gain */
        {
        case NORMAL_GAIN: /* Divide by 1/2 */
            average <<= 1;
            abs_average <<= 1;
            break;

        case HIGH_GAIN: /* Divide by 1 */
            break;

        case HIGHER_GAIN: /* Divide by 2 */
            average >>= 1;
            abs_average >>= 1;
            break;

        case HIGHEST_GAIN: /* Divide by 4 */
            average >>= 2;
            abs_average >>= 2;
            break;

        } /* END-SWITCH divide by phase detector gain */

#ifdef TDEBUG
    printf("Avg PDM = %d ; Avg |PDM| = %d\r\n", (int)average, (int)abs_average);
#endif

    /*-----------------------------------------------------------------*/
    /* Phase is locked if PDM and |PDM| are below specified thresholds */
    /*-----------------------------------------------------------------*/
    if ((average <= AVERAGE_TH) && (abs_average <= ABS_AVERAGE_TH)) /* Phase is locked */
	return(1);
    else /* Phase is not locked */
	return(0);

} /* END _IsPhaseLocked() */
