#include "typedefs.h"
#include "util.h"

/***********************************************************/
/*    _LookUpTable()                                       */
/*    Look for a known value in a table. There are two     */
/*    types of tables: NLM table and FELM table. The       */
/*    returned value is the corresponding index.           */
/*                                                         */
/*    returns: BP_U_8BIT table index                       */
/*                                                         */
/*    Input variables: BP_TABLE *table_ptr                 */
/*                     BP_S_16BIT value                    */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           noise_margin = _LookUpTable(_nlm_table,       */
/*                                  nlm_meter);            */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                29-Nov-1993              */
/*                                                         */
/* revision history:                                       */
/*   Function is completely changed to increase the look-  */
/*   up-speed. The original linear search algorithm is     */
/*   replaced by a logarithmic algorithm. This way the     */
/*   result is found after 1 up to maximal 7 loops.        */
/*   Measuring results have shown that "LookUpTable()" is  */
/*   now ten times faster. For a 11.0592 MHz 8032 this is: */
/*     more than 6.6 ms before -> less than 750 s now     */
/*                                                         */
/*   Note that using an logarithmic algorithm assumes a    */
/*   sorted list to work properly!                         */
/*                                                         */
/*   This function does a comparison on the high byte then */
/*   checks the low byte.                                  */
/*                                                         */
/*   For non-C51, use a simple 16-bit comparison.          */
/*                                                         */
/***********************************************************/

int
_LookUpTable (BP_TABLE *table_ptr, BP_TABLE value)
{
    int lwr_index, upr_index, index;

    lwr_index = 0;
    upr_index = TABLE_LENGTH - 1;
    do
        {
        index = (upr_index + lwr_index) / 2;

        if ( table_ptr[index] == value )
            {
            return index;
            }
        else if ( table_ptr[index] < value )
            {
            upr_index = index;
            }
        else 
            {
            lwr_index = index;
            }

        } while( (upr_index - lwr_index) > 1 );

    if ( lwr_index == 0 )
        {
        return 0;
        }

    return (upr_index);

} /* END _LookUpTable */
