/*
 * The function that does the "Init regs" line lives here
 */

#include "../libc/types.h"
#include "typedefs.h"
#include "config.h"
#include "bitpump.h"
#include "util.h"
#include "defmeter.h"
#include "bpstate.h"

#define NL_EC_DELAY_INIT_VALUE        3
#define PHASE_OFFSET_INIT_VALUE       0         /* No phase offset */
#define VCXO_FREQ_INIT_VALUE          0         /* Nominal frequency */
#define CURSOR_LEVEL_INIT_VALUE       1024
#define NOISE_HISTOGRAM_TH_INIT_VALUE 512
#define SCR_SYNC_TH_INIT_VALUE        0
#define SNR_ALARM_INIT_VALUE          2       /* 580 per sample = 10dB SNR */
#define EP_PAUSE_INIT_VALUE           0xffff    /* No EP Pause */


/***********************************************************/
/*    _BtInitialize()                                      */
/*    Initialize Bitpump to startup initialization state.  */
/*    User defined parameters are NOT lost.                */
/*                                                         */
/*    returns: void                                        */
/*                                                         */
/*    Input variables: BP_U_8BIT no                        */
/*                                                         */
/*    Output variables: None                               */
/*                                                         */
/*    example:                                             */
/*           _BtInitialize(no);                            */
/*                                                         */
/* Programmer:                                             */
/*     Iris Shuker                24-Oct-1993              */
/*                                                         */
/* revision history:                                       */
/*                                                         */
/***********************************************************/
void _BtInitialize (BP_U_8BIT no)
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;

#ifdef TDEBUG
    PREFIX;
    printf("Init regs\r\n");
#endif

    /*----------------------------------------------------------------*/
    /* Initialize Bitpump registers for startup state.                */
    /*----------------------------------------------------------------*/

    /*----------------------------------------------------------------*/
    /* A/D sampling phase + Impulse Shortening (Bt8973)               */
    /*----------------------------------------------------------------*/
    BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, rphs, 0);

    /*----------------------------------------------------------------*/
    /* Linear EC                                                      */
    /*----------------------------------------------------------------*/
    BP_WRITE_REG(bp_ptr, linear_ec_modes, RESET); /* Reset linear EC modes register */
    SET_RESET(bp_mode_ptr, linear_ec_modes, zero_coefficients); /* Clear linear EC coefficients */
    BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_gain, HIGHEST_GAIN); /* EC at highest adaptation gain */
    BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, enable_dc_tap, ON); /* Enale adaptive DC cancellation */

    /*----------------------------------------------------------------*/
    /* NL EC                                                          */
    /*----------------------------------------------------------------*/
    BP_WRITE_REG(bp_ptr, nonlinear_ec_modes, RESET); /* Reset NL EC modes register */
    SET_RESET(bp_mode_ptr, nonlinear_ec_modes, zero_coefficients); /* Clear NL EC coefficients */
    BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_coefficients, OFF);
    BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, zero_output, ON); /* Zero NL EC output */
    BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_gain, HIGH_GAIN); /* Non linear EC at high adaptation gain */
    BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, symbol_delay, NL_EC_DELAY_INIT_VALUE); /* Set NL EC symbol delay */

    /*----------------------------------------------------------------*/
    /* DFE                                                            */
    /*----------------------------------------------------------------*/
    BP_WRITE_REG(bp_ptr, dfe_modes, RESET); /* Reset DFE modes register */
    SET_RESET(bp_mode_ptr, dfe_modes, zero_coefficients); /* Clear DFE coefficients */
    BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_gain, HIGH_GAIN); /* DFE at high adaptation gain */

    /*----------------------------------------------------------------*/
    /* Transmitter                                                    */
    /*----------------------------------------------------------------*/
    BP_WRITE_REG(bp_ptr, transmitter_modes, RESET); /* Reset transmitter modes register */
#if SUPPORT_BT8960 || SUPPORT_BT8970
    if ( GET_BITPUMP_TYPE() <= BT8970 )
        {
        /* Fixes potential Bt8960/70 scrambler dead space issue */
        BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, 0x04);
        }
#endif
    BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, transmitter_off, ON); /* Set transmitter OFF */
    BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_TWO_LEVEL_ONES); /* Internal 2-level scrambled 1's */

    /*----------------------------------------------------------------*/
    /* PLL & Timing                                                   */
    /*----------------------------------------------------------------*/
    BP_WRITE_BIT(bp_mode_ptr, pll_modes, freeze_pll, ON);
    BP_WRITE_BIT(bp_mode_ptr, pll_modes, negate_symbols, OFF); /* Don't Invert PD symbols */
    BP_WRITE_BIT(bp_mode_ptr, pll_modes, pll_gain, HIGHEST_GAIN); /* PLL at highest adaptation gain */
    BP_WRITE_BIT(bp_mode_ptr, pll_modes, phase_detector_gain, HIGH_GAIN); /* Phase detector at high gain */
    SET_WORD(bp_ptr, pll_phase_offset_low, pll_phase_offset_high, PHASE_OFFSET_INIT_VALUE); /* Init Pll offset register */
    SET_WORD(bp_ptr, vcxo_frequency_low, vcxo_frequency_high, VCXO_FREQ_INIT_VALUE); /* Init fixed frequency value */

    /*----------------------------------------------------------------*/
    /* Detector                                                       */
    /*----------------------------------------------------------------*/
    BP_WRITE_REG(bp_ptr, detector_modes, RESET); /* Reset detector mode register */
    SET_WORD(bp_ptr, cursor_level_low, cursor_level_high, CURSOR_LEVEL_INIT_VALUE); /* Set slicer cursor level init value */
    SET_WORD(bp_ptr, noise_histogram_th_low, noise_histogram_th_high, NOISE_HISTOGRAM_TH_INIT_VALUE);
    SET_WORD(bp_ptr, ep_pause_th_low, ep_pause_th_high, EP_PAUSE_INIT_VALUE);
    BP_WRITE_REG(bp_ptr, scr_sync_th, SCR_SYNC_TH_INIT_VALUE); /* Init scrambler syncronization threshold */
    SET_WORD(bp_ptr, snr_alarm_th_low, snr_alarm_th_high, SNR_ALARM_INIT_VALUE); /* Set snr alarm threshold init value */

    /*----------------------------------------------------------------*/
    /* AGC & DC Offset                                                */
    /*----------------------------------------------------------------*/
    BP_WRITE_BIT(bp_mode_ptr, adc_control, lb, 0);  /* loopback control */

    BP_WRITE_BIT(bp_mode_ptr, adc_control, again, AGAIN9DB);
    SET_WORD(bp_ptr, dc_offset_low, dc_offset_high, 0);

    /*----------------------------------------------------------------*/
    /* DAGC                                                           */
    /*----------------------------------------------------------------*/
    SET_WORD(bp_ptr, dagc_target_low, dagc_target_high, DAGC_TARGET_INIT_VALUE);
    BP_WRITE_REG(bp_ptr, dagc_modes, RESET); /* Reset DAGC mode register */
    BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_gain, HIGH_GAIN); /* DAGC at high adaptation gain */

    /*----------------------------------------------------------------*/
    /* FFE                                                            */
    /*----------------------------------------------------------------*/
    BP_WRITE_REG(bp_ptr, ffe_modes, RESET); /* Reset FFE mode register */

    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, ON); /* Adapt FFE coefficients */
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, zero_coefficients, ON); /* Clear FFE coefficients */
    DELAY2SYMBOL;
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, OFF); /* Freeze FFE coefficients */
    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, zero_coefficients, OFF);

    BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_gain, HIGH_GAIN); /* FFE at high adaptation gain */

    /*----------------------------------------------------------------*/
    /* EP                                                             */
    /*----------------------------------------------------------------*/
    BP_WRITE_REG(bp_ptr, ep_modes, RESET); /* Reset EP mode register */
    BP_WRITE_BIT(bp_mode_ptr, ep_modes, adapt_coefficients, ON); /* Adapt EP coefficients */
    BP_WRITE_BIT(bp_mode_ptr, ep_modes, zero_coefficients, ON); /* Clear EP coefficients */
    DELAY2SYMBOL;
    BP_WRITE_BIT(bp_mode_ptr, ep_modes, adapt_coefficients, OFF); /* Freeze EP coefficients */
    BP_WRITE_BIT(bp_mode_ptr, ep_modes, zero_coefficients, OFF);
    BP_WRITE_BIT(bp_mode_ptr, ep_modes, adapt_gain, HIGH_GAIN); /* EP at high adaptation gain */

    /*----------------------------------------------------------------*/
    /* Interrupts                                                     */
    /*----------------------------------------------------------------*/
    BP_WRITE_REG(bp_ptr, mask_low_reg, MASK_ALL); /* Mask all interrupts */
    BP_WRITE_REG(bp_ptr, mask_high_reg, MASK_ALL); /* Mask all interrupts */

    BP_WRITE_REG(bp_ptr, timer_enable, RESET); /* Disable all timers */
    BP_WRITE_REG(bp_ptr, timer_continous, RESET); /* Reset timer continous mode register */

    BP_WRITE_REG(bp_ptr, timer_source, RESET); /* Reset all interrupt sources */
    BP_WRITE_REG(bp_ptr, irq_source, RESET); /* Reset all interrupt sources */

    BP_global_state.latched_timer_irq = 0;
    BP_global_state.latched_alarm_irq = 0;

    BP_WRITE_REG(bp_ptr, timer_enable, 0xFF);  /* Enable all timers */
    BP_WRITE_BIT(bp_mode_ptr, timer_continous, meter, ON); /* METER timer at continous mode */

} /* END _BtInitialize() */
