/*
 * See the header comment in defmeter.h
 */

#include "../libc/types.h"
#include "typedefs.h"
#include "config.h"
#include "bitpump.h"
#include "util.h"
#include "defmeter.h"
#include "bpstate.h"

/************************************************************/
/*    _SetMeterTimer                                        */
/*    Set Meter Timer value                                 */
/*                                                          */
/*    returns: void                                         */
/*                                                          */
/*    Input variables: BP_U_8BIT no, BP_U_8BIT value        */
/*                                                          */
/*    Output variables: None                                */
/*                                                          */
/*    example:                                              */
/*           _SetMeterTimer(no, value);                     */
/*                                                          */
/* Programmer:                                              */
/*     Iris Shuker                29-Oct-1993               */
/*                                                          */
/* revision history:                                        */
/*     No longer wait for the meter intervals to elapse     */
/*                                                          */
/************************************************************/

void _SetMeterTimer (BP_U_8BIT value, BP_U_8BIT highfelm, BP_U_8BIT lowfelm)
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;
    int s;

    BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, low_felm, ON); /* Mask low FELM int. */
    BP_WRITE_BIT(bp_mode_ptr, mask_high_reg, high_felm, ON); /* Mask high FELM int. */

    SET_WORD(bp_ptr, meter_low, meter_high, (DEFAULT_METER_VALUE << value));

    SET_WORD(bp_ptr, far_end_high_alarm_th_low, far_end_high_alarm_th_high,
		(highfelm << value) );
    SET_WORD(bp_ptr, far_end_low_alarm_th_low, far_end_low_alarm_th_high,
		(lowfelm << value) );

    s = splup(1);
    bp_ptr->timer_source = ~TIMERMASK_METER;
    BP_global_state.latched_timer_irq &= ~TIMERMASK_METER;
    BP_WRITE_BIT(bp_mode_ptr, timer_restart, meter, ON);
    splx(s);

} /* END _SetMeterTimer() */

/*
 * This meter normalization stuff needs some explanation
 * so that others don't have to rediscover it like I did.
 *
 * Most meters in the bitpump take some 16-bit signal value in the DSP path,
 * sum it up over the meter timer interval in a 32-bit accumulator,
 * and put the *high* 16 bits of that accumulator in the readable registers
 * at the end of each interval.  Thus if the meter interval were 65536 symbols,
 * the value read out of the registers would correspond to the average
 * of the original instantaneous signal value being sampled.
 *
 * Well, a meter interval of 65536 symbols is not possible, but
 * the ZipWire code uses various meter intervals which are all powers of 2.
 * If one reads some meter registers and left-shifts the value read by
 * the _NormlizeMeter() return value, one will get the meter value that
 * would have been accumulated over 65536 symbols, which is the same as
 * the average value of the actual signal being measured.
 */
_NormlizeMeter()
{
    BP_U_8BIT x;
    DECLARE_PTR;

    x = BP_READ_REG(bp_ptr, meter_high); 
    switch(x) 
    { 
        case 0x1: 
            return 8; 
        case 0x2: 
            return 7; 
        case 0x4: 
            return 6; 
        case 0x8: 
            return 5; 
        case 0x10: 
            return 4; 
        case 0x20: 
            return 3; 
        case 0x40: 
            return 2; 
        case 0x80: 
            return 1; 
    } /* END-SWITCH */
    
    return (0);	/* XXX */

} /* END NormlizeMeter() */
