/*
 * The stage-driven analog test code (ALB, ERLE and AAGC) lives here.
 * Based on testmode.c from ZipWire.
 */

#include "../libc/types.h"
#include "typedefs.h"
#include "config.h"
#include "bitpump.h"
#include "util.h"
#include "defmeter.h"
#include "bpstate.h"
#include "timermac.h"
#include "analogtests.h"

/*---------------------------------------------------------*/
/*  Local Defines                                          */
/*---------------------------------------------------------*/

#define ALB_EC_TAP      3

#define TM_ALB_INTERVAL     (BP_U_16BIT)6000
#define TM_ERLE_INTERVAL    (BP_U_16BIT)6000  /* 10.5s at 1168k */
#define TM_AAGC_INTERVAL    (BP_U_16BIT)6000

void
analogtests_main(state)
	struct analogtests_state *state;
{
    DECLARE_PTR;
    DECLARE_MODE_PTR;
    BP_S_8BIT temp;
    BP_S_16BIT value;
    BP_U_16BIT slm, felm;
    u_char data[4];

    switch (state->stage)
        {
        case TEST_MODE_IDLE:
            return;

        /*------------------------------------------------------------*/
        /* Wait Meter Intervals                                       */
        /* The next stage after waiting the meter intervals is set    */
        /* by the upper byte of the STAGE2 EQ Reg File.               */
        /*------------------------------------------------------------*/

        case TM_WAIT_METER_INTERVAL1:
	    RESTART_METER;
            state->stage = TM_WAIT_METER_INTERVAL2;
            break;

    
        case TM_WAIT_METER_INTERVAL2:
            TIMER_BREAK(TIMERMASK_METER);
            /* Set Stage to the next stage after wait meter intervals */
	    state->stage = state->stage2;
            break;


        /*------------------------------------------------------------*/
        /* Analog Loop Back Testmodes                                 */
        /*------------------------------------------------------------*/
        case TM_ALB_INITIALIZE:
            _BtInitialize(0);
	    _SetMeterTimer(ALT_METER, HIGH_FELM_ALARM_TH, LOW_FELM_ALARM_TH);
            BP_WRITE_BIT(bp_mode_ptr, adc_control, lb, state->albtype);
	    bp_set_lfsr(0, _HTUC, 1);
	    printf("Starting analog loopback\r\n");

            /* Set Activation Interval */
            SET_WORD(bp_ptr, sut4_low, sut4_high, TM_ALB_INTERVAL);
            RESTART_SUT4;

            state->stage = TM_WAIT_METER_INTERVAL1;
            state->stage2 = TM_ALB_ACTIVATE;
	    state->status = TM_STATUS_INPROGRESS;
            return;


        case TM_ALB_ACTIVATE:
            TIMER_BREAK(TIMERMASK_METER);

            /*------------------------------------------------------------*/
            /* Trasmit 2-Level Scrambled Ones                             */
            /*------------------------------------------------------------*/
            BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_TWO_LEVEL_ONES);
            BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, transmitter_off, OFF); /* Transmit! */

            BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, rphs, 0); /* Init receive sampling phase */

	    BP_WRITE_BIT(bp_mode_ptr, adc_control, again, AGAIN0DB);

            state->stage = TM_ALB_ADAPT_EC;
            break;

        case TM_ALB_ADAPT_EC:

            /*------------------------------------------------------------*/
            /* Adapt EC.                                                  */
            /*------------------------------------------------------------*/
            BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_coefficients, ON);

            SET_WORD(bp_ptr, t3_low, t3_high, 30000);
	    RESTART_T3;
            state->stage = TM_ALB_ADAPT_DAGC;
            break;


        case TM_ALB_ADAPT_DAGC:
            TIMER_BREAK(TIMERMASK_T3);

            BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_coefficients, OFF); /* Freeze EC coefficient */

	    bp_lec_read(ALB_EC_TAP, data);
            value = BYTE2WORD(data[3], data[2]);
            value -= 2500;
	    data[0] = 0;
	    data[1] = 0;
	    data[2] = LOW(value);
	    data[3] = HIGH(value);
	    bp_lec_write(ALB_EC_TAP, data);

            /*------------------------------------------------------------*/
            /* Adapt DAGC.                                                */
            /*------------------------------------------------------------*/
            SET_WORD(bp_ptr, dagc_target_low, dagc_target_high, DAGC_TARGET_INIT_VALUE); /* Set DAGC target init value */
            _InitDagc();

            _AdaptDagcSelfMode();
            SET_WORD(bp_ptr, t3_low, t3_high, 20000);
	    RESTART_T3;

            state->stage = TM_ALB_OPEN_EYE1;
            break;

        case TM_ALB_OPEN_EYE1:
            TIMER_BREAK(TIMERMASK_T3);

            BP_WRITE_BIT(bp_mode_ptr, dagc_modes, adapt_coefficient, OFF); /* Freeze DAGC coefficient */

#ifdef TDEBUG
            printf("After Self Mode - ");
            _PrintfDAGC();
#endif

            /*------------------------------------------------------------*/
            /* Open eye process.                                          */
            /*------------------------------------------------------------*/
    
            _InitDfe();

            _OpenEyeLastFfe(); /* PKD, adapt last FFE */

            SET_WORD(bp_ptr, t3_low, t3_high, 4501);
	    RESTART_T3;
            state->stage = TM_ALB_OPEN_EYE2;
            break;

        case TM_ALB_OPEN_EYE2:
            TIMER_BREAK(TIMERMASK_T3);

            _AdaptDfe();

            SET_WORD(bp_ptr, t3_low, t3_high, 6700);
	    RESTART_T3;

            state->stage = TM_ALB_OPEN_EYE3;
            break;

        case TM_ALB_OPEN_EYE3:
            TIMER_BREAK(TIMERMASK_T3);

            BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_gain, NORMAL_GAIN);
            BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_gain, NORMAL_GAIN);
            BP_WRITE_BIT(bp_mode_ptr, detector_modes, enable_peak_detector, OFF); /* Disable peak detector */ 

            SET_WORD(bp_ptr, t3_low, t3_high, 55000);
	    RESTART_T3;

            state->stage = TM_ALB_OPEN_EYE4;
            break;

        case TM_ALB_OPEN_EYE4:
            TIMER_BREAK(TIMERMASK_T3);
            
            _FreezeDfeFfe();

            RESTART_METER; 

            state->stage = TM_WAIT_METER_INTERVAL1;
	    state->stage2 = TM_ALB_OPEN_EYE5;
            break;

        case TM_ALB_OPEN_EYE5:
            TIMER_BREAK(TIMERMASK_METER);

            if (!_IsEyeOpen()) /* Open eye failed, continue to next phase */
                {
                temp = BP_READ_BIT(bp_mode_ptr, receive_phase_select, rphs);
                temp++; /* move to next phase */
                if (temp > 15)
                    {
                    /* If pass through all phases, set to IDLE */

		    printf("Opening of the eye failed at all phases!\r\n");
		    state->stage = TEST_MODE_IDLE;
		    state->status = TM_STATUS_FAILED;
		    return;
                    }
#ifdef TDEBUG
                PREFIX;
                printf("Failed Eye Opening, trying next phase = %d.\r\n",
			(int)temp);
#endif

                BP_WRITE_BIT(bp_mode_ptr, receive_phase_select, rphs, temp); /* Init receive sampling phase */
                state->stage = TM_ALB_ADAPT_EC;
                break;
                } /* END-ELSE continue to the next phase */

#ifdef TDEBUG
            PREFIX;
            printf("Eye Opened\r\n");
#endif
            /*------------------------------------------------------------*/
            /* Adapt equalizer.                                           */
            /*------------------------------------------------------------*/
            BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_gain, NORMAL_GAIN); /* DFE at high adaptation gain */
            BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_gain, NORMAL_GAIN); /* FFE at high adaptation gain */
            BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_last_coeff, OFF); /* FFE adapt all mode */
            BP_WRITE_BIT(bp_mode_ptr, dfe_modes, adapt_coefficients, ON); /* Adapt DFE */
            BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, ON); /* Adapt FFE */

            SET_WORD(bp_ptr, t3_low, t3_high, 60000);
	    RESTART_T3;

            state->stage = TM_ALB_TX_4LEVEL;
            break;

        case TM_ALB_TX_4LEVEL:
            TIMER_BREAK(TIMERMASK_T3);

            BP_WRITE_BIT(bp_mode_ptr, ffe_modes, adapt_coefficients, OFF); /* Freeze FFE */

#ifdef TDEBUG
            _PrintfDAGC();
#endif

            /*------------------------------------------------------------*/
            /* Move to 4 level scrambled 1s.                              */
            /*------------------------------------------------------------*/
            BP_WRITE_BIT(bp_mode_ptr, detector_modes, two_level, OFF); /* 4-level Slicing */

            BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_FOUR_LEVEL_ONES); /* Internal 4 level scrambled 1's */

#ifdef TDEBUG
            PREFIX;
            printf("Loopback Complete...\r\n");
#endif

	    /* for anyone wanting to run the BER meter */
	    _SetMeterTimer(NORMAL_METER, HIGH_FELM_ALARM_TH, LOW_FELM_ALARM_TH);
            state->stage = TEST_MODE_IDLE;
            state->status = TM_STATUS_SUCCESS;
            return;


        /*------------------------------------------------------------*/
        /* ERLE Test Mode                                             */
        /*------------------------------------------------------------*/
        case TM_ERLE_INIT:
            _BtInitialize(0);

            /* Set Activation Interval */
            SET_WORD(bp_ptr, sut4_low, sut4_high, TM_ERLE_INTERVAL);
            RESTART_SUT4;


            /*---------------------------------------*/
            /* Configure Transmitter                 */
            /*---------------------------------------*/
            if ( state->erlemode.bits.transmit_level )
                {
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_TWO_LEVEL_ONES);
                }
            else
                {
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_FOUR_LEVEL_ONES);
                }

            if ( state->erlemode.bits.transmit_state )
                {
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, transmitter_off, OFF); /* Transmit! */
                }
            else
                {
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, transmitter_off, ON); /* Transmit OFF */
                }

            /*---------------------------------------*/
            /* Configure Analog Gain Control         */
            /*---------------------------------------*/
	    BP_WRITE_BIT(bp_mode_ptr, adc_control, again,
			 state->erlemode.bits.again);
            SET_WORD(bp_ptr, dc_offset_low, dc_offset_high , 0x00); /* Set DC offset register */

#ifdef TDEBUG
            PREFIX;
            printf("AAGC %d dB\r\n", (int) (state->erlemode.bits.again * 3) );
#endif

            /*---------------------------------------*/
            /* Configure Meter Interval              */
            /*---------------------------------------*/

            SET_WORD(bp_ptr, meter_low, meter_high, 0xFFFF);

            BP_WRITE_BIT(bp_mode_ptr, timer_continous, meter, ON); /* Meter timer continous mode */

            RESTART_METER;


            state->stage = TM_WAIT_METER_INTERVAL1;
            state->stage2 = TM_ERLE_DC_CANCEL;
	    state->status = TM_STATUS_INPROGRESS;
            return;

        case TM_ERLE_DC_CANCEL:
            TIMER_BREAK(TIMERMASK_METER);

            /*----------------------------------------------------*/
            /* Read DC meter value                                */
            /*----------------------------------------------------*/
            READ_METER_REG(dc_meter_low, dc_meter_high, value, 0);

            /*----------------------------------------------------------*/
            /* Set DC offset register to compensate for measured offset */
            /*----------------------------------------------------------*/
            SET_WORD(bp_ptr, dc_offset_low, dc_offset_high , value); /* Set DC offset register */

#ifdef TDEBUG
            PREFIX;
            printf("DC Offset = %d\r\n", (int)value);
#endif

            state->stage = TM_ERLE_ADAPT_EC1;
            break;

        case TM_ERLE_ADAPT_EC1:
#ifdef TDEBUG
            PREFIX;
            printf("EC Highest, ");
#endif

            BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_coefficients, ON); /* Adapt EC */
	    BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_gain, HIGHEST_GAIN); /* EC at highest adaptation gain */

            SET_WORD(bp_ptr, sut1_low, sut1_high, 500);  /* 500,000 symbols */
	    RESTART_SUT1;

            state->stage = TM_ERLE_ADAPT_EC2;
            break;

        case TM_ERLE_ADAPT_EC2:
            TIMER_BREAK(TIMERMASK_SUT1);

#ifdef TDEBUG
            printf("Higher, ");
#endif

	    BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_gain, HIGHER_GAIN); /* EC at higher adaptation gain */

            SET_WORD(bp_ptr, sut1_low, sut1_high, 500); /* 500,000 symbols */
	    RESTART_SUT1;

            state->stage = TM_ERLE_ADAPT_EC3;
            break;

        case TM_ERLE_ADAPT_EC3:
            TIMER_BREAK(TIMERMASK_SUT1);

#ifdef TDEBUG
            printf("High  ");
#endif
	    BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_gain, HIGH_GAIN); /* EC at high adaptation gain */

            SET_WORD(bp_ptr, sut1_low, sut1_high, 500); /* 500,000 symbols */
	    RESTART_SUT1;

            if ( state->erlemode.bits.nl_ec )
                {
                state->stage = TM_ERLE_ADAPT_NLEC1;
                }
            else
                {
                BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_coefficients, OFF); /* Freeze LEC */
                state->stage = TM_ERLE_BYPASS_NLEC;
                }
            break;

        case TM_ERLE_BYPASS_NLEC:
            TIMER_BREAK(TIMERMASK_SUT1);

#ifdef TDEBUG
            PREFIX;
            printf("Bypass NLEC.\r\n");
#endif

            state->stage = TM_WAIT_METER_INTERVAL1;
            state->stage2 = TM_ERLE_MEASURE;
            break;


        case TM_ERLE_ADAPT_NLEC1:
            TIMER_BREAK(TIMERMASK_SUT1);

#ifdef TDEBUG
            printf(":  NLEC High, ");
#endif
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_gain, HIGH_GAIN); /* NL EC high adaptation gain */
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_coefficients, ON); /* Adapt NL EC */

            SET_WORD(bp_ptr, sut1_low, sut1_high, 1000);
	    RESTART_SUT1;
            state->stage = TM_ERLE_ADAPT_NLEC2;
            break;

        case TM_ERLE_ADAPT_NLEC2:
            TIMER_BREAK(TIMERMASK_SUT1);

#ifdef TDEBUG
            printf("Normal\r\n");
#endif
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_gain, NORMAL_GAIN); /* NL EC normal adaptation gain */
            SET_WORD(bp_ptr, sut1_low, sut1_high, 1000);
	    RESTART_SUT1;

            state->stage = TM_ERLE_ADAPT_NLEC3;
            break;

        case TM_ERLE_ADAPT_NLEC3:
            TIMER_BREAK(TIMERMASK_SUT1);

            BP_WRITE_BIT(bp_mode_ptr, linear_ec_modes, adapt_coefficients, OFF); /* Freeze EC */
            BP_WRITE_BIT(bp_mode_ptr, nonlinear_ec_modes, adapt_coefficients, OFF); /* Freeze NLEC */

            state->stage = TM_WAIT_METER_INTERVAL1;
            state->stage2 = TM_ERLE_MEASURE;
            break;

        case TM_ERLE_MEASURE:
            TIMER_BREAK(TIMERMASK_METER);

            /* Read SLM meter */
            READ_METER_REG(slm_low, slm_high, slm, 0);

            /* Read FELM meter */
            READ_METER_REG(felm_low, felm_high, felm, 0);

	    printf("ERLE Result: SLM = %u FELM = %u\r\n", slm, felm);

            BP_WRITE_BIT(bp_mode_ptr, adc_control, lb, ALB_BYPASS_HYBRID);

            state->stage = TM_WAIT_METER_INTERVAL1;
            state->stage2 = TM_AERLE_MEASURE;
            break;

        case TM_AERLE_MEASURE:
            TIMER_BREAK(TIMERMASK_METER);

            /* Read SLM meter - 'value = SLM2' */
            READ_METER_REG(slm_low, slm_high, slm, 0);
	    printf("ERLE Result: SLM2 = %u\r\n", slm);

            BP_WRITE_BIT(bp_mode_ptr, adc_control, lb ,  ALB_DISABLED);

            state->stage = TEST_MODE_IDLE;
            state->status = TM_STATUS_SUCCESS;
            return;



        /*------------------------------------------------------------*/
        /* Measure AAGC                                               */
        /*------------------------------------------------------------*/
        case TM_AAGC_INIT:
            _BtInitialize(0);

            /* Set Activation Interval */
            SET_WORD(bp_ptr, sut4_low, sut4_high, TM_AAGC_INTERVAL);
            RESTART_SUT4;



            /* Enable 2 or 4-Level Scrambled 1's */
            if ( state->erlemode.bits.transmit_level )
                {
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_TWO_LEVEL_ONES);
                }
            else
                {
                BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, data_source, SCRAMBLED_FOUR_LEVEL_ONES);
                }
            BP_WRITE_BIT(bp_mode_ptr, transmitter_modes, transmitter_off, OFF); /* Transmit! */

            SET_WORD(bp_ptr, meter_low, meter_high, 0xFFFF);

            BP_WRITE_BIT(bp_mode_ptr, timer_continous, meter, ON); /* Meter timer continous mode */

            RESTART_METER;

	    BP_WRITE_BIT(bp_mode_ptr, adc_control, again, AGAIN0DB);

            state->stage = TM_WAIT_METER_INTERVAL1;
            state->stage2 = TM_AAGC_DC_CANCEL;
	    state->status = TM_STATUS_INPROGRESS;
            return;


        case TM_AAGC_DC_CANCEL:
            TIMER_BREAK(TIMERMASK_METER);

            /*----------------------------------------------------*/
            /* Read DC meter value                                */
            /*----------------------------------------------------*/
            READ_METER_REG(dc_meter_low, dc_meter_high, value, 0);

            /*----------------------------------------------------------*/
            /* Set DC offset register to compensate for measured offset */
            /*----------------------------------------------------------*/
            SET_WORD(bp_ptr, dc_offset_low, dc_offset_high , value); /* Set DC offset register */

            state->stage = TM_WAIT_METER_INTERVAL1;
            state->stage2 = TM_AAGC_MEASURE;
            break;



        case TM_AAGC_MEASURE:
            TIMER_BREAK(TIMERMASK_METER);

            temp = bp_mode_ptr->adc_control.again;

            /* Read SLM meter */
            READ_METER_REG(slm_low, slm_high, slm, 0);

	    printf("AAGC Test: AAGC %d dB: SLM = %u\r\n", temp * 3, slm);

            if ( temp < AGAIN15DB )
                {
		bp_mode_ptr->adc_control.again = temp + 1;
		SET_WORD(bp_ptr, dc_offset_low, dc_offset_high, 0);

                state->stage = TM_WAIT_METER_INTERVAL1;
                state->stage2 = TM_AAGC_DC_CANCEL;
                break;
                }
            else
                {
                state->stage = TEST_MODE_IDLE;
		state->status = TM_STATUS_SUCCESS;
                return;
                }

        default:
            state->stage = TEST_MODE_IDLE;
            return;

        } /* end switch tm_stage */

	if (state->status == TM_STATUS_INPROGRESS &&
	    (bp_ptr->timer_source & TIMERMASK_SUT4))
		state->status = TM_STATUS_TIMEOUT;

} /* END _HandleTestMode() */
