/*
 * The hard part of L2CONV initialization: reading and parsing VCLIST.TXT
 */

#include "types.h"
#include "stdio.h"
#include "ctype.h"
#include "strings.h"
#include "rt11ffs.h"
#include "../libffs/ffsimpl.h"
#include "monitor_api.h"
#include "../libutil/hec.h"
#include "struct.h"
#include "global.h"
#include "param.h"

extern struct vc vc_table[MAX_VCS];
extern struct l2conv_global_config global_cfg;

static const u_short vclist_txt_filename[3]
	__attribute__ ((section (".rodata")))
	= {0x8A04, 0x3B4C, 0x80D4};

static struct find_file_results findstruct;
static int curfrag, charsleft;
static char *frag_char_ptr;
static int lineno;
static char linebuf[256];

static
frag_getc()
{
	int c;

loop:	if (charsleft) {
		c = *frag_char_ptr++;
		charsleft--;
		if (!charsleft)
			curfrag++;
		return(c);
	}
	if (curfrag >= findstruct.nfrags)
		return(EOF);
	else {
		frag_char_ptr = findstruct.frags[curfrag].addr;
		charsleft = findstruct.frags[curfrag].nblks * RT11FFS_AU_SIZE;
		goto loop;
	}
}

static
getline()
{
	char *cp;
	int c, i;

	for (cp = linebuf, i = 0; ; ) {
		c = frag_getc();
		if (c == EOF) {
			if (i)
				break;
			else
				return(-1);
		}
		if (c == '\n')
			break;
		if (!c || c == '\r')
			continue;
		if (i >= sizeof(linebuf) - 1) {
			printf("VCLIST.TXT line %d is too long\r\n", lineno);
			return(-1);
		}
		*cp++ = c;
		i++;
	}
	*cp = '\0';
	return(0);
}

static
parse_line()
{
	struct vc *vc;
	unsigned vpi, vci, val;
	int llc, iponly;
	char *cp, *np, *arg, *err;
	u_long atmaddr;
	u_short q922;
	int i;

	for (cp = linebuf; isspace(*cp); cp++)
		;
	if (!*cp || *cp == '!')
		return(0);
	if (!isdigit(*cp)) {
inv:		err = "syntax error";
bail:		printf("VCLIST.TXT line %d: %s\r\n", lineno, err);
		return(-1);
	}
	for (np = cp; isdigit(*cp); cp++)
		;
	vpi = atoi(np);
	if (!ispunct(*cp++))
		goto inv;
	if (!isdigit(*cp))
		goto inv;
	for (np = cp; isdigit(*cp); cp++)
		;
	vci = atoi(np);
	if (!isspace(*cp++))
		goto inv;
	if (vpi > 255) {
		err = "VPI value is too large";
		goto bail;
	}
	if (vci > 65535) {
		err = "VCI value is too large";
		goto bail;
	}
	atmaddr = (vpi << 20) | (vci << 4);
	if (!atmaddr) {
		err = "ATM address of all zeros is reserved";
		goto bail;
	}

	for (i = 0; i < global_cfg.nvcs; i++)
		if (*(u_long *)vc_table[i].cellhdr_normal == atmaddr) {
			err = "duplicate VC definition";
			goto bail;
		}
	if (global_cfg.nvcs >= MAX_VCS) {
		err = "too many VCs defined";
		goto bail;
	}
	vc = vc_table + global_cfg.nvcs;
	*(u_long *)vc->cellhdr_normal = atmaddr;
	vc->cellhdr_normal[4] = compute_hec(vc->cellhdr_normal);
	*(u_long *)vc->cellhdr_last = atmaddr | 0x02;
	vc->cellhdr_last[4] = compute_hec(vc->cellhdr_last);

	while (isspace(*cp))
		cp++;
	if (!isalpha(*cp))
		goto inv;
	for (np = cp; isgraph(*cp); cp++)
		;
	if (isspace(*cp))
		*cp++ = '\0';
	else if (*cp)
		goto inv;
	if (!strcasecmp(np, "FRF.5") || !strcasecmp(np, "FRF5"))
		vc->convmode = CONVMODE_FRF5;
	else if (!strcasecmp(np, "FRF.8") || !strcasecmp(np, "FRF8"))
		vc->convmode = CONVMODE_FRF8;
	else if (!strcasecmp(np, "FUNI"))
		vc->convmode = CONVMODE_FUNI;
	else if (!strcasecmp(np, "PPP") || !strcasecmp(np, "PPPoA"))
		vc->convmode = CONVMODE_PPPOA_LLC;
	else
		goto inv;

	if (vc->convmode == CONVMODE_FUNI) {
		q922 = 0x0001;
		q922 |= (vpi & 15) << 10;
		q922 |= (vci & 0x30) << 10;
		q922 |= (vci & 15) << 4;
	} else
		q922 = 0;
	llc = 1;
	iponly = 0;

param:	while (isspace(*cp))
		cp++;
	if (!*cp || *cp == '!')
		goto finish;
	if (!isalpha(*cp))
		goto inv;
	for (arg = cp; isalnum(*cp); cp++)
		;
	if (*cp == '=')
		*cp++ = '\0';
	else
		goto inv;
	if (!isdigit(*cp))
		goto inv;
	for (np = cp; isdigit(*cp); cp++)
		;
	if (isspace(*cp))
		*cp++ = '\0';
	else if (*cp)
		goto inv;
	val = atoi(np);
	if (!strcasecmp(arg, "DLCI")) {
		if (vc->convmode == CONVMODE_PPPOA_LLC) {
			err = "DLCI parameter is meaningless in this mode";
			goto bail;
		}
		if (val > 1023) {
			err = "DLCI value is too large";
			goto bail;
		}
		q922 = 0x0001;
		q922 |= (val & 0x3F0) << 6;
		q922 |= (val & 15) << 4;
	} else if (!strcasecmp(arg, "LLC")) {
		if (vc->convmode != CONVMODE_PPPOA_LLC) {
			err = "LLC parameter is meaningful only in PPPoA mode";
			goto bail;
		}
		llc = val;
	} else if (!strcasecmp(arg, "IPONLY")) {
		if (vc->convmode != CONVMODE_FRF8) {
		    err = "IPONLY parameter is meaningful only in FRF.8 mode";
		    goto bail;
		}
		iponly = val;
	} else {
		err = "invalid additional parameter";
		goto bail;
	}
	goto param;

finish:	switch (vc->convmode) {
	case CONVMODE_FRF5:
		if (q922)
			goto set_q922_mode;
		if (global_cfg.hdlc_handling_mode) {
hdlc_conflict:		err = "HDLC->ATM handling mode conflict";
			goto bail;
		}
		global_cfg.hdlc_handling_mode = HDLC_MODE_FRF5;
		break;
	case CONVMODE_FRF8:
		if (!q922) {
			err = "DLCI must be set for FRF.8";
			goto bail;
		}
		if (iponly)
			vc->convmode = CONVMODE_IPV4_VC;
		/* FALL THRU */
	case CONVMODE_FUNI:
	set_q922_mode:
		vc->q922_hdr = q922;
		if (!global_cfg.hdlc_handling_mode)
			global_cfg.hdlc_handling_mode = HDLC_MODE_Q922;
		else if (global_cfg.hdlc_handling_mode != HDLC_MODE_Q922)
			goto hdlc_conflict;
		for (i = 0; i < global_cfg.nvcs; i++)
			if (vc_table[i].q922_hdr == q922) {
				err = "DLCI conflict with another VC";
				goto bail;
			}
		break;
	case CONVMODE_PPPOA_LLC:
		if (!llc)
			vc->convmode = CONVMODE_PPPOA_VC;
		if (global_cfg.hdlc_handling_mode)
			goto hdlc_conflict;
		global_cfg.hdlc_handling_mode = HDLC_MODE_PPP;
		break;
	}
	global_cfg.nvcs++;
	return(0);
}

l2conv_init()
{
	if (monapi_ffsfindwrap(vclist_txt_filename, &findstruct) < 0)
		return(-1);
	curfrag = 0;
	frag_char_ptr = findstruct.frags[0].addr;
	charsleft = findstruct.frags[0].nblks * RT11FFS_AU_SIZE;

	bzero(vc_table, sizeof vc_table);
	bzero(&global_cfg, sizeof global_cfg);

	for (lineno = 1; ; lineno++) {
		if (getline() < 0)
			break;
		if (parse_line() < 0)
			return(-1);
	}
	if (!global_cfg.nvcs) {
		monapi_error("At least one VC must be defined in VCLIST.TXT");
		return(-1);
	}
	global_cfg.vc_table = vc_table;

	init_buf_freelist();
	return(0);
}

get_global_config(copy)
	struct l2conv_global_config *copy;
{
	bcopy(&global_cfg, copy, sizeof global_cfg);
	return(0);
}
