/*
 * Packet conversion in the ATM->HDLC direction
 */

#include "types.h"
#include "l2encap.h"
#include "struct.h"
#include "frf8.h"

extern u_char llc_magic_snap[3];
extern u_char llc_magic_nlpid[3];
extern u_char oui_ethertype[3];
extern u_char oui_bridgetypes[3];

/*
 * Assumptions:
 *   buf_vc field set correctly
 *   AAL5 trailer already processed
 *   buf_datalen already set to trim the AAL5 trailer and padding
 *
 * Return value 0:  converted successfully
 * Return value -1: some rule violated, discard packet
 * rx_encap_* stats in the VC struct are counted here
 */
pkt_atm2hdlc(buf)
	struct buffer *buf;
{
	struct vc *vc;
	u_char *headptr;
	struct rfc1483_hdr_short *s1483;
	struct rfc1483_hdr_long *l1483;
	struct fr_hdr_short *frs;
	struct fr_hdr_long *frl;
	int frf8_mode, frf8_proto;
	u_char oui[3];

	vc = buf->buf_vc;
	headptr = buf->buf_dataspc + buf->buf_dataoff;

	/* handle the simple conversion modes first */
	switch (vc->convmode) {
	case CONVMODE_FRF5:
		/* vacuous! */
		return(0);
	case CONVMODE_FUNI:
		/* prepend Q.922 header */
		headptr -= 2;
		buf->buf_dataoff -= 2;
		buf->buf_datalen += 2;
		*(u_short *)headptr = vc->q922_hdr;
		return(0);
	case CONVMODE_PPPOA_VC:
		/* prepend FF 03 */
ppp_out:	headptr -= 2;
		buf->buf_dataoff -= 2;
		buf->buf_datalen += 2;
		*(u_short *)headptr = 0xFF03;
		return(0);
	case CONVMODE_IPV4_VC:
		/* prepend RFC 1490 header for IP */
		frf8_mode = FRF8_MODE_NLPID;
		frf8_proto = NLPID_IPv4;
		goto frf8_out;
	}

	/* the remaining modes expect RFC 1483 headers from the ATM side */
	if (buf->buf_datalen < sizeof(struct rfc1483_hdr_short)) {
		vc->rx_encap_invalid++;
		return(-1);
	}
	if (!bcmp(headptr, llc_magic_snap, 3)) {
		if (buf->buf_datalen < sizeof(struct rfc1483_hdr_long)) {
			vc->rx_encap_invalid++;
			return(-1);
		}
		l1483 = (struct rfc1483_hdr_long *) headptr;
		headptr += sizeof(struct rfc1483_hdr_long);
		buf->buf_dataoff += sizeof(struct rfc1483_hdr_long);
		buf->buf_datalen -= sizeof(struct rfc1483_hdr_long);
		bcopy(l1483->snap_oui, oui, 3);
		frf8_proto = l1483->snap_type;
		if (!bcmp(l1483->snap_oui, oui_ethertype, 3)) {
			if (frf8_proto == ETHERTYPE_IP) {
				frf8_mode = FRF8_MODE_NLPID;
				frf8_proto = NLPID_IPv4;
				vc->rx_encap_ipv4++;
			} else if (frf8_proto == ETHERTYPE_IPv6) {
				frf8_mode = FRF8_MODE_NLPID;
				frf8_proto = NLPID_IPv6;
				vc->rx_encap_ipv6++;
			} else {
				frf8_mode = FRF8_MODE_SNAP_ROUTED;
				vc->rx_encap_routed++;
			}
		} else if (!bcmp(l1483->snap_oui, oui_bridgetypes, 3)) {
			frf8_mode = FRF8_MODE_SNAP_BRIDGED;
			switch (frf8_proto) {
			case 0x0001:
			case 0x0002:
			case 0x0003:
			case 0x0004:
			case 0x0007:
			case 0x0008:
			case 0x0009:
			case 0x000A:
				/* strip 2 pad octets */
				if (buf->buf_datalen < 2) {
					vc->rx_encap_invalid++;
					return(-1);
				}
				headptr += 2;
				buf->buf_dataoff += 2;
				buf->buf_datalen -= 2;
				/* FALL THRU */
			case 0x000B:
			case 0x000E:
			case 0x000F:
				vc->rx_encap_bridged++;
				break;
			default:
				vc->rx_encap_invalid++;
				return(-1);
			}
		} else {
			frf8_mode = FRF8_MODE_SNAP_UNKNOWN;
			vc->rx_encap_otheroui++;
		}
	} else if (!bcmp(headptr, llc_magic_nlpid, 3)) {
		s1483 = (struct rfc1483_hdr_short *) headptr;
		headptr += sizeof(struct rfc1483_hdr_short);
		buf->buf_dataoff += sizeof(struct rfc1483_hdr_short);
		buf->buf_datalen -= sizeof(struct rfc1483_hdr_short);
		frf8_mode = FRF8_MODE_NLPID;
		frf8_proto = s1483->nlpid;
		if (frf8_proto == NLPID_PAD || frf8_proto == NLPID_SNAP) {
			vc->rx_encap_invalid++;
			return(-1);
		}
		if (frf8_proto == NLPID_PPP)
			vc->rx_encap_ppp++;
		else
			vc->rx_encap_nlpid++;
	} else {
		vc->rx_encap_invalid++;
		return(-1);
	}

	/* Weed out PPPoA/LLC conversion mode */
	if (vc->convmode == CONVMODE_PPPOA_LLC) {
		if (frf8_mode == FRF8_MODE_NLPID && frf8_proto == NLPID_PPP)
			goto ppp_out;
		else
			return(-1);
	}

	/* FRF.8 output stage */
frf8_out:
	if (frf8_mode == FRF8_MODE_NLPID) {
		headptr -= sizeof(struct fr_hdr_short);
		buf->buf_dataoff -= sizeof(struct fr_hdr_short);
		buf->buf_datalen += sizeof(struct fr_hdr_short);
		frs = (struct fr_hdr_short *) headptr;
		frs->fr_q922_addr = vc->q922_hdr;
		frs->fr_control = FR_CONTROL_UI;
		frs->fr_nlpid = frf8_proto;
	} else {
		headptr -= sizeof(struct fr_hdr_long);
		buf->buf_dataoff -= sizeof(struct fr_hdr_long);
		buf->buf_datalen += sizeof(struct fr_hdr_long);
		frl = (struct fr_hdr_long *) headptr;
		frl->fr_q922_addr = vc->q922_hdr;
		frl->fr_control = FR_CONTROL_UI;
		frl->fr_nlpid1 = NLPID_PAD;
		frl->fr_nlpid2 = NLPID_SNAP;
		bcopy(oui, frl->fr_snap_oui, 3);
		frl->fr_snap_type = frf8_proto;
	}
	return(0);
}
