/*
 * FFS commands executable from FLASHW interactive mode
 */

#include "types.h"
#include "stdio.h"
#include "ctype.h"
#include "strings.h"
#include "console.h"
#include "rt11ffs.h"
#include "../libffs/ffsimpl.h"
#include "api.h"

extern struct keyword hexdump_quals[];

cmd_ffs_delete(cmd)
	char *cmd;
{
	struct param params[1];
	u_short filename_rad50[3];

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return;
	if (ascii_to_rad50_filename(params[0].text, params[0].len,
					filename_rad50, 0) < 0) {
		error("Invalid filename");
		return;
	}
	ffs_delete_file(filename_rad50, 0);
}

cmd_ffs_directory(cmd)
	char *cmd;
{
	if (parse_cmd(cmd, 0, 0, NULL, NULL, NULL, NULL))
		return;
	ffs_scan_flash(1, NULL);
}

/* see the header comment for ffs_erase_reinit() */
cmd_ffs_erase(cmd)
	char *cmd;
{
	struct param params[1];
	u_long segno;

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return;
	if (parse_hexnum(params[0].text, params[0].len, &segno))
		return;
	if (segno < FFS_MINSEG || segno > FFS_MAXSEG) {
		error("Invalid segment number");
		return;
	}
	ffs_erase_reinit(segno);
}

cmd_ffs_hexdump(cmd)
	char *cmd;
{
	struct param params[1];
	struct qual quals[1];
	u_short filename_rad50[3];
	char filename_print[16];
	struct find_file_results findstruct;
	int stat, idx;
	u_char *cp, *end;
	u_char buf[16];
	int i, c;

	quals[0].id = 0;
	if (parse_cmd(cmd, 1, 1, params, NULL, hexdump_quals, quals))
		return;
	if (quals[0].id && quals[0].value) {
		error("No value qualifier");
		return;
	}
	if (ascii_to_rad50_filename(params[0].text, params[0].len,
					filename_rad50, 0) < 0) {
		error("Invalid filename");
		return;
	}
	stat = ffs_find_file(filename_rad50, &findstruct);
	switch (stat) {
	case FIND_STAT_FOUND:
		break;
	case FIND_STAT_NOTFOUND:
		rad50_filename_to_ascii_trimspaces(filename_rad50,
						   filename_print);
		error("%s not found", filename_print);
		return;
	case FIND_STAT_INVALID_DIR:
		error("RT11FFS appears to be corrupt");
		return;
	case FIND_STAT_INVALID_FRAG:
		rad50_filename_to_ascii_trimspaces(filename_rad50,
						   filename_print);
		error("%s: invalid fragmentation", filename_print);
		return;
	}
	for (idx = 0; idx < findstruct.nfrags; idx++) {
		cp = findstruct.frags[idx].addr;
		end = cp + findstruct.frags[idx].nblks * RT11FFS_AU_SIZE;
		for (; cp < end; cp += 16) {
			if (check_intr())
				return;
			bcopy(cp, buf, 16);
			printf("%08X:", cp);
			for (i = 0; i < 16; i++) {
				if ((i & 3) == 0)
					putchar(' ');
				printf(" %02X", buf[i]);
			}
			if (quals[0].id) {
				printf("  ");
				for (i = 0; i < 16; i++) {
					c = buf[i];
					if (!isprint(c))
						c = '.';
					putchar(c);
				}
			}
			printf("\r\n");
		}
	}
}

cmd_ffs_initialize(cmd)
	char *cmd;
{
	struct param params[1];
	u_long segno;

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return;
	if (parse_hexnum(params[0].text, params[0].len, &segno))
		return;
	if (segno < FFS_MINSEG || segno > FFS_MAXSEG) {
		error("Invalid segment number");
		return;
	}
	ffs_init_segment(segno);
}

static const struct keyword store_quals[] = {
	{"CONTIGUOUS", 1, 'C'},
	{"SEGMENT", 1, 'S'},
	{NULL, 0, -1}};

cmd_ffs_store(cmd)
	char *cmd;
{
	struct param params[3];
	struct qual quals[3], *qp;
	u_short filename_rad50[3];
	struct ffs_create_options opts;
	u_long datasrc, nblocks, segno;

	bzero(quals, sizeof quals);
	if (parse_cmd(cmd, 3, 3, params, NULL, store_quals, quals))
		return;
	if (parse_hexnum(params[0].text, params[0].len, &datasrc))
		return;
	if (datasrc & 1) {
		error("Odd address");
		return;
	}
	if (parse_hexnum(params[1].text, params[1].len, &nblocks))
		return;
	if (ascii_to_rad50_filename(params[2].text, params[2].len,
					filename_rad50, 0) < 0) {
		error("Invalid filename");
		return;
	}
	bzero(&opts, sizeof opts);
	for (qp = quals; qp->id; qp++)
		switch (qp->id) {
		case 'C':
			if (qp->value) {
				error("No value qualifier");
				return;
			}
			opts.contig = 1;
			continue;
		case 'S':
			if (!qp->value) {
				error("Value qualifier");
				return;
			}
			if (parse_hexnum(qp->value, qp->vallen, &segno))
				return;
			if (segno < FFS_MINSEG || segno > FFS_MAXSEG) {
				error("Invalid segment number");
				return;
			}
			opts.startseg = segno;
			continue;
		}
	ffs_create_file(filename_rad50, nblocks, datasrc, &opts);
}

static const struct keyword ffs_subcmds[] = {
	{"DELETE", 2, (long) &cmd_ffs_delete},
	{"DIRECTORY", 2, (long) &cmd_ffs_directory},
	{"ERASE", 1, (long) &cmd_ffs_erase},
	{"HEXDUMP", 1, (long) &cmd_ffs_hexdump},
	{"INITIALIZE", 1, (long) &cmd_ffs_initialize},
	{"STORE", 1, (long) &cmd_ffs_store},
	{NULL, 0, -1}};

cmd_ffs(cmd)
	char *cmd;
{
	char *cp = cmd, *subcmd;
	int count;
	void (*subcmdfunc)();
	int ambigsubcmd;

	while (*cp == ' ')
		cp++;
	if (!issym(*cp)) {
		error("Subcommand required");
		return(0);
	}
	for (subcmd = cp, count = 0; issym(*cp); cp++)
		count++;
	ambigsubcmd = 0;
	subcmdfunc = (void (*)())
			find_keyword(subcmd, count, ffs_subcmds, &ambigsubcmd);
	if ((long) subcmdfunc != -1)
		(*subcmdfunc)(cp);
	else {
		if (ambigsubcmd)
			error("Ambiguous subcommand");
		else
			error("Illegal subcommand");
	}
	return(0);
}
