#include "types.h"
#include "stdio.h"
#include "ctype.h"
#include "strings.h"
#include "console.h"
#include "../lib8973/typedefs.h"
#include "../lib8973/bitpump.h"
#include "../lib8973/analogtests.h"

struct analogtests_state albstate;

alb_runloop()
{
	while (1) {
		if (check_intr())
			return(1);
		analogtests_main(&albstate);
		switch (albstate.status) {
		case TM_STATUS_INPROGRESS:
			continue;
		case TM_STATUS_SUCCESS:
			return(0);
		case TM_STATUS_TIMEOUT:
			printf("Test ended with a timeout\r\n");
			/* FALL THRU */
		case TM_STATUS_FAILED:
			return(1);
		default:
		    printf("Invalid status code from analogtests_main()!\r\n");
			return(1);
		}
	}
}

static const struct keyword alb_modes[] = {
	{"INTERNAL", 2, ALB_TRANSMITTING},
	{"ISOLATED", 2, ALB_SILENT},
	{"NOHYBRID", 3, ALB_BYPASS_HYBRID},
	{"NORMAL", 3, ALB_DISABLED},
	{"SILENT", 1, ALB_SILENT},
	{"TRANSMITTING", 1, ALB_TRANSMITTING},
	{"TX", 1, ALB_TRANSMITTING},
	{NULL, 0, -1}};

cmd_alb(cmd)
	char *cmd;
{
	struct param params[1];
	int code, ambig = 0;

	if (parse_cmd(cmd, 1, 1, params, NULL, NULL, NULL))
		return(0);
	code = find_keyword(params[0].text, params[0].len, alb_modes, &ambig);
	if (code == -1) {
		if (ambig)
			error("Ambiguous mode keyword");
		else
			error("Invalid mode keyword");
		return(0);
	}
	bzero(&albstate, sizeof albstate);
	albstate.albtype = code;
	albstate.stage = TM_ALB_INITIALIZE;
	alb_runloop();
	return(0);
}

static const struct keyword erle_quals[] = {
	{"AAGC", 1, 'A'},
	{"LEVEL", 1, 'L'},
	{"NLEC", 1, 'N'},
	{NULL, 0, -1}};

cmd_erle(cmd)
	char *cmd;
{
	struct qual quals[4];
	int i, num;
	int aagc = 12, level = 4, nlec = 0;	/* Conexant recom. defaults */

	bzero(quals, sizeof quals);
	if (parse_cmd(cmd, 0, 0, NULL, NULL, erle_quals, quals))
		return(0);
	for (i = 0; quals[i].id; i++) {
		if (!quals[i].value) {
			error("Value qualifier");
			return(0);
		}
		if (parse_decnum(quals[i].value, quals[i].vallen, &num))
			return(0);
		switch (quals[i].id) {
		case 'A':
			if (num % 3 || num > 15) {
				error("Invalid AAGC value");
				return(0);
			}
			aagc = num;
			break;
		case 'L':
			if (num != 2 && num != 4) {
				error("Invalid level setting");
				return(0);
			}
			level = num;
			break;
		case 'N':
			if (num > 1) {
				error("Invalid NLEC setting");
				return(0);
			}
			nlec = num;
			break;
		}
	}

	/* do it! */
	bzero(&albstate, sizeof albstate);
	albstate.erlemode.bits.again = aagc / 3;
	albstate.erlemode.bits.transmit_level = (level == 2);
	albstate.erlemode.bits.nl_ec = nlec;

	printf("Background Test - Transmitter OFF\r\n");
	albstate.erlemode.bits.transmit_state = 0;
	albstate.stage = TM_ERLE_INIT;
	if (alb_runloop())
		return(0);

	printf("ERLE Test - Transmitter ON, %d Level\r\n", level);
	albstate.erlemode.bits.transmit_state = 1;
	albstate.stage = TM_ERLE_INIT;
	if (alb_runloop())
		return(0);

	printf("AAGC Check - Transmitter ON, %d Level\r\n", level);
	albstate.stage = TM_AAGC_INIT;
	if (alb_runloop())
		return(0);

	printf("Test Complete...\r\n");
	return(0);
}
