
/*  ************************************************************************  *
 *				     list.c				      *
 *  ************************************************************************  */

/*  This file contains routines for constructing and maintaining a linked
    list in which entries describe XMS allocations.  */

#include    <malloc.h>
#include    <memory.h>

#include    "standard.h"

#include    "xmswatch.h"

struct LIST *CreateList (WORD handles)
{
    register struct LIST *list;
    WORD space;

    /*	The memory requirement is the given number of HANDLE structures,
	plus the few fields in the list header.  Allocate the necessary
	memory from the heap, initialise the header and clear the space
	occupied by the HANDLE structures.  */

    space = handles * sizeof (struct HANDLE);
    list = (struct LIST *) malloc (sizeof (struct LIST) + space);
    if (list != NULL) {
	list -> pLowest = NULL;
	list -> pUnused = list -> array;
	list -> pEnd = list -> pUnused + handles;
	memset (list -> array, 0, space);
    }
    return (list);
}

struct HANDLE *GetUnusedEntry (register struct LIST *list)
{
    register struct HANDLE *ptr;

    /*	Advance through the array of HANDLE structures, looking for one that
	has not been marked as used.  Keep the pUnused field in the LIST
	header as a 'bookmark' for the next time the function is called.  */

    ptr = list -> pUnused;
    while (ptr < list -> pEnd) {
	if (NOT (ptr -> wFlags & ENTRY_USED)) {
	    ptr -> wFlags = ENTRY_USED;
	    list -> pUnused = ptr + 1;
	    return (ptr);
	}
	ptr ++;
    }
    ptr = list -> array;
    while (ptr < list -> pUnused) {
	if (NOT (ptr -> wFlags & ENTRY_USED)) {
	    ptr -> wFlags = ENTRY_USED;
	    list -> pUnused = ptr + 1;
	    return (ptr);
	}
	ptr ++;
    }
    return (NULL);
}

/*  Given the base address for an EMB that is not yet listed, the following
    function assigns the EMB to an unused HANDLE structure in the list and
    connects the new entry to the existing chain.  */

struct HANDLE *AddEntry (struct LIST *list, DWORD newbase)
{
    register struct HANDLE **mark, *ptr;
    struct HANDLE *unused;

    /*	Claim an unused entry.	*/

    unused = GetUnusedEntry (list);
    if (unused == NULL) return (NULL);

    /*	Entries are ordered according to the linear address of the EMBs that
	they describe.	Advance the ptr variable from the entry for the
	lowest EMB until it describes an EMB with higher base address than
	the argument newbase.  Before each advance, save the address of the
	pNext field - this will be the insertion point for a new entry when
	the search concludes.  */

    mark = &(list -> pLowest);
    ptr = list -> pLowest;
    while (ptr != NULL AND ptr -> dwBase <= newbase) {
	mark = &(ptr -> pNext);
	ptr = ptr -> pNext;
    }

    /*	Thread the next unused HANDLE entry into the list between the
	entries pointed to by mark (indirectly) and ptr.  */

    unused -> pNext = ptr;
    *mark = unused;

    /*	Fill in the linear address field.  */

    unused -> dwBase = newbase;

    /*	Return the address of the newly-assigned HANDLE structure so that
	the caller may enter additional information.  */

    return (unused);
}

/*  The next function is similar to the one above, but works with a HANDLE
    structure that has already been initialised and need only be linked with
    the other entries in the list.  */

void LinkEntry (struct LIST *list, struct HANDLE *entry)
{
    register struct HANDLE **mark, *ptr;
    DWORD newbase = entry -> dwBase;

    mark = &(list -> pLowest);
    ptr = list -> pLowest;
    while (ptr != NULL AND ptr -> dwBase <= newbase) {
	mark = &(ptr -> pNext);
	ptr = ptr -> pNext;
    }

    entry -> wFlags |= ENTRY_USED;
    entry -> pNext = ptr;
    *mark = entry;
}

/*  ************************************************************************  */

