
/*  ************************************************************************  *
 *				     free.c				      *
 *  ************************************************************************  */

#include    <stdio.h>

#include    "standard.h"

#include    "xmsfunc.h"
#include    "xmswatch.h"

/*  XMS services (the familiar pointer to the XMS calling address, some
    macros that serve as in-line functions and some genuine functions)	*/

extern void (_far *lpXMS) (void);

extern BOOL LockEMB (WORD, DWORD *);			// in xmsfunc.c
extern BOOL GetEMBInfo (WORD, struct EMBINFO *);	// in xmsfunc.c

BOOL AllocateLargestEMB (WORD *, WORD *);
WORD QueryFreeHandles (void);

/*  A pointer and function which assist the management of data about free
    memory blocks and unassigned handles  */

struct LIST *pFreeEMBs;
WORD BuildListOfFreeEMBs (struct LIST *);

/*  In addition to various sentences, the report contains a table with two
    columns under the following header.  */

static CHAR header [] = "\n\
  Linear Address  Size (KB)  Handle\n\
  ---------------------------------";

extern BOOL LockAllEMBs;

/*  Functions that deal with the presentation of particular items of data  */

void PutLinearAddress (struct HANDLE *);
void PutSizeInKB (struct HANDLE *);
void PutShortfall (WORD);

/*  ========================================================================  */

void ReportFreeEMBs (void)
{
    register struct HANDLE *ptr;
    register WORD handles;
    WORD unaccounted;

    printf ("\nReport on Free Extended Memory:");

    /*	Begin with the total amount of free extended memory managed through
	the XMS.  This is returned in dx by the XMS function which provides
	information about extended memory availability.  */

    printf ("\n\n  Free memory (KB):  %u", HIGHWORD (QueryFreeXMS ()));

    /*	Find out how many XMS handles are available for association with
	extended memory blocks.  */

    handles = QueryFreeHandles ();
    printf ("\n  Available handles:  %u\n", handles);
    if (handles == 0) return;

    /*	The XMS provides only a little information about free extended
	memory - the total amount available and the largest free block.  To
	discover the distribution of free blocks requires that each such
	block be allocated temporarily - working from the largest downwards.
	Do not display information until the complete picture has been
	uncovered.  Instead, build the results into a sorted list so they
	may be presented in order of increasing base address rather than
	decreasing size.

	Note that the BuildListOfFreeEMBs function releases all the
	temporary allocations once it has completed its data collection, so
	the list is just a snapshot of the situation some time during the
	execution of that function.

	One side-effect of delaying the report is that no DOS functions are
	called while this program has allocated XMS handles.  This avoids
	complications with DOS exceptions that might terminate the program
	before the XMS handles have been returned to the system.  */

    pFreeEMBs = CreateList (handles);
    if (pFreeEMBs == NULL) {
	printf ("\nInsufficient internal heap space to complete analysis\n");
	return;
    }

    unaccounted = BuildListOfFreeEMBs (pFreeEMBs);

    /*	If the only entries in the table describe handles taken from the
	pool, do not present an empty table of results.  In this case, the
	list has been built only to obtain the numerical values of the spare
	handles (a meaningful notion only for some XMS servers) should they
	be needed by a future extension to the program.  */

    if (pFreeEMBs -> pLowest == NULL)
	return;

    /*	Retrieve the results from the list, following the links that have
	been set up in ascending order of base address.  */

    printf (header);

    for (ptr = pFreeEMBs -> pLowest; ptr != NULL; ptr = ptr -> pNext) {

	/*  For each handle associated with a non-empty block of memory,
	    display the base address and size, separating them with enough
	    spaces to fit correctly in the columns defined by the header.  */

	printf ("\n     ");
	PutLinearAddress (ptr);
	printf ("     ");
	PutSizeInKB (ptr);
	printf ("     %04Xh", ptr -> wHandle);
    }
    if (unaccounted) PutShortfall (unaccounted);
    printf ("\n");
}

void PutLinearAddress (struct HANDLE *ptr)
{
    /*	If the EMB described by the designated HANDLE structure could not be
	locked, display a set of question marks to indicate that the base
	address is unknown.  */

    if (ptr -> wFlags & PHYSICAL_EMB) printf ("%08lXh", ptr -> dwBase);
    else printf ("???????? ");
}

void PutSizeInKB (struct HANDLE *ptr)
{
    printf ("%5u", ptr -> wSizeInKB);
}

void PutShortfall (WORD amount)
{
    printf ("\n\n  Unable to account for additional %uKB", amount);
}

WORD BuildListOfFreeEMBs (register struct LIST *list)
{
    register struct HANDLE *ptr;
    WORD handle;
    WORD sizeinkb, unaccounted;
    DWORD linear_address;

    /*	Keep allocating XMS handles, working downwards from the largest
	free block.  */

    while (AllocateLargestEMB (&handle, &sizeinkb)) {

	/*  Lock the EMB in order to obtain its linear address.  Obtain an
	    entry for this block in the designated list, using the base
	    address as the sort key.  A lock failure presumably indicates
	    virtual memory - place such blocks at the end of the list.	*/

	if (LockAllEMBs && LockEMB (handle, &linear_address)) {
	    ptr = AddEntry (list, linear_address);
	    if (ptr == NULL) break;		    // shouldn't happen
	    ptr -> wFlags |= PHYSICAL_EMB;	    //	 but check anyway
	}
	else {
	    ptr = AddEntry (list, -1);
	    if (ptr == NULL) break;
	}

	/*  The AddEntry function records the block's linear address in the
	    appropriate field:	since it has already used the base address
	    to find a suitable place in the list, it may as well also store
	    the value.	The handle and size, however, still need to be
	    entered, along with flags that describe the block's status.  */

	ptr -> wHandle = handle;
	ptr -> wSizeInKB = sizeinkb;
	ptr -> wFlags |= (sizeinkb ? FREE_EMB : POOL_HANDLE);
    }

    /*	Check to see that the handle supply was not exhausted before all
	extended memory had been allocated.  */

    unaccounted = HIGHWORD (QueryFreeXMS ());

    /*	Tidy up.  All the XMS handles that have been allocated just to
	discover the layout of free extended memory must be unlocked (if
	necessary) and returned to the system.	*/

    for (ptr = list -> pLowest; ptr != NULL; ptr = ptr -> pNext) {
	if (ptr -> wFlags & (FREE_EMB | POOL_HANDLE)) {
	    handle = ptr -> wHandle;
	    if (ptr -> wFlags & PHYSICAL_EMB) UnlockEMB (handle);
	    ReleaseEMB (handle);
	}
    }

    /*	Return the amount of extended memory that could not be allocated.  */

    return (unaccounted);
}

BOOL AllocateLargestEMB (WORD *handleptr, WORD *sizeptr)
{
    register WORD sizeinkb;
    DWORD temp;
    struct EMBINFO info;

    /*	Determine the size of the largest free EMB, then attempt to allocate
	it.  */

    sizeinkb = (WORD) QueryFreeXMS ();
    temp = AllocateEMB (sizeinkb);
    if ((BOOL) temp) {

	/*  If successful, record the handle (returned in register dx by the
            allocation function and subsequently stored as the high word of
	    the temp variable).  */

	*handleptr = HIGHWORD (temp);

	/*  As a defensive measure, use XMS function 0Eh to check the EMB
	    size:  it is actually possible for HIMEM.SYS to allocate a
	    larger block than requested (in an admittedly exceptional
	    circumstance where the allocation happens to exhaust the supply
	    of spare handles).	*/

	*sizeptr = GetEMBInfo (*handleptr, &info) ? info.wSizeInKB : sizeinkb;
	return (TRUE);
    }
    return (FALSE);
}

WORD QueryFreeHandles (void)
{
    register WORD freehandles = 0;
    register WORD dummyhandle;
    DWORD temp;
    struct EMBINFO info;

    /*	Allocate a dummy handle so that XMS function 0Eh can be used to
	obtain the number of other handles available.  */

    temp = AllocateEMB (0);
    if ((BOOL) temp) {
	dummyhandle = HIGHWORD (temp);
	if (GetEMBInfo (dummyhandle, &info)) freehandles = info.cFreeHandles;

	/*  Release the dummy handle and adjust the count of available
	    handles accordingly.  */

	if (ReleaseEMB (dummyhandle)) freehandles ++;
    }
    return (freehandles);
}

/*  ************************************************************************  */

