
/*  ************************************************************************  *
 *				    arena.c				      *
 *  ************************************************************************  */

#include    "arena.h"
#include    "psp.h"
#include    "standard.h"
#include    "sysvars.h"

/*  The module requires the address of the DOS kernel's SYSVARS structure in
    order to locate the first memory arena header.  */

extern struct SYSVARS _far *lpSysvars;

/*  By defining the symbol _MANUAL_UMBLINK_ at compile time and initialising
    the following pointer at run-time, this module's search of the memory
    arena system extends to the upper memory chain without requiring
    explicit UMB linkage through a DOS function.  This allows a search of
    all DOS memory to be conducted (by the GetArenaSegment function) in
    circumstances where DOS functions cannot be used (e.g., from within a
    device driver call).  */

#ifdef _MANUAL_UMBLINK_
extern WORD _far *lpUMBLinkSeg;
#endif

/*  In the situation where this module is imagined to be needed while DOS
    calls are not available, the default data segments in DGROUP and the
    default code segment _TEXT will have been discarded in favour of keeping
    only a very small resident footprint for the program.  In this case, the
    variables _osmajor and _osminor in which the C run-time records the DOS
    version for later reference will no longer be present:  initialisation
    code for this module must record the version number at a location
    guaranteed to remain part of the program.  */

#ifdef _MANUAL_UMBLINK_
extern struct {
    BYTE minor;
    BYTE major;
} wDOSVersion;
#define _osmajor wDOSVersion.major
#define _osminor wDOSVersion.minor
#else
extern BYTE _osminor;			// defined in STDLIB.H
extern BYTE _osmajor;			// defined in STDLIB.H
#endif

/*  Macros to create far pointers to memory arena headers and PSPs from just
    the segment addresses  */

#define MK_ARENA_FP(seg)    ((struct ARENA_HEADER _far *) MK_FP (seg, 0))
#define MK_PSP_FP(seg)	    ((struct PSP _far *) MK_FP (seg, 0))

/*  ------------------------------------------------------------------------  */

/*  Begin with two short routines for basic arena header management.  The
    first function extracts an arena header's signature byte after checking
    its validity.  */

BYTE GetValidArenaID (_segment arenaseg)
{
    BYTE id = MK_ARENA_FP (arenaseg) -> cSignature;
    if (id == ARENA_IN_CHAIN OR id == ARENA_LAST) return (id);
    else return (0);
}

/*  The second function takes the segment address of a supposed arena header
    and returns the segment address of the corresponding memory block's
    upper bound - where there will usually be the next arena header.  */

_segment GetNextArena (_segment arenaseg)
{
    register WORD inc;

    /*	The distance (in paragraphs) to the next header is the size of the
	memory block which follows the header, plus one more paragraph for
	the header itself.  */

    inc = MK_ARENA_FP (arenaseg) -> wParagraphs + 1;

    /*	Before adding this increment to the given arena segment, consider
	that two pathological cases may occur if the arena header is
	corrupt (or if the segment does not in fact address a true arena
	header).

	The main problem occurs if the number of paragraphs in the block is
	recorded as 0xFFFF, giving an increment of zero.  This case is
	dangerous, since its omission while following an arena chain (or a
	supposed one) results in an infinite loop.

	Secondly, the addition may overflow the available 16 bits (the
	memory arena chain does not wrap around the 1MB boundary, so neither
	should the calculation).  In a routine that "walks" the arena chain,
	this case would probably be caught as an invalid signature on the
	next round and is arguably not so important.

	Incidentally, DOS checks neither condition when it examines arena
	headers.  */

    if (inc == 0 OR inc > 0xFFFF - arenaseg) return (0);
    return (arenaseg + inc);
}

/*  ------------------------------------------------------------------------  */

/*  At the next level are functions which concentrate on the larger scale of
    the arena chain or on memory block ownership.  The simplest example is a
    function which tests whether a given segment is plausibly a memory arena
    header by checking that it is followed by a well-defined chain.  This
    function does nothing else along the way;  note though, that the two
    earlier functions spare this one from bothering with the details of
    validation and defective iteration.  */

_segment ValidateArenaChain (register _segment seg)
{
    BYTE id;

    do {
	/*  At each step, use the functions provided earlier to validate the
	    supposed arena header and find the upper bound to the
	    corresponding memory block.  */

	id = GetValidArenaID (seg);
	if (id == 0) return (0);
	seg = GetNextArena (seg);
	if (seg == 0) return (0);

	/*  Continue for as long as there seem to be more headers.  */

    } while (id == ARENA_IN_CHAIN);

    /*	Return the chain's upper bound.  Thus, for a machine with 640KB of
	conventional memory and no upper memory under DOS's control, the
	function returns 0xA000 if given the segment address of a valid
	arena header in low memory.  With upper memory available in a
	dos=umb configuration but with the UMB link disabled, the function
	returns 0x9FFF, it having no knowledge that DOS starts a second
	chain where the first one ends.  */

    return (seg);
}

/*  The following function finds the segment address of the arena header
    for the memory block containing a given segment address.  */

_segment GetArenaSegment (_segment givenseg)
{
    register _segment seg;
    _segment markseg;
    BYTE id;

    /*	In DOS versions 5 and higher, there may be two chains of memory
	arena headers.	The DOS kernel has a variable in which to keep the
	segment address of the header at the boundary.	The address of this
	kernel variable (if it exists) must be recorded by initialisation
	code (in another module) before using this function.  */

    #ifdef _MANUAL_UMBLINK_
    _segment umblinkseg = lpUMBLinkSeg != NULL ? *lpUMBLinkSeg : 0xFFFF;
    #endif

    /*	Seed the search with the segment address of the first memory arena
	header - available as the word before the SYSVARS structure in the
	DOS kernel's data area.  Then make sure that the paragraph address
	given as the function's argument does not lie below DOS's memory
	control system.  */

    seg = *((WORD _far *) lpSysvars - 1);
    if (givenseg < seg) return (0);

    /*	Follow the chain of memory arena headers upwards through memory
	until either the desired segment is passed or the chain is
	exhausted.  */

    for (;;) {

	markseg = seg;

	/*  At each step, validate the supposed arena header and find the
	    upper bound to the corresponding memory block.  */

	id = GetValidArenaID (seg);
	if (id == 0) return (0);
	seg = GetNextArena (seg);
	if (seg == 0) return (0);

	/*  If markseg <= givenseg < seg, then the desired block has been
	    found.  The first half of this inequality is already established
	    (either from the preceding step or as an entry condition for the
	    iteration).  */

	if (givenseg < seg) break;

	/*  If no more blocks exist in the chain, then abort the search.  In
	    DOS 5 and higher, a second chain may start immediately after the
	    first.  The _MANUAL_UMBLINK_ symbol supports examination of the
	    second chain in cases where explicit linkage through int 21h
	    function 5802h is not possible.  */

	if (id == ARENA_LAST) {
	    #ifndef _MANUAL_UMBLINK_
	    return (0);
	    #else
	    if (umblinkseg == 0xFFFF OR umblinkseg != seg) return (0);
	    #endif
	}
    }

    /*	If the block containing the given segment is not the last in the
	chain, check that the remainder of the arena chain is plausible
	before returning the result.  */

    if (id == ARENA_IN_CHAIN) {
	if (ValidateArenaChain (seg) == 0) return (0);
    }
    return (markseg);
}

/*  ------------------------------------------------------------------------  */

/*  As an aside, observe that the memory arena chain may conceivably provide
    the most reliable way to determine whether code is being executed during
    device driver initialisation.  Of the various traces left by the
    temporary PSP used by SYSINIT when building the system (and particularly
    while loading device drivers), the existence of a memory block owned by
    the temporary PSP is fairly secure against interference from
    user-supplied code.  */

static BOOL TestSysinitPSP (_segment seg)
{
    /*	If the segment does not look like it addresses a PSP, then fail.  */

    if (*((WORD _far *) MK_PSP_FP (seg) -> Int20h) != 0x20CD) return (FALSE);

    /*	The SYSINIT PSP is best distinguished by having no parent.  Fail the
	function if the segment addresses an ordinary PSP.  */

    if (MK_PSP_FP (seg) -> spParentPSP != 0) return (FALSE);

    /*	Otherwise, return TRUE - not as strict affirmation that the segment
	addresses the SYSINIT PSP, but that this identification is not
	implausible.  */

    return (TRUE);
}

static _segment GetSysinitPSP (void)
{
    register _segment seg, ownerseg;
    BYTE id;

    /*	Search the arena chain in conventional memory for a block assigned
	to the SYSINIT PSP.  In versions of DOS up to and including 5.0,
	the block of low memory into which SYSINIT loads device drivers and
	builds tables is not assigned to owner = 0x0008 until the system is
	formally ready to execute programs.  */

    seg = *((WORD _far *) lpSysvars - 1);
    do {
	id = GetValidArenaID (seg);
	if (id == 0) break;

	ownerseg = MK_ARENA_FP (seg) -> spOwner;
	if (TestSysinitPSP (ownerseg)) return (ownerseg);

	seg = GetNextArena (seg);
	if (seg == 0) break;

    } while (id == ARENA_IN_CHAIN);
    return (0);
}

#define IsInConfig()	(GetSysinitPSP () != 0 ? TRUE : FALSE)

/*  ========================================================================  */

static BOOL SupportsSubArenas (_segment seg)
{
    register _segment ownerseg;

    /*	The subarena scheme for detailing memory allocations made by DOS for
	device drivers and kernel structures such as SFTs, CDSs, etc was
	introduced with DOS 4.	*/

    if (_osmajor < 4) return (FALSE);

    /*	In general, the only memory blocks subdivided this way have owner
	fields indicating their private use by DOS (owner == 0x0008).  */

    ownerseg = MK_ARENA_FP (seg) -> spOwner;
    if (ownerseg == ARENA_DOS) {

	/*  Additionally, in DOS 5 and higher, the letters 'SD' (for System
	    Data) must start the name field.  */

	if (_osmajor >= 5) {
	    #define DEREF_WORD(fp) *((WORD _far *) (fp))
	    if (DEREF_WORD (MK_ARENA_FP (seg) -> cName) != MK_WORD ('D', 'S'))
		return (FALSE);
	}
    }
    else {

	/*  There is a special case, however, which must be considered when
	    the subdivision of a memory block is examined during device
	    driver initialisation.  While SYSINIT is still building drivers
	    and tables into a memory block, ownership lies with SYSINIT's
	    temporary PSP and is not assigned the special value 0x0008 until
	    later.  */

	if (NOT TestSysinitPSP (ownerseg)) return (FALSE);
    }
    return (TRUE);
}

static char ValidSubArenaIDs [] = {
    'B',        // buffers
    'D',        // device
    'E',        // buffers /x (EMS hook and driver dummy,
		//   plus hook for int 2Fh function 1Bh - DOS 4 only)
    'F',        // files (SFTs)
    'I',        // ifs (DOS 4 only)
    'L',        // lastdrive (CDSs)
    'S',        // stacks
    'T',        // install (temporary code)
    'X'         // fcbs (FCB-SFTs)
};

BYTE GetValidSubArenaID (_segment subarenaseg)
{
    register BYTE *ptr;
    register WORD count;
    BYTE ch, id;

    /*	Simply check the ID against a list of values known to be used.	*/

    id = MK_ARENA_FP (subarenaseg) -> cSignature;
    ptr = ValidSubArenaIDs;
    count = sizeof (ValidSubArenaIDs) / sizeof (ValidSubArenaIDs [0]);
    do {
	ch = *ptr ++;
	if (ch == id) return (ch);
    } while (-- count);
    return (0);
}

_segment GetSubArenaSegment (_segment baseseg, _segment givenseg)
{
    register _segment seg;
    _segment markseg, topseg;
    BYTE id;

    /*	Begin by checking that the arena header at the segment address given
	by the argument baseseg supports a chain of subarena headers.  */

    if (NOT SupportsSubArenas (baseseg)) return (0);

    /*	The subarena headers should lie within the block of memory described
	by the main header.  */

    topseg = GetNextArena (baseseg);
    if (topseg == 0) return (0);

    /*	The subheaders begin at the paragraph immediately after the main
	arena header.  */

    seg = baseseg + 1;
    markseg = 0;

    /*	Follow the chain of memory arena headers upwards through memory
	until either the desired segment is passed or the chain is
	exhausted.  */

    for (;;) {

	id = GetValidSubArenaID (seg);
	if (id == 0) break;

	/*  Having encountered a plausible subarena header, record its
	    segment address, since it may be the one that is wanted.  */

	markseg = seg;

	/*  Find the end of the sub-block.  Stop looping if this overruns
	    the main block (which the subarenas are supposed to subdivide).  */

	seg = GetNextArena (seg);
	if (seg == 0) break;
	if (seg >= topseg) break;

	/*  If markseg <= givenseg < seg, then the desired sub-block has
	    been found.  Otherwise, continue the search.  */

	if (givenseg < seg) return (markseg);
    }

    /*	There is a circumstance in which the chain of subarena headers may
	overrun the upper bound obtained from the main block and not
	indicate corruption.  When device drivers are initialised, their
	subarena header is only partially valid.  The size field is
	untouched until the device driver's initialisation is complete.
	This means that following a subarena chain at this stage of the
	system's startup sequence is technically unreliable.  Defensive
	coding should prevent mishap and reduce the opportunity for spurious
	identification, but the latter cannot be elimated entirely.  */

    if (NOT IsInConfig ()) return (0);
    else return (markseg);
}

/*  ************************************************************************  */

