#include "nucleus.h"
#include "storage/pcdisk.h"
#include "wpa_supplicant/src/utils/includes.h"
#include "wpa_supplicant/src/utils/common.h"
#include "wpa_supplicant/src/utils/os.h"
#include "wpa_supplicant/src/common/defs.h"
#include "wpa_supplicant/src/common/wpa_ctrl.h"
#include "wpa_supplicant/wpa_supplicant_api.h"
#include <stdio.h>

/* Please set the following macro to the name of the WLAN interface. */
#define WDEMO_IFACE_NAME                "ar6k0"

/* Demo initialization task configuration. */
#define WDEMO_TASK_STACK_SIZE           5000
#define WDEMO_TASK_PRIORITY             3
#define WDEMO_TASK_TIME_SLICE           0

/* WPA network configuration to be used. The default configuration of
 * the WPA Supplicant will be overridden by the following configuration.
 *
 * Please see the following file for help with the syntax of this file:
 * nucleus/os/networking/wpa_supplicant/wpa_supplicant/wpa_supplicant.conf
 */
CHAR WDEMO_Wpa_Config[] =
    "ap_scan=2\n"
    "network={\n"
    "    ssid=\"sample_wifi\"\n"
    "    key_mgmt=WPA-PSK\n"
    "    proto=WPA\n"
    "    pairwise=TKIP\n"
    "    group=TKIP\n"
    "    psk=\"password\"\n"
    "}\n";

/* Global variables. */
NU_TASK WDEMO_Main_Task_CB;

extern int wpa_debug_level;
int old_wpa_debug_level;

/* Function prototypes. */
VOID WDEMO_Main_Task(UNSIGNED argc, VOID *argv);

/*************************************************************************
*
*   FUNCTION
*
*       Application_Initialize
*
*   DESCRIPTION
*
*       This function creates different tasks of the demo application.
*
**************************************************************************/
VOID Application_Initialize(NU_MEMORY_POOL *system_pool,
                            NU_MEMORY_POOL *uncached_system_pool)
{
    STATUS          status;
    VOID            *pointer;

    /* Reduce the supplicant's debug level so that it produced
     * minimum output. */
    old_wpa_debug_level = wpa_debug_level;
    wpa_debug_level = MSG_ERROR + 1;

    /* Create the main demo task. */
    status = NU_Allocate_Memory(system_pool, &pointer,
                                WDEMO_TASK_STACK_SIZE,
                                NU_NO_SUSPEND);

    if (status != NU_SUCCESS)
    {
        printf("Cannot allocate demo task stack memory.\r\n");
        while(1);
    }

    status = NU_Create_Task(&WDEMO_Main_Task_CB, "demo_wl",
                            WDEMO_Main_Task, 1, system_pool, pointer,
                            WDEMO_TASK_STACK_SIZE,
                            WDEMO_TASK_PRIORITY,
                            WDEMO_TASK_TIME_SLICE,
                            NU_PREEMPT, NU_START);

    if (status != NU_SUCCESS)
    {
        printf("Cannot create WDEMO_Main_Task.\r\n");
        while(1);
    }

    return;

} /* Application_Initialize */

/*************************************************************************
*
*   FUNCTION
*
*       WDEMO_Write_File
*
*   DESCRIPTION
*
*       This function writes the contents of a specified file
*       to the Ramdisk.
*
**************************************************************************/
STATUS WDEMO_Write_File(CHAR *path, CHAR *data, INT data_len)
{
    INT fd;
    STATUS status = NU_SUCCESS;

    /* Open the destination and source files. */
    fd = NU_Open(path, PO_CREAT | PO_TRUNC | PO_WRONLY, PS_IWRITE);

    if (fd >= 0)
    {
        NU_Write(fd, data, data_len);

        /* Close the file now. */
        NU_Close(fd);
    }
    else
    {
        status = (STATUS)fd;
    }

    return (status);

} /* WDEMO_Write_File */

/*************************************************************************
*
*   FUNCTION
*
*      WDEMO_Get_IP_Address
*
*   DESCRIPTION
*
*      This function returns the currently configured IP address of the
*      wireless interface. It calls NET APIs to determine the IP.
*
**************************************************************************/
VOID WDEMO_Get_IP_Address(CHAR *ip_str)
{
    NU_IOCTL_OPTION ioctl_opt;
    CHAR if_name[32];
    STATUS      status = -1;

    strcpy(if_name, WDEMO_IFACE_NAME);

    /* Initialize the IOCTL option to zero. */
    memset(&ioctl_opt, 0, sizeof(NU_IOCTL_OPTION));

    /* Find the IP address attached to the network device. */
    ioctl_opt.s_optval = (UINT8*)if_name;

    /* Call NU_Ioctl to get the IP address. */
    status = NU_Ioctl(SIOCGIFADDR, &ioctl_opt, sizeof(ioctl_opt));

    /* Check if we got the IP */
    if (status == NU_SUCCESS)
    {
        /* Create IP Address string */
        sprintf(ip_str, "%d.%d.%d.%d",
                ioctl_opt.s_ret.s_ipaddr[0],
                ioctl_opt.s_ret.s_ipaddr[1],
                ioctl_opt.s_ret.s_ipaddr[2],
                ioctl_opt.s_ret.s_ipaddr[3]);
    }
    else
    {
        /* Format the string */
        sprintf(ip_str, "%.*s",
                (strlen("Not assigned") + 1),
                "Not assigned");
    }

} /* WDEMO_Get_IP_Address */

/*************************************************************************
*
*   FUNCTION
*
*       WDEMO_Main_Task
*
*   DESCRIPTION
*
*       This is the task entry point of the main demo task.
*
**************************************************************************/
VOID WDEMO_Main_Task(UNSIGNED argc, VOID *argv)
{
    STATUS status;
    CHAR ip_addr[24];

    printf("Wireless Sample Application\r\n\r\n");

    printf("*** Waiting for a WLAN interface\r\n");

    /* Wait until at least one WLAN interface is detected. */
    while (WPA_Supplicant_Get_Iface_Count() < 1)
        NU_Sleep(NU_PLUS_TICKS_PER_SEC);

    printf("*** WLAN interface detected\r\n");

    /* Override the default configuration of the WPA Supplicant. */
    status = WDEMO_Write_File("A:\\wpa_sup.cfg", WDEMO_Wpa_Config,
                                          strlen(WDEMO_Wpa_Config));

    if (status == NU_SUCCESS)
    {
        /* Make the supplicant reload the new configuration file. */
        status = WPA_Supplicant_Reload_Config();
    }

    if (status == NU_SUCCESS)
    {
        printf("*** Successfully loaded a new WPA configuration\r\n");
    }
    else
    {
        printf("*** Unable to load new WPA configuration\r\n");
        while(1);
    }

    printf("*** Restoring WPA Supplicant debug level\r\n");

    /* Restore the supplicant's debug level now. */
    wpa_debug_level = old_wpa_debug_level;

    if (status == NU_SUCCESS)
    {
        /* Wait until the network interface is "up". This code
         * assumes that the WLAN card is the only network interface
         * being used when running this sample. */
        status = NETBOOT_Wait_For_Network_Up(NU_SUSPEND);

        if (status == NU_SUCCESS)
        {
            /* Display the IP address of the WLAN interface. */
            WDEMO_Get_IP_Address(ip_addr);
            printf("*** WLAN IP address: %s\r\n", ip_addr);
        }
    }

    return;

} /* WDEMO_Main_Task */
