/* Include */
#include <stdio.h>
#include "nucleus.h"
#include "storage/nu_storage.h"

/* Macros */
#define TASK_STACK_SIZE     (NU_MIN_STACK_SIZE * 32)
#define TASK_PRIORITY       31
#define TASK_TIMESLICE      0

#define BUFFER_SIZE     200
#define SAMPLE_COUNT    10

/* Internal globals */
static  NU_TASK Storage_Sample_CB;
static  VOID    Storage_Sample_Task(UNSIGNED argc, VOID *argv);
static  CHAR    outbuf[BUFFER_SIZE];    /* Buffer for writing file data */
static  CHAR    inbuf[BUFFER_SIZE];     /* Buffer for reading in file data */
static  CHAR    root_string[4];         /* Default drive root pathname. */

/* Internal function prototypes */
static STATUS Find_Storage_Device(VOID);
static STATUS Create_Log_File(CHAR *log_file_name);
static STATUS Append_To_Log_File(CHAR *log_file_name);
static STATUS Read_Log_File(CHAR *log_file_name);

/*************************************************************************
*
*   FUNCTION
*
*       Application_Initialize
*
*   DESCRIPTION
*
*       Initializes application by creating a Print Task
*
*   CALLED BY
*
*       System App Init service
*
*   CALLS
*
*       NU_Allocate_Memory
*       NU_Create_Task
*       NU_Deallocate_Memory
*
*   INPUTS
*
*       mem_pool                            Memory pool
*       uncached_mem_pool                   Uncached memory pool
*
*   OUTPUTS
*
*       None
*
*************************************************************************/
VOID Application_Initialize (NU_MEMORY_POOL* mem_pool, NU_MEMORY_POOL* uncached_mem_pool)
{
    VOID* pointer;
    STATUS status;
    
    /* Allocate memory for Print Time Task */
    status = NU_Allocate_Memory(mem_pool, &pointer, TASK_STACK_SIZE, NU_NO_SUSPEND);

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Create the task  */
        status = NU_Create_Task(&Storage_Sample_CB, "PRNTTSK", Storage_Sample_Task, 0, NU_NULL, pointer,
                                TASK_STACK_SIZE, TASK_PRIORITY, TASK_TIMESLICE,
                                NU_PREEMPT, NU_START);
        
        /* If previous operation was not successful */
        if(status != NU_SUCCESS)
        {
            (VOID)NU_Deallocate_Memory(pointer);
        }
    }
}


/*************************************************************************
*
*   FUNCTION
*
*       Storage_Sample_Task
*
*   DESCRIPTION
*
*       This task:
*        1) Creates a log file on the drive
*        2) Writes test strings to the log file
*        3) Closes the log file
*        4) Creates a second log file on the drive
*        5) Writes test strings to the log file
*        6) Closes the log file
*        7) Re-opens the first log file
*        8) Appends updated test strings to the log file
*        9) Closes the log file
*       10) Opens the first log file
*       11) Reads the log file and prints the information
*       12) Closes the log file
*       13) Opens the second log file
*       14) Reads the log file and prints the information
*       15) Closes the log file
*
*   CALLED BY
*
*       Task Scheduler
*
*   CALLS
*
*       printf
*       Find_Storage_Device
*       Create_Log_File
*       Append_To_Log_File
*       Read_Log_File
*
*   INPUTS
*
*       argc
*       argv
*
*   OUTPUTS
*
*       None
*
*************************************************************************/
static VOID Storage_Sample_Task (UNSIGNED argc, VOID *argv)
{
    STATUS      status;         /* Status */
    CHAR		test_1[] = "X:\\Test_1.log";
    CHAR		test_2[] = "X:\\Test_2.log";

    printf("\r\n\nStorage Sample Starting\r\n\n");
    
    /************************************
     * Find a storage device we can use *
     ************************************/
    status = Find_Storage_Device();

    if (status == NU_SUCCESS)
    {
        /**********************************************************
         * Use the drive letter discovered by Find_Storage_Device *
         **********************************************************/
    	test_1[0] = test_2[0] = root_string[0];

    	/*********************************************
         * Create, open and write to Test_1.log file *
         *********************************************/
    
        status = Create_Log_File(test_1);
    }

    if (status == NU_SUCCESS)
    {
        /*********************************************
         * Create, open and write to Test_2.log file *
         *********************************************/
    
        status = Create_Log_File(test_2);
    }

    if (status == NU_SUCCESS)
    {
        /**************************************
         * Open and append to Test_1.log file *
         **************************************/
    
        status = Append_To_Log_File(test_1);
    }

    if (status == NU_SUCCESS)
    {
        /*********************************
         * Open and read Test_1.log file *
         *********************************/
    
        status = Read_Log_File(test_1);
    }

    if (status == NU_SUCCESS)
    {
        /*********************************
         * Open and read Test_2.log file *
         *********************************/
    
        status = Read_Log_File(test_2);
    }

    printf("Storage Sample Complete\r\n\n");
}


/*************************************************************************
*
*   FUNCTION
*
*       Find_Storage_Device
*
*   DESCRIPTION
*
*       This function performs all the application level initialization:
*       find an available STORAGE device and initialize the root directory
*       string.
*
*   CALLED BY
*
*       Storage_Sample_Task
*
*   CALLS
*
*       printf
*       NU_Storage_Device_Wait
*       NU_List_Mount
*       NU_Free_List
*
*   INPUTS
*
*       None
*
*   OUTPUTS
*
*       status
*
*************************************************************************/
static STATUS Find_Storage_Device(VOID)
{
    STATUS      status = -1;            /* Status */
    INT         i;                      /* General purpose variable */
    MNT_LIST_S  *mount_list = NU_NULL;  /* Pointer to list of mounted drives */

    /*********************************************************
     * Wait for a Nucleus Storage device to become available *
     *********************************************************/

    /* Check every 2 seconds for up to 30 seconds. */
    for(i = 0; ((i < 15) && (status != NU_SUCCESS)); i++)
    {
        printf("Searching for a device!\r\n\n");
        
        status = NU_Storage_Device_Wait(NU_NULL, (NU_PLUS_TICKS_PER_SEC*2));
    }

    /* If we found a device */
    if (status == NU_SUCCESS)
    {
        /******************************************
         * Get a list of currently mounted drives *
         ******************************************/
    
		status = NU_List_Mount(&mount_list);

		if (status != NU_SUCCESS)
		{
			printf("No drives mounted!\r\n\n");
		}
		else
        {
			/*****************************************************
			 * Create the root directory string based on the     *
			 * information for the first drive in the mount list *
			 *****************************************************/
    
            /* Append ":\\" to drive letter to create the root directory string. */
            root_string[0] = mount_list->mnt_name[0];
            root_string[1] = ':';
            root_string[2] = '\\';
            root_string[3] = '\0';
    
            /* Free the mount list memory */
            NU_Free_List((VOID **)&mount_list);
        }
    }
    
    return(status);
}


/*************************************************************************
*
*   FUNCTION
*
*       Create_Log_File
*
*   DESCRIPTION
*
*       This function performs all the application level steps to
*       open a file and create it if it does not exist, then write
*       test strings to the file.
*
*   CALLED BY
*
*       Storage_Sample_Task
*
*   CALLS
*
*       NU_Open
*       printf
*       NU_Sleep
*       NU_Retrieve_Clock
*       memset
*       sprintf
*       NU_Write
*       NU_Close
*
*   INPUTS
*
*       *log_file_name
*
*   OUTPUTS
*
*       status
*
*************************************************************************/
static STATUS Create_Log_File(CHAR *log_file_name)
{
    STATUS      status = NU_SUCCESS;    /* Status */
    INT         fd;                     /* File descriptor for file access */
    INT         bytes_written;          /* Number of bytes written */
    INT         i;                      /* General purpose variable */
    INT         seconds;                /* System clock time in seconds */

    /**********************************************************
     * Create and open a log file for Read/Write on the drive *
     **********************************************************/

    fd = NU_Open(log_file_name,(PO_TEXT|PO_RDWR|PO_CREAT|PO_TRUNC), (PS_IWRITE | PS_IREAD));
    
    if (fd < 0)
    {
        printf("Failed to open %s file for Read/Write!\r\n\n", log_file_name);
        
        status = fd;
    }

    /*************************************
     * Write information to the log file *
     *************************************/

    if (status == NU_SUCCESS)
    {
        for (i = 0; ((i < SAMPLE_COUNT) && (status == NU_SUCCESS)); i++)
        {
            /* Suspend for 1 second */
            NU_Sleep(NU_PLUS_TICKS_PER_SEC);
            
            /* Calculate current system clock time in seconds */
            seconds = (INT)(NU_Retrieve_Clock()/NU_PLUS_TICKS_PER_SEC);

            /* Create the string to write to the log file */
            (VOID)memset (outbuf, 0, BUFFER_SIZE);
            sprintf(outbuf, "Test string 1 written to %s at system time %d\r\n", log_file_name, seconds);

            /* Write the information to the drive */
            bytes_written = NU_Write(fd, outbuf, BUFFER_SIZE);

            /* If the bytes where written */
            if (bytes_written < 0)
            {
                printf("Failed to write to %s file!\r\n\n", log_file_name);
            }
            else
            {
                printf("Write Test string 1 to %s at system time %d\r", log_file_name, seconds);
            }
        }
        
        /* Output a blank line */
        printf("\r\n\n");

        /* Close the log file */
        status = NU_Close(fd);

        if (status != NU_SUCCESS)
        {
            printf("Failed to close %s file!\r\n\n", log_file_name);
        }
        
        if (bytes_written < 0)
        {
            status = bytes_written;
        }
    }
    
    return(status);
}


/*************************************************************************
*
*   FUNCTION
*
*       Append_To_Log_File
*
*   DESCRIPTION
*
*       This function performs all the application level steps to
*       open a file then append test strings to the end of the file.
*
*   CALLED BY
*
*       Storage_Sample_Task
*
*   CALLS
*
*       NU_Open
*       printf
*       NU_Sleep
*       NU_Retrieve_Clock
*       memset
*       sprintf
*       NU_Write
*       NU_Close
*
*   INPUTS
*
*       *log_file_name
*
*   OUTPUTS
*
*       status
*
*************************************************************************/
static STATUS Append_To_Log_File(CHAR *log_file_name)
{
    STATUS      status = NU_SUCCESS;    /* Status */
    INT         fd;                     /* File descriptor for file access */
    INT         bytes_written;          /* Number of bytes written */
    INT         i;                      /* General purpose variable */
    INT         seconds;                /* System clock time in seconds */

    /**************************************************
     * Re-open the log file for Append and Write only *
     **************************************************/

    fd = NU_Open(log_file_name,(PO_TEXT|PO_WRONLY|PO_APPEND), PS_IWRITE);
    
    if (fd < 0)
    {
        printf("Failed to open %s file for Write only!\r\n\n", log_file_name);
        
        status = fd;
    }

    /**************************************
     * Append information to the log file *
     **************************************/

    if (status == NU_SUCCESS)
    {
        for (i = 0; ((i < SAMPLE_COUNT) && (status == NU_SUCCESS)); i++)
        {
            /* Suspend for 1 second */
            NU_Sleep(NU_PLUS_TICKS_PER_SEC);
            
            /* Calculate current system clock time in seconds */
            seconds = (INT)(NU_Retrieve_Clock()/NU_PLUS_TICKS_PER_SEC);

            /* Create the string to append to the log file */
            (VOID)memset (outbuf, 0, BUFFER_SIZE);
            sprintf(outbuf, "Test string 2 appended to %s at system time %d\r\n", log_file_name, seconds);

            /* Write the information to the drive */
            bytes_written = NU_Write(fd, outbuf, BUFFER_SIZE);

            /* If the bytes where written */
            if (bytes_written < 0)
            {
                printf("Failed to write to %s file!\r\n\n", log_file_name);
            }
            else
            {
                printf("Append Test string 2 to %s at system time %d\r", log_file_name, seconds);
            }
        }
        
        /* Output a blank line */
        printf("\r\n\n");

        /* Close the log file */
        status = NU_Close(fd);

        if (status != NU_SUCCESS)
        {
            printf("Failed to close %s file!\r\n\n", log_file_name);
        }
        
        if (bytes_written < 0)
        {
            status = bytes_written;
        }
    }

    return(status);
}


/*************************************************************************
*
*   FUNCTION
*
*       Read_Log_File
*
*   DESCRIPTION
*
*       This function performs all the application level steps to
*       open a file, read the information, then print it to the terminal.
*
*   CALLED BY
*
*       Storage_Sample_Task
*
*   CALLS
*
*       NU_Open
*       printf
*       memset
*       NU_Read
*       NU_Close
*
*   INPUTS
*
*       *log_file_name
*
*   OUTPUTS
*
*       status
*
*************************************************************************/
static STATUS Read_Log_File(CHAR *log_file_name)
{
    STATUS      status = NU_SUCCESS;    /* Status */
    INT         fd;                     /* File descriptor for file access */
    INT         bytes_read;             /* Number of bytes read */

    /**************************************
     * Re-open the log file for Read only *
     **************************************/

    fd = NU_Open(log_file_name,(PO_TEXT|PO_RDONLY), PS_IREAD);
    
    if (fd < 0)
    {
        printf("Failed to open %s file for Read only!\r\n\n", log_file_name);
        
        status = fd;
    }

    /**************************************
     * Read information from the log file *
     **************************************/

    if (status == NU_SUCCESS)
    {
        /* Print the file name */
        printf(" *** Contents of %s ***\r\n\n", log_file_name);

        do
        {
            /* Read the information from the drive */
            (VOID)memset (inbuf, 0, BUFFER_SIZE);
            bytes_read = NU_Read(fd, inbuf, BUFFER_SIZE);

            if (bytes_read < 0)
            {
                printf("Failed to read from %s file!\r\n\n", log_file_name);
            }
            else
            {
                /* Print the information */
                printf("%s", inbuf);
            }

        /* Read until we reach the end of the file */
        } while (bytes_read > 0);

        /* Output a blank line */
        printf("\r\n");

        /* Close the log file */
        status = NU_Close(fd);

        if (status != NU_SUCCESS)
        {
            printf("Failed to re-close %s file!\r\n\n", log_file_name);
        }
        
        if (bytes_read < 0)
        {
            status = bytes_read;
        }
    }

    return(status);
}
