/* Include */
#include <stdio.h>
#include "nucleus.h"
#include "storage/nu_storage.h"

/* Macros */
#define TASK_STACK_SIZE     (NU_MIN_STACK_SIZE * 32)
#define TASK_PRIORITY       31
#define TASK_TIMESLICE      0

#define YEARMASK            0x007F
#define MONTHMASK           0x000F
#define DAYMASK             0x001F

#define HOURMASK            0x001F
#define MINUTEMASK          0x003F
#define SECONDMASK          0x003F

/* Internal globals */
static  NU_TASK Sample_CB;
static  VOID    Sample_Task(UNSIGNED argc, VOID *argv);


/*************************************************************************
*
*   FUNCTION
*
*       Application_Initialize
*
*   DESCRIPTION
*
*       Initializes application by creating a Print Task
*
*   CALLED BY
*
*       System App Init service
*
*   CALLS
*
*       NU_Allocate_Memory
*       NU_Create_Task
*       NU_Deallocate_Memory
*
*   INPUTS
*
*       mem_pool                            Memory pool
*       uncached_mem_pool                   Uncached memory pool
*
*   OUTPUTS
*
*       None
*
*************************************************************************/
VOID Application_Initialize (NU_MEMORY_POOL* mem_pool, NU_MEMORY_POOL* uncached_mem_pool)
{
    VOID* pointer;
    STATUS status;
    
    /* Allocate memory for Print Time Task */
    status = NU_Allocate_Memory(mem_pool, &pointer, TASK_STACK_SIZE, NU_NO_SUSPEND);

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Create the task  */
        status = NU_Create_Task(&Sample_CB, "PRNTTSK", Sample_Task, 0, NU_NULL, pointer,
                                TASK_STACK_SIZE, TASK_PRIORITY, TASK_TIMESLICE,
                                NU_PREEMPT, NU_START);
        
        /* If previous operation was not successful */
        if(status != NU_SUCCESS)
        {
            (VOID)NU_Deallocate_Memory(pointer);
        }
    }
}


/*************************************************************************
*
*   FUNCTION
*
*       Sample_Task
*
*   DESCRIPTION
*
*       This task:
*       1) Reads through the directory using find first and find next.
*       2) Prints the resulting file tree to the serial port.
*             name, size, date. (whatever makes sense)
*       3) Waits for a new disk insertion event and list its files.
*
*   CALLED BY
*
*       Task Scheduler
*
*   CALLS
*
*       printf
*       NU_Storage_Device_Wait
*       NU_List_Mount
*       NU_Free_List
*       NU_Get_First
*       NU_Get_Next
*       NU_Sleep
*
*   INPUTS
*
*       argc
*       argv
*
*   OUTPUTS
*
*       None
*
*************************************************************************/
static VOID Sample_Task (UNSIGNED argc, VOID *argv)
{
    STATUS      status = -1;            /* Status */
    MNT_LIST_S  *mount_list = NU_NULL;  /* Pointer to list of mounted drives */
    INT         i;                      /* General purpose variable */
    DSTAT       statobj;                /* File information structure */
    CHAR        pattern[] = "X:\\*";    /* File pattern to search for */


    printf("\r\n\nStorage Sample Starting\r\n\n");

    /*********************************************************
     * Wait for a Nucleus Storage device to become available *
     *********************************************************/

    /* Check every 2 seconds for up to 30 seconds. */
    for(i = 0; ((i < 15) && (status != NU_SUCCESS)); i++)
    {
        printf("Searching for a device!\r\n\n");
        
        status = NU_Storage_Device_Wait(NU_NULL, (NU_PLUS_TICKS_PER_SEC*2));
    }

    /* If at least one device is available, list all devices */
    while (status == NU_SUCCESS)
    {
        /******************************************
         * Get a list of currently mounted drives *
         ******************************************/
    
        if (status == NU_SUCCESS)
        {
            status = NU_List_Mount(&mount_list);
    
            if (status != NU_SUCCESS)
            {
                printf("No drives mounted!\r\n\n");
            }
        }

        for (;mount_list != NU_NULL; mount_list = mount_list->next)
        {
			/* Copy the drive letter */
			pattern[0] = mount_list->mnt_name[0];

			/* A simple directory list subroutine. */
			status = NU_Get_First(&statobj, pattern);

            if (status != NU_SUCCESS)
                printf("Directory listing from device (%s) : EMPTY!\n\r", mount_list->dev_name);
            else
                printf("Directory listing from device (%s) :\n\r", mount_list->dev_name);

			while(status == NU_SUCCESS)
			{

				/* Print file name and extension */
				printf("    Short File Name => %s",statobj.sfname);

				if( statobj.fext[0] != ' ' )
				{
					printf(".%s\r\n", statobj.fext );
				}
				else
				{
					printf("\r\n");
				}

				if( statobj.fattribute & ADIRENT)
				{
					/* Print directory */
					printf("    <DIR>\r\n");
				}
				else if (statobj.fattribute & AVOLUME)
				{
					/* Print volume */
					printf("    <VOL>\r\n");
				}
				else
				{
					/* Print file size */
					printf("    Size => %6d\r\n", (INT)statobj.fsize );
				}

				/* Print date last updated */
				printf("    Updated Date => %d-%d-%d, Updated Time => %d:%d:%d\r\n",
					   ((statobj.fupdate >> 9 ) & YEARMASK) + 1980,
					   ((statobj.fupdate >> 5 ) & MONTHMASK),
					   ( statobj.fupdate & DAYMASK),
					   ((statobj.fuptime >> 11 ) & HOURMASK),
					   ((statobj.fuptime >> 5 ) & MINUTEMASK),
					   ((statobj.fuptime << 1 ) & SECONDMASK));

				/* Print date and time created */
				printf("    Created Date => %d-%d-%d, Created Time => %d:%d:%d\r\n",
					   ((statobj.fcrdate >> 9) & YEARMASK) + 1980,
					   ((statobj.fcrdate >> 5) & MONTHMASK),
					   ( statobj.fcrdate & DAYMASK),
					   ((statobj.fcrtime >> 11) & HOURMASK),
					   ((statobj.fcrtime >> 5) & MINUTEMASK),
					   ((statobj.fcrtime << 1) & SECONDMASK));

				/* Print long file name */
				printf("    Full Name => %s\r\n",statobj.lfname );

				/* Find next file/dir/volume */
				status = NU_Get_Next(&statobj);
			}

			/* Free the statobj structure */
			(VOID)NU_Done (&statobj);

			printf("\r\n");
        }

        /* Free the mount list memory */
        NU_Free_List((VOID **)&mount_list);

		/* Reset status */
        status = -1;

        /* Check every 2 seconds for up to 30 seconds. */
        for(i = 0; ((i < 15) && (status != NU_SUCCESS)); i++)
        {
            status = NU_Storage_Device_Wait(NU_NULL, (NU_PLUS_TICKS_PER_SEC*2));

            if (status != NU_SUCCESS)
            {
                printf("Searching for device!\r\n\n");
            }
        }

        if(i == 1)
        {
        	/* Add a delay to prevent constant scrolling */
        	NU_Sleep(NU_PLUS_TICKS_PER_SEC*2);
        }
    }

    printf("Storage Sample Complete\r\n\n");
}
