/* Include */
#include <stdio.h>
#include "nucleus.h"
#include "kernel/nu_kernel.h"
#include "drivers/serial.h"

/* Macros */
#define TASK_STACK_SIZE     (NU_MIN_STACK_SIZE * 32)
#define TASK_PRIORITY       31
#define TASK_TIMESLICE      0

/* Print string buffer size */
#define BUFFER_SIZE         200

/* Internal globals */
static  NU_TASK Time_Task;
static  VOID    Print_Time_Task(UNSIGNED argc, VOID *argv);

/*************************************************************************
*
*   FUNCTION
*
*       Application_Initialize
*
*   DESCRIPTION
*
*       Initializes application by creating a Print Task
*
*   CALLED BY
*
*       System App Init service
*
*   CALLS
*
*       NU_Allocate_Memory
*       NU_Create_Task
*       NU_Deallocate_Memory
*
*   INPUTS
*
*       mem_pool                            Memory pool
*       uncached_mem_pool                   Uncached memory pool
*
*   OUTPUTS
*
*       None
*
*************************************************************************/
VOID Application_Initialize (NU_MEMORY_POOL* mem_pool, NU_MEMORY_POOL* uncached_mem_pool)
{
    VOID* pointer;
    STATUS status;


    /* Allocate memory for Print Time Task */
    status = NU_Allocate_Memory(mem_pool, &pointer,
                                TASK_STACK_SIZE, NU_NO_SUSPEND);

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Create the task  */
        status = NU_Create_Task(&Time_Task, "PRNTTSK", Print_Time_Task, 0, NU_NULL, pointer,
                                TASK_STACK_SIZE, TASK_PRIORITY, TASK_TIMESLICE,
                                NU_PREEMPT, NU_START);

        /* If previous operation was not successful */
        if(status != NU_SUCCESS)
        {
            (VOID)NU_Deallocate_Memory(pointer);
        }
    }

}


/*************************************************************************
*
*   FUNCTION
*
*       Print_Time_Task
*
*   DESCRIPTION
*
*       This task opens all available serial devices,
*       prints the number of seconds elapsed since the
*       demonstration was started and
*       closes all the opened serial devices
*
*   CALLED BY
*
*       Task Scheduler
*
*   CALLS
*
*       NU_Serial_Open
*       NU_Serial_Puts
*       NU_Serial_Close
*
*   INPUTS
*
*       argc
*       argv
*
*   OUTPUTS
*
*       None
*
*************************************************************************/
static VOID Print_Time_Task (UNSIGNED argc, VOID *argv)
{
    INT seconds;
    CHAR outbuf[BUFFER_SIZE];
    STATUS status;
    SERIAL_SESSION *serial_dev_id[CFG_NU_OS_DRVR_SERIAL_MAX_DEVS_SUPPORTED];
    INT index = 0;

    printf("*** Starting Serial Demo ***\r\n\n");
    printf(">This output is printed using the STDIO interface. The demo will now look for other available serial devices. ");
    printf("If no other serial device is found and connected for terminal output, then nothing else will be printed.\r\n");

    /* Initialize device id list  */
    for(index = 0; index < CFG_NU_OS_DRVR_SERIAL_MAX_DEVS_SUPPORTED; index++)
    {
        serial_dev_id[index] = (SERIAL_SESSION*)-1;
    }

    index = 0;


    /* Open all serial devices */
    do
    {
        /* Open the next available serial port */
        status = NU_Serial_Open(NU_NULL, &serial_dev_id[index]);

        index++;

    } while(status == NU_SUCCESS);

    index = 0;

    /* Print from all serial devices */
    while(serial_dev_id[index] != (SERIAL_SESSION*)-1)
    {
        seconds = (INT)(NU_Retrieve_Clock()/NU_PLUS_TICKS_PER_SEC) + 1;

        sprintf(outbuf, "Printing on Serial Port #%d\n", index+1);
        status = NU_Serial_Puts(serial_dev_id[index], outbuf);

        sprintf(outbuf, "Time elapsed since beginning of demo is : %d seconds\n", seconds);
        status = NU_Serial_Puts(serial_dev_id[index], outbuf);

        NU_Sleep(100);

        index++;
    }

    index = 0;

    /* Close all serial devices */
    while(serial_dev_id[index] != (SERIAL_SESSION*)-1)
    {
        /* Close the device */
        status = NU_Serial_Close(serial_dev_id[index]);

        index++;
    }

}
