/***********************************
* Include Files
***********************************/
#include                <stdio.h>
#include                <string.h>
#include                "nucleus.h"
#include                "kernel/nu_kernel.h"
#include                "services/nu_services.h"
#include                "drivers/nu_drivers.h"
#include                "power_demo.h"

VOID             Display_OP_Table(VOID);
VOID             Switch_OP(UINT8);
VOID             Show_Tick_Suppression_Status(VOID);
VOID             Enable_Tick_Suppression(VOID);
VOID             Disable_Tick_Suppression(VOID);
VOID             Display_Peripheral_State_Table(VOID);
VOID             Switch_Peripheral_State(CHAR dev_name[15], UINT8 state_num);
VOID             Display_System_States_Table(VOID);
VOID             Switch_System_State(UINT8 state_num);
VOID             Create_System_States(UINT8 state_num);
VOID             Map_Peripheral(CHAR dev_name[15], UINT8 sys_state, UINT8 periph_state);
VOID             Unmap_Peripheral(CHAR dev_name[15]);

extern BOOLEAN   PMS_TS_Enabled_Flag;

#if (CFG_NU_OS_SVCS_PWR_CORE_ENABLE_DVFS == NU_TRUE)

/***********************************************************************
*
*   FUNCTION
*
*       Display_OP_Table
*
*   DESCRIPTION
*
*       This functions displays Operating Points table.
*
***********************************************************************/
VOID Display_OP_Table()
{
    UINT8   op_pt_cnt;
    UINT8   current_op;
    STATUS  status;
    CHAR    buf[80];
    CHAR    temp_buf[10];
    INT     i;
    CHAR    border_buffer[100] = "\0";

    /* Determine number of operating points */
    status = NU_PM_Get_OP_Count(&op_pt_cnt);

    if (status != NU_SUCCESS)
    {
        printf("*** Failed to get OP Count\r\n");
        return;
    }

    /* Get the current operating point */
    status = NU_PM_Get_Current_OP(&current_op);

    if (status != NU_SUCCESS)
    {
        printf("*** Failed to get current OP\r\n");
        return;
    }

    printf("\r\n");
    printf("      *** Operating Points Table ***\r\n");

    strcpy(buf, "| OP   |");
    for (i = op_pt_cnt - 1; i >= 0; i--)
    {
        if (i == current_op)
        {
            sprintf(temp_buf, "  %d*    |", i);
        }
        else
        {
            sprintf(temp_buf, "  %d     |", i);
        }
        strcat(buf, temp_buf);
    }

    memset(border_buffer, '-', strlen(buf));

    printf("%s\r\n", border_buffer);
    printf("%s\r\n", buf);
    printf("%s\r\n", border_buffer);

    strcpy(buf, "| CPU  |");
    for (i = op_pt_cnt - 1; i >= 0; i--)
    {
        UINT8       freq_id, volt_id;
        UINT32      frequency;

        /* Retrieve the Voltage and Frequency ids */
        NU_PM_Get_OP_VF(i, &freq_id, &volt_id);

        /* Retrieve the frequency (in Hz) associated with the freq_id */
        NU_PM_Get_Freq_Info(freq_id, &frequency);

        sprintf(temp_buf, " %d MHz |", (INT)frequency/1000000);
        strcat(buf, temp_buf);
    }

    printf("%s\r\n", buf);

    printf("%s\r\n", border_buffer);
    printf("(*) Current Operating Point\n");
    printf("\r\n");
    printf("-- Enter 'op <op index>' command to change Operating Point\r\n");
    printf("\r\n");
}

/***********************************************************************
*
*   FUNCTION
*
*       Switch_OP
*
*   DESCRIPTION
*
*       This functions switches Operating Point.
*
***********************************************************************/
VOID Switch_OP(UINT8 op_pt)
{
    STATUS status;

    /* Place system in the required operating point */
    status = NU_PM_Set_Current_OP(op_pt);

    if(status != NU_SUCCESS)
    {
        printf("\r\n");
        printf("Failed to set OP %d\r\n", op_pt);
        printf("\r\n");
    }
    else
    {
        printf("\r\n");
        printf("*** Successful transition to new Operating Point (OP %d)\r\n", op_pt);
        printf("\r\n");
    }
}

#endif /* (CFG_NU_OS_SVCS_PWR_CORE_ENABLE_DVFS == NU_TRUE) */

#if (CFG_NU_OS_KERN_PLUS_CORE_TICK_SUPPRESSION == NU_TRUE)

/***********************************************************************
*
*   FUNCTION
*
*       Show_Tick_Suppression_Status
*
*   DESCRIPTION
*
*       This functions shows tick suppression status.
*
***********************************************************************/
VOID Show_Tick_Suppression_Status(VOID)
{
    if (PMS_TS_Enabled_Flag == NU_TRUE)
    {
        printf("\r\n");
        printf("*** Tick Suppression is Enabled\r\n");
        printf("\r\n");
    }
    else
    {
        printf("\r\n");
        printf("*** Tick Suppression is Disabled\r\n");
        printf("\r\n");
    }

    printf("\r\n*** Enter 'ts <0/1>' command to disable/enable Tick Suppression\r\n\r\n");
}

/***********************************************************************
*
*   FUNCTION
*
*       Enable_Tick_Suppression
*
*   DESCRIPTION
*
*       This functions enables tick suppression.
*
***********************************************************************/
VOID Enable_Tick_Suppression()
{
    STATUS status;

    status = NU_PM_Start_Tick_Suppress();

    if (status == NU_SUCCESS)
    {
        printf("\r\n");
        printf("*** Tick Suppression is Enabled\r\n");
        printf("\r\n");
    }
    else
    {
        printf("\r\n");
        printf("*** Tick Suppression Enabling Failed!\r\n");
        printf("\r\n");
    }
}

/***********************************************************************
*
*   FUNCTION
*
*       Disable_Tick_Suppression
*
*   DESCRIPTION
*
*       This functions disables tick suppression.
*
***********************************************************************/
VOID Disable_Tick_Suppression()
{
    STATUS status;

    status = NU_PM_Stop_Tick_Suppress();

    if (status == NU_SUCCESS)
    {
        printf("\r\n");
        printf("*** Tick Suppression is Disabled\r\n");
        printf("\r\n");
    }
    else
    {
        printf("\r\n");
        printf("*** Tick Suppression Disabling Failed!\r\n");
        printf("\r\n");
    }
}

#endif /* (CFG_NU_OS_KERN_PLUS_CORE_TICK_SUPPRESSION == NU_TRUE) */

#if (CFG_NU_OS_SVCS_PWR_CORE_ENABLE_PERIPHERAL == NU_TRUE)

/***********************************************************************
*
*   FUNCTION
*
*       Get_Peripheral_States_Count
*
*   DESCRIPTION
*
*       This function gets peripheral states count for the give device.
*
***********************************************************************/
static UINT8 Get_Peripheral_States_Count(DV_DEV_LABEL dev_label)
{
    STATUS      status;
    INT         dev_id_cnt = 1;
    DV_DEV_ID   dev_id_list[1];
    PM_STATE_ID pwr_states_count;

    /* Get device ID of the device being tested */
    status = DVC_Dev_ID_Get(&dev_label, 1, dev_id_list, &dev_id_cnt);

    /* We expect to find only one device enabled. */
    if((status == NU_SUCCESS) && (dev_id_cnt == 1))
    {
        /* Determine number of states */
        status = NU_PM_Get_Power_State_Count(dev_id_list[0], &pwr_states_count);

        if (status == NU_SUCCESS)
        {
            return pwr_states_count;
        }
    }

    return 0;
}

/***********************************************************************
*
*   FUNCTION
*
*       Get_Max_Peripheral_States
*
*   DESCRIPTION
*
*       This function calclulates maximum peripheral states.
*
***********************************************************************/
static UINT8 Get_Max_Peripheral_States()
{
    UINT8 temp;
    UINT8 max_peripheral_states = 0;

#ifdef SERIAL_LABEL
    {
        DV_DEV_LABEL dev_label = {SERIAL_LABEL};

        temp = Get_Peripheral_States_Count(dev_label);

        if (temp > max_peripheral_states)
        {
            max_peripheral_states = temp;
        }
    }
#endif

#ifdef DISPLAY_LABEL
    {
        DV_DEV_LABEL dev_label = {DISPLAY_LABEL};

        temp = Get_Peripheral_States_Count(dev_label);

        if (temp > max_peripheral_states)
        {
            max_peripheral_states = temp;
        }
    }
#endif

#ifdef ETHERNET_LABEL
    {
        DV_DEV_LABEL dev_label = {ETHERNET_LABEL};

        temp = Get_Peripheral_States_Count(dev_label);

        if (temp > max_peripheral_states)
        {
            max_peripheral_states = temp;
        }
    }
#endif

#ifdef TOUCHPANEL_LABEL
    {
        DV_DEV_LABEL dev_label = {TOUCHPANEL_LABEL};

        temp = Get_Peripheral_States_Count(dev_label);

        if (temp > max_peripheral_states)
        {
            max_peripheral_states = temp;
        }
    }
#endif

#ifdef I2C_LABEL
    {
        DV_DEV_LABEL dev_label = {I2C_LABEL};

        temp = Get_Peripheral_States_Count(dev_label);

        if (temp > max_peripheral_states)
        {
            max_peripheral_states = temp;
        }
    }
#endif

#ifdef SPI_LABEL
    {
        DV_DEV_LABEL dev_label = {SPI_LABEL};

        temp = Get_Peripheral_States_Count(dev_label);

        if (temp > max_peripheral_states)
        {
            max_peripheral_states = temp;
        }
    }
#endif

#ifdef CAN_LABEL
    {
        DV_DEV_LABEL dev_label = {CAN_LABEL};

        temp = Get_Peripheral_States_Count(dev_label);

        if (temp > max_peripheral_states)
        {
            max_peripheral_states = temp;
        }
    }
#endif

#ifdef SDIO_HCD_LABEL
    {
        DV_DEV_LABEL dev_label = {SDIO_HCD_LABEL};

        temp = Get_Peripheral_States_Count(dev_label);

        if (temp > max_peripheral_states)
        {
            max_peripheral_states = temp;
        }
    }
#endif

    return max_peripheral_states;
}

/***********************************************************************
*
*   FUNCTION
*
*       Display_Peripheral_State
*
*   DESCRIPTION
*
*       This function displays a single peripheral's state in the table.
*
***********************************************************************/
static VOID Display_Peripheral_State(DV_DEV_LABEL dev_label, CHAR *dev_name, UINT8 max_states)
{
    STATUS      status;
    INT         dev_id_cnt = 1;
    DV_DEV_ID   dev_id_list[1];
    PM_STATE_ID pwr_states_count, cur_pwr_state;
    INT         i;

    /* Get device ID of the device being tested */
    status = DVC_Dev_ID_Get(&dev_label, 1, dev_id_list, &dev_id_cnt);

    /* We expect to find only one device enabled. */
    if((status == NU_SUCCESS) && (dev_id_cnt == 1))
    {
        /* Determine number of states */
        status = NU_PM_Get_Power_State_Count(dev_id_list[0], &pwr_states_count);

        if (status == NU_SUCCESS)
        {
            /* Ensure the device is in the correct power state */
            status = NU_PM_Get_Power_State(dev_id_list[0], &cur_pwr_state);

            if (status == NU_SUCCESS)
            {
                printf("| %-10s |", dev_name);

                if ((pwr_states_count - 1) == cur_pwr_state)
                {
                    printf("   %d*  |", (INT)pwr_states_count - 1);
                }
                else
                {
                    printf("   %d   |", (INT)pwr_states_count - 1);
                }

                for (i = max_states - 2; i >= 0; i--)
                {
                    if (i <= (max_states - (max_states - pwr_states_count + 2)))
                    {
                        if (i == cur_pwr_state)
                        {
                            printf("   %d*  |", i);
                        }
                        else
                        {
                            printf("   %d   |", i);
                        }
                    }
                    else
                    {
                        printf("   -   |");
                    }
                }
                printf("\r\n");
            }
        }
    }
}

/***********************************************************************
*
*   FUNCTION
*
*       Display_Peripheral_State_Table
*
*   DESCRIPTION
*
*       This function displays peripheral state table.
*
***********************************************************************/
VOID Display_Peripheral_State_Table(VOID)
{
    INT   i;
    CHAR  border_buffer[100] = "\0";
    CHAR  buf[100] = "\0";
    UINT8 max_states = Get_Max_Peripheral_States();

    printf("\r\n");
    printf("      *** Peripheral Power State Table ***\r\n");

    sprintf(buf, "| %-10s |  ON   |", "PS");

    for (i = max_states - 3; i >= 0; i--)
    {
        CHAR  temp_buf[15] = "\0";

        sprintf(temp_buf, " MID-%d |", i);
        strcat(buf, temp_buf);
    }
    strcat(buf, "  OFF  |");

    memset(border_buffer, '-', strlen(buf));

    printf("%s\r\n", border_buffer);
    printf("%s\r\n", buf);
    printf("%s\r\n", border_buffer);

#ifdef SERIAL_LABEL
    {
        DV_DEV_LABEL dev_label = {SERIAL_LABEL};

        Display_Peripheral_State(dev_label, "Serial", max_states);
    }
#endif

#ifdef DISPLAY_LABEL
    {
        DV_DEV_LABEL dev_label = {DISPLAY_LABEL};

        Display_Peripheral_State(dev_label, "Display", max_states);
    }
#endif

#ifdef ETHERNET_LABEL
    {
        DV_DEV_LABEL dev_label = {ETHERNET_LABEL};

        Display_Peripheral_State(dev_label, "Ethernet", max_states);
    }
#endif

#ifdef TOUCHPANEL_LABEL
    {
        DV_DEV_LABEL dev_label = {TOUCHPANEL_LABEL};

        Display_Peripheral_State(dev_label, "Touchpanel", max_states);
    }
#endif

#ifdef I2C_LABEL
    {
        DV_DEV_LABEL dev_label = {I2C_LABEL};

        Display_Peripheral_State(dev_label, "I2C", max_states);
    }
#endif

#ifdef SPI_LABEL
    {
        DV_DEV_LABEL dev_label = {SPI_LABEL};

        Display_Peripheral_State(dev_label, "SPI", max_states);
    }
#endif

#ifdef CAN_LABEL
    {
        DV_DEV_LABEL dev_label = {CAN_LABEL};

        Display_Peripheral_State(dev_label, "CAN", max_states);
    }
#endif

#ifdef SDIO_HCD_LABEL
    {
        DV_DEV_LABEL dev_label = {SDIO_HCD_LABEL};

        Display_Peripheral_State(dev_label, "SDIOH", max_states);
    }
#endif

    printf("%s\r\n", border_buffer);
    printf("(*) Current Peripheral Power State\n");
    printf("\r\n");
    printf("-- Enter 'ps <device name> <ps index>' command to change Peripheral Power State\r\n");
    printf("\r\n");

}

#endif /* (CFG_NU_OS_SVCS_PWR_CORE_ENABLE_PERIPHERAL == NU_TRUE) */

/***********************************************************************
*
*   FUNCTION
*
*       Get_Device_ID
*
*   DESCRIPTION
*
*       This function gets device id for the given device name.
*
***********************************************************************/
static STATUS Get_Device_ID(CHAR dev_name[15], DV_DEV_ID *dev_id)
{
    INT       dev_id_cnt = 1;
    DV_DEV_ID dev_id_list[1];
    STATUS    status = INVALID_DEV_NAME;

#ifdef SERIAL_LABEL
    if (strcmp(dev_name, "serial") == 0)
    {
        DV_DEV_LABEL dev_label = {SERIAL_LABEL};

        /* Get device ID of the device being tested */
        status = DVC_Dev_ID_Get(&dev_label, 1, dev_id_list, &dev_id_cnt);
    }
#endif
#ifdef ETHERNET_LABEL
    else if (strcmp(dev_name, "ethernet") == 0)
    {
        DV_DEV_LABEL dev_label = {ETHERNET_LABEL};

        /* Get device ID of the device being tested */
        status = DVC_Dev_ID_Get(&dev_label, 1, dev_id_list, &dev_id_cnt);
    }
#endif
#ifdef DISPLAY_LABEL
    else if (strcmp(dev_name, "display") == 0)
    {
        DV_DEV_LABEL dev_label = {DISPLAY_LABEL};

        /* Get device ID of the device being tested */
        status = DVC_Dev_ID_Get(&dev_label, 1, dev_id_list, &dev_id_cnt);
    }
#endif
#ifdef TOUCHPANEL_LABEL
    else if (strcmp(dev_name, "touchpanel") == 0)
    {
        DV_DEV_LABEL dev_label = {TOUCHPANEL_LABEL};

        /* Get device ID of the device being tested */
        status = DVC_Dev_ID_Get(&dev_label, 1, dev_id_list, &dev_id_cnt);
    }
#endif
#ifdef I2C_LABEL
    else if (strcmp(dev_name, "i2c") == 0)
    {
        DV_DEV_LABEL dev_label = {I2C_LABEL};

        /* Get device ID of the device being tested */
        status = DVC_Dev_ID_Get(&dev_label, 1, dev_id_list, &dev_id_cnt);
    }
#endif
#ifdef SPI_LABEL
    else if (strcmp(dev_name, "spi") == 0)
    {
        DV_DEV_LABEL dev_label = {SPI_LABEL};

        /* Get device ID of the device being tested */
        status = DVC_Dev_ID_Get(&dev_label, 1, dev_id_list, &dev_id_cnt);
    }
#endif
#ifdef CAN_LABEL
    else if (strcmp(dev_name, "can") == 0)
    {
        DV_DEV_LABEL dev_label = {CAN_LABEL};

        /* Get device ID of the device being tested */
        status = DVC_Dev_ID_Get(&dev_label, 1, dev_id_list, &dev_id_cnt);
    }
#endif
#ifdef SDIO_HCD_LABEL
    else if (strcmp(dev_name, "sdioh") == 0)
    {
        DV_DEV_LABEL dev_label = {SDIO_HCD_LABEL};

        /* Get device ID of the device being tested */
        status = DVC_Dev_ID_Get(&dev_label, 1, dev_id_list, &dev_id_cnt);
    }
#endif

    /* We expect to find only one device enabled. */
    if((status == NU_SUCCESS) && (dev_id_cnt == 1))
    {
        *dev_id = dev_id_list[0];
    }

    return status;
}

#if (CFG_NU_OS_SVCS_PWR_CORE_ENABLE_PERIPHERAL == NU_TRUE)

/***********************************************************************
*
*   FUNCTION
*
*       Switch_Peripheral_State
*
*   DESCRIPTION
*
*       This function switches state of the given peripheral.
*
***********************************************************************/
VOID Switch_Peripheral_State(CHAR dev_name[15], UINT8 state_num)
{
    STATUS    status;
    DV_DEV_ID dev_id;

    status = Get_Device_ID(dev_name, &dev_id);

    /* We expect to find only one device enabled. */
    if(status == NU_SUCCESS)
    {
        /* Ensure the device is in the correct power state */
        status = NU_PM_Set_Power_State(dev_id, state_num);

        if (status == NU_SUCCESS)
        {
            printf("\r\n");
            printf("*** Successful transition to new Peripheral Power State (%s = PS %d)\r\n", dev_name, state_num);
            printf("\r\n");
        }
        else
        {
            printf("\r\n");
            printf("*** Failed to set power state %d for device %s\r\n", state_num, dev_name);
            printf("\r\n");
        }
    }
    else if (status == INVALID_DEV_NAME)
    {
        printf("\r\n");
        printf("*** Invalid Device name %s\r\n", dev_name);
        printf("\r\n");
    }
    else
    {
        printf("\r\n");
        printf("*** Failed to get Device ID for %s\r\n", dev_name);
        printf("\r\n");
    }
}

#endif /*  (CFG_NU_OS_SVCS_PWR_CORE_ENABLE_PERIPHERAL == NU_TRUE) */

#if (CFG_NU_OS_SVCS_PWR_CORE_ENABLE_SYSTEM == NU_TRUE)

/***********************************************************************
*
*   FUNCTION
*
*       Display_Peripheral_System_State
*
*   DESCRIPTION
*
*       This function displays a single peripheral state in the table.
*
***********************************************************************/
static VOID Display_Peripheral_System_State(DV_DEV_LABEL dev_label, CHAR dev_name[15], UINT8 num_states)
{
    STATUS    status;
    INT       dev_id_cnt = 1;
    DV_DEV_ID dev_id_list[1];

    /* Get device ID of the device being tested */
    status = DVC_Dev_ID_Get(&dev_label, 1, dev_id_list, &dev_id_cnt);

    /* We expect to find only one device enabled. */
    if((status == NU_SUCCESS) && (dev_id_cnt == 1))
    {
        CHAR  buf[100] = "\0";
        UINT8 sys_mapping_exists = 0;
        INT   i;

        sprintf(buf, "| %-10s |", dev_name);
        for (i = num_states - 1; i >= 0; i--)
        {
            PM_STATE_ID  periph_state;
            CHAR  temp_buf[15] = "\0";

            status = NU_PM_Get_System_State_Map(i, dev_id_list[0], &periph_state);

            if (status == NU_SUCCESS)
            {
                sprintf(temp_buf, "  %d     |", (INT)periph_state);
                strcat(buf, temp_buf);
                sys_mapping_exists = 1;
            }
            else
            {
                sprintf(temp_buf, "  -     |");
                strcat(buf, temp_buf);
            }

        }

        if (sys_mapping_exists == 1)
        {
            printf("%s\r\n", buf);
        }
    }
}

/***********************************************************************
*
*   FUNCTION
*
*       Display_System_States_Table
*
*   DESCRIPTION
*
*       This function displays system state table.
*
***********************************************************************/
VOID Display_System_States_Table(VOID)
{
    STATUS    status = INVALID_DEV_NAME;
    UINT8     num_states, current_state;
    INT       i;
    CHAR      border_buffer[100] = "\0";
    CHAR      buf[100] = "\0";

    status = NU_PM_Get_System_State_Count(&num_states);

    if (status == PM_SYSTEM_STATE_NEED_INIT)
    {
        printf("\r\n");
        printf("*** Please Create System States first by using 'ss -create <num-states>'\r\n");
        printf("\r\n");

        return;
    }

    status = NU_PM_Get_System_State(&current_state);

    if (status != NU_SUCCESS)
    {
        printf("\r\n");
        printf("*** Unable to get current system state.\r\n");
        printf("\r\n");
    }

    printf("\r\n");
    printf("      *** System Power States Table ***\r\n");

    strcpy(buf, "| SS         |");
    for (i = num_states - 1; i >= 0; i--)
    {
        CHAR temp_buf[15];

        if (i == current_state)
        {
            sprintf(temp_buf, "  %d*    |", i);
        }
        else
        {
            sprintf(temp_buf, "  %d     |", i);
        }
        strcat(buf, temp_buf);
    }

    memset(border_buffer, '-', strlen(buf));

    printf("%s\r\n", border_buffer);
    printf("%s\r\n", buf);
    printf("%s\r\n", border_buffer);

#ifdef SERIAL_LABEL
    {
        DV_DEV_LABEL dev_label = {SERIAL_LABEL};

        Display_Peripheral_System_State(dev_label, "Serial", num_states);
    }
#endif

#ifdef DISPLAY_LABEL
    {
        DV_DEV_LABEL dev_label = {DISPLAY_LABEL};

        Display_Peripheral_System_State(dev_label, "Display", num_states);
    }
#endif

#ifdef ETHERNET_LABEL
    {
        DV_DEV_LABEL dev_label = {ETHERNET_LABEL};

        Display_Peripheral_System_State(dev_label, "Ethernet", num_states);
    }
#endif

#ifdef TOUCHPANEL_LABEL
    {
        DV_DEV_LABEL dev_label = {TOUCHPANEL_LABEL};

        Display_Peripheral_System_State(dev_label, "Touchpanel", num_states);
    }
#endif

#ifdef I2C_LABEL
    {
        DV_DEV_LABEL dev_label = {I2C_LABEL};

        Display_Peripheral_System_State(dev_label, "I2C", num_states);
    }
#endif

#ifdef SPI_LABEL
    {
        DV_DEV_LABEL dev_label = {SPI_LABEL};

        Display_Peripheral_System_State(dev_label, "SPI", num_states);
    }
#endif

#ifdef CAN_LABEL
    {
        DV_DEV_LABEL dev_label = {CAN_LABEL};

        Display_Peripheral_System_State(dev_label, "CAN", num_states);
    }
#endif

#ifdef SDIO_HCD_LABEL
    {
        DV_DEV_LABEL dev_label = {SDIO_HCD_LABEL};

        Display_Peripheral_System_State(dev_label, "SDIOH", num_states);
    }
#endif

    printf("%s\r\n", border_buffer);
    printf("(*) Current System Power State\n");
    printf("\r\n");
    printf("-- Enter 'ss -create <ss count>' command to initialize System Power States\r\n");
    printf("-- Enter 'ss <ss index>' command to change System Power State\r\n");
    printf("-- Enter 'ss [-map|-unmap] <device name> <ss index> <ps index>' command\r\n");
    printf("   to map/unmap Peripheral states to System Power states\r\n");
    printf("\r\n");
}

/***********************************************************************
*
*   FUNCTION
*
*       Switch_System_State
*
*   DESCRIPTION
*
*       This function switches system state.
*
***********************************************************************/
VOID Switch_System_State(UINT8 state_num)
{
    STATUS status;

    status = NU_PM_Set_System_State(state_num);

    if (status == NU_SUCCESS)
    {
        printf("\r\n");
        printf("*** Successful transition to System States %d\r\n", state_num);
        printf("\r\n");
    }
    else
    {
        printf("\r\n");
        printf("*** Failed to switch to System States %d\r\n", state_num);
        printf("\r\n");
    }
}

/***********************************************************************
*
*   FUNCTION
*
*       Create_System_States
*
*   DESCRIPTION
*
*       This function initializes system states.
*
***********************************************************************/
VOID Create_System_States(UINT8 state_num)
{
    STATUS status;

    status = NU_PM_System_State_Init(state_num);

    if (status == NU_SUCCESS)
    {
        printf("\r\n");
        printf("*** Successful initialization of System States %d\r\n", state_num);
        printf("\r\n");
    }
    else
    {
        printf("\r\n");
        printf("*** Failed to initialize System States; number of states %d\r\n", state_num);
        printf("\r\n");
    }
}

/***********************************************************************
*
*   FUNCTION
*
*       Map_Peripheral
*
*   DESCRIPTION
*
*       This function maps a peripheral's state to system's state.
*
***********************************************************************/
VOID Map_Peripheral(CHAR dev_name[15], UINT8 sys_state, UINT8 periph_state)
{
    STATUS    status;
    DV_DEV_ID dev_id;

    status = Get_Device_ID(dev_name, &dev_id);

    /* We expect to find only one device enabled. */
    if(status == NU_SUCCESS)
    {
        status = NU_PM_Map_System_Power_State(sys_state, dev_id, periph_state);

        if (status == NU_SUCCESS)
        {
            printf("\r\n");
            printf("*** Successful mapping of %s device's %d state to System State %d\r\n", dev_name, periph_state, sys_state);
            printf("\r\n");
        }
        else
        {
            printf("\r\n");
            printf("*** Failed to map %s's state %d to System States %d\r\n", dev_name, periph_state, sys_state);
            printf("\r\n");
        }
    }
    else if (status == INVALID_DEV_NAME)
    {
        printf("\r\n");
        printf("*** Invalid Device name %s\r\n", dev_name);
        printf("\r\n");
    }
    else
    {
        printf("\r\n");
        printf("*** Failed to get Device ID for %s\r\n", dev_name);
        printf("\r\n");
    }
}

/***********************************************************************
*
*   FUNCTION
*
*       Unmap_Peripheral
*
*   DESCRIPTION
*
*       This function unmaps the given peripheral from system states.
*
***********************************************************************/
VOID Unmap_Peripheral(CHAR dev_name[15])
{
    STATUS    status;
    DV_DEV_ID dev_id;

    status = Get_Device_ID(dev_name, &dev_id);

    /* We expect to find only one device enabled. */
    if(status == NU_SUCCESS)
    {
        status = NU_PM_Unmap_System_Power_State(dev_id);

        if (status == NU_SUCCESS)
        {
            printf("\r\n");
            printf("*** Successful unmapping of %s device\r\n", dev_name);
            printf("\r\n");
        }
        else
        {
            printf("\r\n");
            printf("*** Failed to unmap %s device\r\n", dev_name);
            printf("\r\n");
        }
    }
    else if (status == INVALID_DEV_NAME)
    {
        printf("\r\n");
        printf("*** Invalid Device name %s\r\n", dev_name);
        printf("\r\n");
    }
    else
    {
        printf("\r\n");
        printf("*** Failed to get Device ID for %s\r\n", dev_name);
        printf("\r\n");
    }

}

#endif /* (CFG_NU_OS_SVCS_PWR_CORE_ENABLE_SYSTEM == NU_TRUE) */
