/***********************************
* Include Files
***********************************/
#include                <stdio.h>
#include                "nucleus.h"
#include                "kernel/nu_kernel.h"
#include                "power_demo.h"

/***********************************
* Global Variables
***********************************/
static NU_TASK          Task_Control_Block;
CHAR                    Command[MAX_COMMAND_LENGTH];

/***********************************
* Function Prototypes
***********************************/

/* Static local functions. */
static VOID             Serial_Console_Task(UNSIGNED argc, VOID *argv);
static VOID             Process_Command(VOID);

/* External functions. */
extern VOID             Display_OP_Table(VOID);
extern VOID             Switch_OP(UINT8);
extern VOID             Show_Tick_Suppression_Status(VOID);
extern VOID             Enable_Tick_Suppression(VOID);
extern VOID             Disable_Tick_Suppression(VOID);
extern VOID             Display_Peripheral_State_Table(VOID);
extern VOID             Switch_Peripheral_State(CHAR dev_name[15], UINT8 state_num);
extern VOID             Display_System_States_Table(VOID);
extern VOID             Switch_System_State(UINT8 state_num);
extern VOID             Create_System_States(UINT8 state_num);
extern VOID             Map_Peripheral(CHAR dev_name[15], UINT8 sys_state, UINT8 periph_state);
extern VOID             Unmap_Peripheral(CHAR dev_name[15]);

/***********************************************************************
*
*   FUNCTION
*
*       Application_Initialize
*
*   DESCRIPTION
*
*       Demo application entry point - initializes the
*       demonstration application by creating and initializing necessary
*       tasks, memory pools, and communication components.
*
***********************************************************************/
VOID    Application_Initialize(NU_MEMORY_POOL* mem_pool, NU_MEMORY_POOL* uncached_mem_pool)
{
    VOID *         pointer;
    STATUS         status;
    
    /* Reference unused parameters to avoid toolset warnings */
    NU_UNUSED_PARAM(uncached_mem_pool);

    /* Allocate memory for task 0 */
    status = NU_Allocate_Memory(mem_pool, &pointer,
                                CONSOLE_TASK_STACK_SIZE, NU_NO_SUSPEND);

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Create task 0.  */
        status = NU_Create_Task(&Task_Control_Block, "TASK 0", Serial_Console_Task, 0, NU_NULL, pointer,
                                CONSOLE_TASK_STACK_SIZE, CONSOLE_TASK_PRIORITY, CONSOLE_TASK_TIMESLICE,
                                NU_PREEMPT, NU_START);
    }

    if (status != NU_SUCCESS)
    {
        printf("*** Error creating task, Error Code: %d\n", status);

        /* De-allocate memory. */
        (VOID)NU_Deallocate_Memory(pointer);

        /* Error trap. */
        for (;;);
    }
}


/***********************************************************************
*
*   FUNCTION
*
*       Serial_Console_Task
*
*   DESCRIPTION
*
*       This task implements serial console. It reads commands input
*       from serial and passes the command to command parser.
*
***********************************************************************/
static VOID    Serial_Console_Task(UNSIGNED argc, VOID *argv)
{
    CHAR   ch;
    UINT8  i;
    UINT8  bckspace_sequence[] = {BS, 0x020, BS, 0}; /* Used for Serial 0x020 is the char ' ' */

    /* Reference all parameters to ensure no toolset warnings */
    NU_UNUSED_PARAM(argc);
    NU_UNUSED_PARAM(argv);

    printf("************************************************************\r\n");
    printf("                Nucleus Power Demonstration\r\n");
    printf("************************************************************\r\n");

    printf("-- Use the following commands to control System Power.\r\n");
    printf("   op - Operating Point\r\n");
    printf("   ps - Peripheral Power State\r\n");
    printf("   ss - System Power State\r\n");
    printf("   ts - Tick Suppression\r\n");
    printf("   h/help/? - Print this help message\r\n");
    printf("\r\n");

    /* Continue this process forever.  */
    while(1)
    {
        printf("DEMO> ");

        i = 0;
        while(1) 
        {
            ch = getchar();

            if ((ch == '\r') || (ch == '\n'))
            {
                printf("\r\n");
                break;
            }

            else if (ch == BS)
            {
                if (i > 0)
                {
                    i--;
                    printf("%s", bckspace_sequence);
                }

                continue;
            }

            if (i >= MAX_COMMAND_LENGTH)
            {
                continue;
            }

            Command[i++] = ch;
            putchar(ch);
        }

        Command[i] = '\0';

        /* Process command. */
        Process_Command();
    }
}

/***********************************************************************
*
*   FUNCTION
*
*       Process_Command
*
*   DESCRIPTION
*
*       This function processes user command stored in global command
*       buffer.
*
***********************************************************************/
static VOID Process_Command()
{
    switch(Command[0])
    {
    case 'o':

        if (Command[1] == 'p')
        {
#if (CFG_NU_OS_SVCS_PWR_CORE_ENABLE_DVFS == NU_TRUE)
            if (Command[2] == '\0')
            {
                Display_OP_Table();
            }
            else if (Command[2] == ' ' && Command[3] >= '0' && Command[3] <= '9' && Command[4] == '\0')
            {
                Switch_OP(Command[3] - '0');
            }
            else
            {
                printf("\r\n");
                printf("*** Invalid OP Command\r\n");
                printf("\r\n");
            }
#else
            printf("\r\n");
            printf("*** Please enable DVFS support in configuration to use this service\r\n");
            printf("\r\n");
#endif
        }
        else
        {
            printf("\r\n");
            printf("*** Unknown Command\r\n");
            printf("\r\n");
        }

        break;

    case 'p':

        if (Command[1] == 's')
        {
#if (CFG_NU_OS_SVCS_PWR_CORE_ENABLE_PERIPHERAL == NU_TRUE)
            if (Command[2] == '\0')
            {
                Display_Peripheral_State_Table();
            }
            else
            {
                CHAR dev_name[15] = "\0";
                INT i;

                /* Extract peripheral name and required state. */
                for (i = 3; i < MAX_COMMAND_LENGTH; ++i)
                {
                    if (Command[i] == ' ')
                    {
                        break;
                    }
                }

                strncpy(dev_name, &Command[3], i - 3);

                /* Switch power state of specified peripheral. */
                Switch_Peripheral_State(dev_name, Command[i+1] - '0');
            }
#else
            printf("\r\n");
            printf("*** Please enable Peripheral States support in configuration to use this service\r\n");
            printf("\r\n");
#endif
        }
        else
        {
            printf("\r\n");
            printf("*** Unknown Command\r\n");
            printf("\r\n");
        }

        break;

    case 's':

        if (Command[1] == 's')
        {
#if (CFG_NU_OS_SVCS_PWR_CORE_ENABLE_SYSTEM == NU_TRUE)
            if (Command[2] == '\0')
            {
                Display_System_States_Table();
            }
            else if (Command[2] == ' ' && Command[3] >= '0' && Command[3] <= '9' && Command[4] == '\0')
            {
                Switch_System_State(Command[3] - '0');
            }
            else if (Command[2] == ' ' && Command[3] == '-')
            {
                CHAR map_cmd[15] = "\0";
                INT i;

                /* Extract peripheral name and required state. */
                for (i = 4; i < MAX_COMMAND_LENGTH; ++i)
                {
                    if (Command[i] == ' ')
                    {
                        break;
                    }
                }

                strncpy(map_cmd, &Command[4], i - 4);

                if (strcmp(map_cmd, "create") == 0)
                {
                    /* Call create function. */
                    Create_System_States(Command[11] - '0');
                }
                else if (strcmp(map_cmd, "map") == 0)
                {
                    CHAR dev_name[15] = "\0";

                    /* Extract peripheral name and required state. */
                    for (i = 8; i < MAX_COMMAND_LENGTH; ++i)
                    {
                        if (Command[i] == ' ')
                        {
                            break;
                        }
                    }

                    strncpy(dev_name, &Command[8], i - 8);

                    /* Extract states */
                    for (i = i + 1; i < MAX_COMMAND_LENGTH; i += 4)
                    {
                        if ((Command[i] >= '0' && Command[i] <= '9') &&
                            (Command[i+2] >= '0' && Command[i+2] <= '9'))
                        {
                            /* Call map function */
                            Map_Peripheral(dev_name, Command[i] - '0', Command[i+2] - '0');

                            if (Command[i+3] == '\0')
                            {
                                break;
                            }
                        }
                        else
                        {
                            break;
                        }
                    }
                }
                else if (strcmp(map_cmd, "unmap") == 0)
                {
                    /* Call unmap function on the device. */
                    Unmap_Peripheral(&Command[10]);
                }
                else
                {
                    printf("\r\n");
                    printf("*** Invalid SS Command, please specify map or unmap\r\n");
                    printf("\r\n");
                }
            }
            else
            {
                printf("\r\n");
                printf("*** Invalid SS Command\r\n");
                printf("\r\n");
            }
#else
            printf("\r\n");
            printf("*** Please enable System States support in configuration to use this service\r\n");
            printf("\r\n");
#endif
        }
        else
        {
            printf("\r\n");
            printf("*** Unknown Command\r\n");
            printf("\r\n");
        }

        break;

    case 't':

        if (Command[1] == 's')
        {
#if (CFG_NU_OS_KERN_PLUS_CORE_TICK_SUPPRESSION == NU_TRUE)
            if (Command[2] == '\0')
            {
                Show_Tick_Suppression_Status();
            }
            else if (Command[2] == ' ' && Command[3] == '0' && Command[4] == '\0')
            {
                Disable_Tick_Suppression();
            }
            else if (Command[2] == ' ' && Command[3] == '1' && Command[4] == '\0')
            {
                Enable_Tick_Suppression();
            }
            else
            {
                printf("\r\n");
                printf("*** Invalid TS Command\r\n");
                printf("\r\n");
            }
#else
            printf("\r\n");
            printf("*** Please enable Tick Suppression support in configuration to use this service\r\n");
            printf("\r\n");
#endif
        }
        else
        {
            printf("\r\n");
            printf("*** Unknown Command\r\n");
            printf("\r\n");
        }

        break;

    case 'h':
    case '?':

        if ((Command[1] == '\0') || (strcmp(Command, "help") == 0))
        {
            printf("\r\n");
            printf("-- Use the following commands to control System Power.\r\n");
            printf("   op - Operating Point\r\n");
            printf("   ps - Peripheral Power State\r\n");
            printf("   ss - System Power State\r\n");
            printf("   ts - Tick Suppression\r\n");
            printf("   h/help/? - Print this help message\r\n");
            printf("\r\n");
        }
        else
        {
            printf("\r\n");
            printf("*** Unknown Command\r\n");
            printf("\r\n");
        }

        break;

    default:
        printf("\r\n");
        printf("*** Unknown Command\r\n");
        printf("\r\n");
    }
}
