#include "net_server_power_demo.h"

NU_TIMER    Timer;
UINT32      last_idle_time, last_total_time;
UINT32      CPU_Idle_Percentage, CPU_Utilization_Percentage;

static UINT8  printFlag = NU_FALSE;

/***********************************************************************
*
*   FUNCTION
*
*       Turn_ON_Device
*
*   DESCRIPTION
*
*       DM callback to turn ON a peripheral device.
*
***********************************************************************/
STATUS Turn_ON_Device(DV_DEV_ID device, VOID* context)
{
    STATUS status;

    status = NU_PM_Set_Power_State (device, POWER_ON_STATE);

    return status;
}

/***********************************************************************
*
*   FUNCTION
*
*       Turn_OFF_Device
*
*   DESCRIPTION
*
*       DM callback to turn OFF a peripheral device.
*
***********************************************************************/
STATUS Turn_OFF_Device(DV_DEV_ID device, VOID* context)
{
    STATUS status;

    status = NU_PM_Set_Power_State (device, POWER_OFF_STATE);

    return status;
}

/***********************************************************************
*
*   FUNCTION
*
*       Enable_Device_Power_State
*
*   DESCRIPTION
*
*       Receives a list of peripheral device lables and enables their
*       peripheral power state.
*
***********************************************************************/
STATUS Enable_Device_Power_State(DV_DEV_LABEL dev_label[])
{
    STATUS  status;
    DV_LISTENER_HANDLE  listener_handle;

    status = DVC_Reg_Change_Notify(dev_label, 1, &Turn_ON_Device,
    &Turn_OFF_Device, NU_NULL, &listener_handle);

    return status;
}

/***********************************************************************
*
*   FUNCTION
*
*       Update_CPU_Utilization
*
*   DESCRIPTION
*
*       This function updates the CPU utilization.
*
***********************************************************************/
static VOID    Update_CPU_Utilization(UNSIGNED argc)
{
    UINT32  total_time, idle_time;

    NU_PM_Get_CPU_Counters(&total_time, &idle_time);

    CPU_Idle_Percentage = 100 * (idle_time - last_idle_time)/(total_time - last_total_time);
    CPU_Utilization_Percentage = 100 - CPU_Idle_Percentage;

    last_idle_time = idle_time;
    last_total_time = total_time;
}

/***********************************************************************
*
*   FUNCTION
*
*       System_Power_Mgmt_Init
*
*   DESCRIPTION
*
*       The function initializes the the system level power management
*       functions.
*
***********************************************************************/
STATUS  System_Power_Mgmt_Init(DEMO_PARAMETERS* p_demo_params)
{
    INT             dev_id_cnt = 1;
    DV_DEV_ID       dev_id_list[1];
    STATUS          status = NU_SUCCESS;

    /* List of peripheral labels the application will be using */
    DV_DEV_LABEL        serial_label[] = {{SERIAL_LABEL}};
    DV_DEV_LABEL        ethernet_label[] = {{ETHERNET_LABEL}};

    CPU_Idle_Percentage = CPU_Utilization_Percentage = 0;

    /* Start an 100 Hz application timer to track CPU utilization */
    NU_Create_Timer(&Timer, "TIMER", Update_CPU_Utilization, 0, 5, 5, NU_ENABLE_TIMER);

    /* For this demo application we will create two system states
     * System State 0 (SS0) - Serial ON, Ethernet ON
     * System State 1 (SS1) - Serial OFF, Ethernet ON */

    /* Initialize number of system states */
    status = NU_PM_System_State_Init(NUM_SYS_STATES);

    /* Create System State 0 (SS0) and map peripherals and their states */
    if(status == NU_SUCCESS)
        status = DVC_Dev_ID_Get(serial_label, 1, dev_id_list, &dev_id_cnt);

    if(status == NU_SUCCESS)
        status = NU_PM_Map_System_Power_State(SYS_STATE_0, dev_id_list[0], STATE_ON);

    if(status == NU_SUCCESS)
        status = DVC_Dev_ID_Get(ethernet_label, 1, dev_id_list, &dev_id_cnt);

    if(status == NU_SUCCESS)
        status = NU_PM_Map_System_Power_State(SYS_STATE_0, dev_id_list[0], STATE_ON);

    /* Create System State 0 (SS0) and map peripherals and their states */
    if(status == NU_SUCCESS)
        status = DVC_Dev_ID_Get(serial_label, 1, dev_id_list, &dev_id_cnt);

    if(status == NU_SUCCESS)
        status = NU_PM_Map_System_Power_State(SYS_STATE_1, dev_id_list[0], STATE_OFF);

    if(status == NU_SUCCESS)
        status = DVC_Dev_ID_Get(ethernet_label, 1, dev_id_list, &dev_id_cnt);

    if(status == NU_SUCCESS)
        status = NU_PM_Map_System_Power_State(SYS_STATE_1, dev_id_list[0], STATE_ON);

    /* Identify and display the supported operating points */
    status = NU_PM_Get_OP_Count(&p_demo_params->num_op_points);

    /* Enable serial msgs to user */
    printFlag = NU_TRUE;

    return status;
}

/***********************************************************************
*
*   FUNCTION
*
*       Process_Request
*
*   DESCRIPTION
*
*       The function processes the client sides requests.
*
***********************************************************************/
STATUS  Process_Request(CHAR* state_string, DEMO_PARAMETERS* p_demo_params)
{
    STATUS status = NU_SUCCESS;
    CHAR*  p_char = state_string;
    UINT8  freq_id, volt_id;
    UINT32 curr_cpu_freq;

    if(strcmp(state_string,"/ss0") == 0)
    {
        /* Switch to system state 0 */
        status = NU_PM_Set_System_State(SYS_STATE_0);
        printFlag = NU_TRUE;

        if(printFlag)
        {
            printf("\r\n***********************************\r\n");
            printf("\r\nSuccessfully switched System State!\r\n");
            printf("\r\n***********************************\r\n");
        }
    }
    else if(strcmp(state_string,"/ss1") == 0)
    {
        /* Cannot print if we are already in SS1. Conditional for this purpose */
        if(printFlag)
        {
            printf("\r\n***************************************************************\r\n");
            printf("\r\nSwitching to System State 1 (Serial OFF, Ethernet ON)\r\n");
            printf("You will have to switch to SS0 to enable serial I/O\r\n");
            printf("Disabling serial device for System State 1 ... Bye!\r\n");
            printf("\r\n***************************************************************\r\n");

            /* Let the warning banner print */
            NU_Sleep(50);
        }

        /* Switch to System state 1 */
        status = NU_PM_Set_System_State(SYS_STATE_1);
        printFlag = NU_FALSE;
    }
    else if(strncmp(state_string,"/op",3) == 0)
    {
        /* Switch operating points */
        p_char += 3;
        status = NU_PM_Set_Current_OP(atoi(p_char));

        if(status == PM_REQUEST_BELOW_MIN)
        {
            if(printFlag)
            {
                printf("\r\n***************************************************************************\r\n");
                printf("\r\nCannot switch! requested operating point below minimum DVFS Operating Point\r\n");
                printf("\r\n***************************************************************************\r\n");
            }
        }
        else
        {
            if(printFlag)
            {
                printf("\r\n***************************************\r\n");
                printf("\r\nSuccessfully switched Operating Points!\r\n");
                printf("\r\n***************************************\r\n");
            }
        }
    }
    else
    {
        /* Update CPU utilization */
        p_demo_params->cpu_util = CPU_Utilization_Percentage;

        /* Get current operating point */
        NU_PM_Get_Current_OP(&p_demo_params->curr_op);

        /* Get current system state */
        NU_PM_Get_System_State(&p_demo_params->curr_ss);

        /* Get the voltage and frequency IDs for this operating point */
        NU_PM_Get_OP_VF(p_demo_params->curr_op, &freq_id, &volt_id);

        /* Get the frequency for the frequency ID */
        NU_PM_Get_Freq_Info(freq_id, &curr_cpu_freq);

        return status;
    }

    /* Update CPU utilization */
    p_demo_params->cpu_util = CPU_Utilization_Percentage;

    /* Get current operating point */
    NU_PM_Get_Current_OP(&p_demo_params->curr_op);

    /* Get current system state */
    NU_PM_Get_System_State(&p_demo_params->curr_ss);

    /* Get the voltage and frequency IDs for this operating point */
    NU_PM_Get_OP_VF(p_demo_params->curr_op, &freq_id, &volt_id);

    /* Get the frequency for the frequency ID */
    NU_PM_Get_Freq_Info(freq_id, &curr_cpu_freq);


    /* Print user message */
    if(printFlag)
    {
        printf("\r\n***********************************************\r\n");
        printf("Current Operating Point ID = %d \r\n", p_demo_params->curr_op);
        printf("Current System State ID = %d \r\n", p_demo_params->curr_ss);
        printf("Current CPU frequency = %d Hz \r\n", (INT)curr_cpu_freq);
        printf("Current CPU Utilization = %d %% \r\n", (INT)p_demo_params->cpu_util);
        printf("\r\n***********************************************\r\n");
    }

    return status;
}
