/* Include files */
#include "net_server_power_demo.h"

/* Internal globals */
static  NU_TASK Simple_Server_CB;

/* Function prototype declarations */
static  VOID   Simple_Server_Task(UNSIGNED argc, VOID *argv);
static  VOID   Get_IP_Address(CHAR *buf);
static  STATUS Get_Foreign_IP_Addr_String_By_Socket(INT socketd, CHAR *buf);
static  STATUS Get_Foreign_IP_Addr_By_Socket(INT socketd, UINT8 *buf);

/**********************************************/
/* Net server power demo function definitions */
/**********************************************/

/*************************************************************************
*
*   FUNCTION
*
*       Application_Initialize
*
*   DESCRIPTION
*
*       Initializes and starts the application.
*
*************************************************************************/
VOID Application_Initialize (NU_MEMORY_POOL* mem_pool,
                             NU_MEMORY_POOL* uncached_mem_pool)
{
    VOID   *pointer;
    STATUS status;
	/* List of peripheral labels the application will be using */
    DV_DEV_LABEL        serial_label[] = {{SERIAL_LABEL}};
    DV_DEV_LABEL        ethernet_label[] = {{ETHERNET_LABEL}};

    UNUSED_PARAMETER(uncached_mem_pool);

    /****************************/
    /* Create application tasks */
    /****************************/

    /* Allocate memory for the Net Sample1 Task. */
    status = NU_Allocate_Memory(mem_pool, &pointer, TASK_STACK_SIZE, NU_NO_SUSPEND);

    /* Create the Net Sample1 Task. */
    if (status == NU_SUCCESS)
    {
        /* Create the Net Sample1 task.  */
        status = NU_Create_Task(&Simple_Server_CB, "NSAMTSK", Simple_Server_Task, 0, NU_NULL, pointer,
                                TASK_STACK_SIZE, TASK_PRIORITY, TASK_TIMESLICE,
                                NU_PREEMPT, NU_START);

        /* On error, deallocate memory. */
        if(status != NU_SUCCESS)
        {
            (VOID)NU_Deallocate_Memory(pointer);
        }
    }

    /******************************************************************/
    /* Enable peripheral devices that will be used by the application */
    /******************************************************************/

    /* When Nucleus power services are enabled all the devices
     * present in the system are in OFF power state by default.
     * The peripheral devices used by the application are turned
     * ON below.*/

    /* Initialize the power state of devices required by the application */
    status = Enable_Device_Power_State(serial_label);
    status = Enable_Device_Power_State(ethernet_label);
}

/*************************************************************************
*
*   FUNCTION
*
*       Simple_Server_Task
*
*   DESCRIPTION
*
*       Open a socket connection, bind the socket with the server address,
*       listen, and accept connections.  When connected call the routine
*       to perform request response processing.
*
*************************************************************************/
static VOID Simple_Server_Task(UNSIGNED argc, VOID *argv)
{
    STATUS              status;
    INT                 retcode;
    INT                 socketd, newsock;    
    struct addr_struct  servaddr;            
    struct addr_struct  client_addr;
    CHAR                command[FMT_SIZE];
    CHAR                resource[FMT_SIZE];
    CHAR                fmt_str[FMT_SIZE];
    NU_MEMORY_POOL*     sys_pool_ptr;
    DEMO_PARAMETERS*    p_demo_params;
    CHAR*               p_buffer;
    CHAR*               p_aux_buff;
    CHAR*               pChar1;
    CHAR*               pChar2;
    INT                 i,n,m;
	
	/* Reference unused parameters to avoid toolset warnings. */
    UNUSED_PARAMETER(argc);
    UNUSED_PARAMETER(argv);

    /* Get system memory pool pointer */
    status = NU_System_Memory_Get(&sys_pool_ptr, NU_NULL);

    /* Allocate memory for demo parameters data structure */
    status = NU_Allocate_Memory(sys_pool_ptr, (void**)&p_demo_params, sizeof(DEMO_PARAMETERS), NU_NO_SUSPEND);

    /* Initialize system power management */
    System_Power_Mgmt_Init(p_demo_params);

    /* Wait until the NET stack is initialized. */
    status = NETBOOT_Wait_For_Network_Up(NU_SUSPEND);

    /* Net stack is UP now get target IP and open a server side socket */
    if (status == NU_SUCCESS)
    {
        /* Get the local IP address. */
        Get_IP_Address(p_demo_params->target_ip);

        /* Print user message */
        printf("Open server IP address http://%s:8080/ in your web browser:\r\n", p_demo_params->target_ip);

        /* Allocate memory for demo HTML landing page */
        status = NU_Allocate_Memory(sys_pool_ptr, (void**)&p_buffer, BUF_SIZE, NU_NO_SUSPEND);

        /* Allocate memory for demo HTML landing page's operating list content */
        status = NU_Allocate_Memory(sys_pool_ptr, (void**)&p_aux_buff, AUX_BUF_SIZE, NU_NO_SUSPEND);

        /* Format and save system state URLs */
        sprintf(p_demo_params->ss0url,"ss0");
        sprintf(p_demo_params->ss1url,"ss1");

        /* Initialize locals */
        pChar1 = p_demo_params->opurls;
        pChar2 = p_aux_buff;

        /* Iterate and construct the platform specific dynamic content in the HTML page */
        for(i = 0, pChar1 = p_demo_params->opurls, pChar2 = p_aux_buff;
                i < p_demo_params->num_op_points; i++)
        {
           /* Construct operating points urls */
           n=sprintf(pChar1,"%s%d","op",i);

           /* Construct operating points list */
           m = sprintf(pChar2, OP_POINTS, pChar1, i);

           /* Update operating pointers */
           pChar1 += (n+1);
           pChar2 += m;
        }

        /* Open a connection via the socket interface. */
        socketd = NU_Socket(NU_FAMILY_IP, NU_TYPE_STREAM, 0);

        if (socketd >=0 )
        {
            /* Fill in a structure with the server address. */
            servaddr.family    = NU_FAMILY_IP;
            servaddr.port      = PORT_NUM;
            *(UINT32 *)servaddr.id.is_ip_addrs = IP_ADDR_ANY;
            servaddr.name       = "SAMP1";

            /* Bind the server's address. */
            if (NU_Bind(socketd, &servaddr, 0) >= 0)
            {
                /* Prepare to accept connection requests. */
                status = NU_Listen(socketd, 10);

                if (status == NU_SUCCESS)
                {
                    for (;;)
                    {
                        /* Block in NU_Accept until a client attempts connection. */
                        newsock = NU_Accept(socketd, &client_addr, 0);

                        if (newsock >= 0)
                        {
                            /* Obtain host IP addresses */
                            Get_Foreign_IP_Addr_String_By_Socket(newsock, p_demo_params->host_ip);

                            /**************************************/
                            /* Receive and process client request */
                            /**************************************/

                            /* Receive a GET request from the Web browser */
                            retcode = NU_Recv(newsock, p_buffer, 1024-1, 0);

                            if (retcode < 0)
                            {
                                printf("Receive failed\r\n");
                                status = STATUS_FAILURE;
                            }
                            else
                            {
                                /* Assure the input buffer is null terminated. */
                                p_buffer[retcode] = 0;
                            }

                            /* Parse the command and resource identifier from the input. */
                            sprintf(fmt_str, "%%%ds %%%ds \r\n", FMT_SIZE, FMT_SIZE);
                            sscanf(p_buffer, fmt_str, command, resource);

                            /* Process the client request */
                            Process_Request(resource, p_demo_params);

                            /* Populate the dynamic fields in the demo HTML page */
                            sprintf(p_buffer, HTML_PAGE, p_demo_params->host_ip, p_demo_params->target_ip,
                            (INT)p_demo_params->cpu_util, (INT)p_demo_params->curr_op, (INT)p_demo_params->curr_ss,
                            p_demo_params->ss0url,  p_demo_params->ss1url,
                            p_demo_params->num_op_points, p_aux_buff);

                            /* Transmit response to client */
                            retcode = NU_Send(newsock, p_buffer, strlen(p_buffer), 0);

                            /* Close socket */
                            NU_Close_Socket(newsock);

                        } /* End successful NU_Accept. */
                        else if (newsock == NU_NOT_CONNECTED)
                        {
                            break;
                        }
                    }

                }
                else
                {
                    /* Sleep for a while if the NU_Listen call failed. */
                    printf("NU_Listen() failed.\r\n");
                }

            }
            else
            {
                /* Sleep for a while if the NU_Bind call failed. */
                printf("NU_Bind() failed.\r\n");
            }

        }
        else
        {
            printf("NU_Socket() failed.\r\n");
        }

    }
    else
    {
        printf("NETBOOT_Wait_For_Network_Up() failed.\r\n");
    }
}

/**************************************/
/* Local utility function definitions */
/**************************************/

/*************************************************************************
*
*   FUNCTION
*
*       Get_Foreign_IP_Addr_String_By_Socket
*
*   DESCRIPTION
*
*       This function gets the foreign IP address of the given socket
*       and converts it into an ASCII string in the given buffer.
*
*************************************************************************/
static STATUS Get_Foreign_IP_Addr_String_By_Socket(INT socketd, CHAR *buf)
{
    STATUS status;
    UINT8  ipaddr[MAX_ADDRESS_SIZE];


    /* Get the foreign IP address for the socket. */
    status = Get_Foreign_IP_Addr_By_Socket(socketd, &ipaddr[0]);

    /* Convert the foreign ip addr to ASCII */
    if ( status == NU_SUCCESS)
    {
        status = NU_Inet_NTOP(NU_FAMILY_IP, &ipaddr[0], buf, 16);
    }

    return (status);
}


/*************************************************************************
*
*   FUNCTION
*
*       Get_Foreign_IP_Addr_By_Socket
*
*   DESCRIPTION
*
*       This function gets the foreign IP address of the given socket.
*
*************************************************************************/
static STATUS Get_Foreign_IP_Addr_By_Socket(INT socketd, UINT8 *addrp)
{
    STATUS  status;
    INT16   addrLength;
    struct  sockaddr_struct peer;


    addrLength = sizeof(struct sockaddr_struct);

    /* Get the client's address info. */
    status = NU_Get_Peer_Name(socketd, &peer, &addrLength);

    if (status == NU_SUCCESS)
    {
        memcpy(addrp, &peer.ip_num, MAX_ADDRESS_SIZE);
    }

    return (status);
}

/*************************************************************************
*
*   FUNCTION
*
*      Get_IP_Address
*
*   DESCRIPTION
*
*      This function returns the currently configured IP address of the
*      wired Ethernet interface. It calls NET APIs to determine the IP.
*
**************************************************************************/
VOID Get_IP_Address(CHAR *ip_str)
{
    NU_IOCTL_OPTION ioctl_opt;
    CHAR if_name[32];
    STATUS      status = STATUS_FAILURE;

    strncpy(if_name, DEMO_IFACE_NAME, sizeof(DEMO_IFACE_NAME));

    /* Initialize the IOCTL option to zero. */
    memset(&ioctl_opt, 0, sizeof(NU_IOCTL_OPTION));

    /* Find the IP address attached to the network device. */
    ioctl_opt.s_optval = (UINT8*)if_name;

    /* Call NU_Ioctl to get the IP address. */
    status = NU_Ioctl(SIOCGIFADDR, &ioctl_opt, sizeof(ioctl_opt));

    /* Check if we got the IP */
    if (status == NU_SUCCESS)
    {
        /* Create IP Address string */
        sprintf(ip_str, "%d.%d.%d.%d",
                ioctl_opt.s_ret.s_ipaddr[0],
                ioctl_opt.s_ret.s_ipaddr[1],
                ioctl_opt.s_ret.s_ipaddr[2],
                ioctl_opt.s_ret.s_ipaddr[3]);
    }
    else
    {
        /* Format the string */
        sprintf(ip_str, "%.*s",
                (strlen("Not assigned") + 1),
                "Not assigned");
    }

} /* Get_IP_Address */
