/***********************************
* Include Files
***********************************/
#include                <stdio.h>
#include                "nucleus.h"
#include                "kernel/nu_kernel.h"
#include                "services/nu_services.h"
#include                "drivers/nu_drivers.h"

/***********************************
* Configuration Setting
***********************************/

/* Define task stack size */
#define         DEMO_STACK_SIZE         (NU_MIN_STACK_SIZE * 4)

/* Define demo queue size (number of unsigned longs) */
#define         DEMO_QUEUE_SIZE         100

/* Define each demo task's priority */
#define         TASK0_PRIORITY          26
#define         TASK1_PRIORITY          29
#define         TASK2_PRIORITY          29
#define         TASK3_PRIORITY          29
#define         TASK4_PRIORITY          29
#define         TASK5_PRIORITY          30
#define         TASK6_PRIORITY          31

/* Define each demo task's time slice */
#define         TASK0_TIMESLICE         20
#define         TASK1_TIMESLICE         5
#define         TASK2_TIMESLICE         5
#define         TASK3_TIMESLICE         0
#define         TASK4_TIMESLICE         0
#define         TASK5_TIMESLICE         0
#define         TASK6_TIMESLICE         1

/***********************************
* Global Variables
***********************************/
static NU_TASK          Task_Control_Block[6];
static NU_TIMER         MyTimer;
static NU_TASK *        Task_3_4_Resource_Owner;
static NU_QUEUE         Queue_0;
static NU_SEMAPHORE     Semaphore_0;
static NU_EVENT_GROUP   Event_Group_0;
static UINT             Task_0_Time;
static UINT             Task_1_Messages_Sent;
static UINT             Task_2_Messages_Received;
static UINT             Task_2_Invalid_Messages;
static UINT             Task_5_Event_Detections;
static INT              Usage_Pct = 0;
static UINT8            Current_Operating_Point;
static UINT32           Frequency;

/***********************************
* Function Prototypes
***********************************/
static VOID             MyTimer_Entry(UNSIGNED argc);
static VOID             Task_0_Entry(UNSIGNED argc, VOID *argv);
static VOID             Task_1_Entry(UNSIGNED argc, VOID *argv);
static VOID             Task_2_Entry(UNSIGNED argc, VOID *argv);
static VOID             Task_3_and_4_Entry(UNSIGNED argc, VOID *argv);
static VOID             Task_5_Entry(UNSIGNED argc, VOID *argv);
static VOID             Print_Log(VOID);
static STATUS           Turn_ON_Device(DV_DEV_ID device, VOID* context);
static STATUS           Turn_OFF_Device(DV_DEV_ID device, VOID* context);
static STATUS           Enable_Device_Power_State(VOID);

/***********************************************************************
*
*   FUNCTION
*
*       Application_Initialize
*
*   DESCRIPTION
*
*       Demo application entry point - initializes Nucleus Power
*       demonstration application by creating and initializing necessary
*       tasks, memory pools, and communication components.
*
***********************************************************************/
VOID    Application_Initialize(NU_MEMORY_POOL* mem_pool, NU_MEMORY_POOL* uncached_mem_pool)
{
    VOID *          pointer;
    STATUS          status;


    /* Reference unused parameters to avoid toolset warnings */
    NU_UNUSED_PARAM(uncached_mem_pool);

    /* Create a timer */
    status = NU_Create_Timer(&MyTimer, "CPU Usage Timer", MyTimer_Entry,
                             0, 1, 2, NU_ENABLE_TIMER);

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Allocate memory for task 0 */
        status = NU_Allocate_Memory(mem_pool, &pointer,
                                    4096, NU_NO_SUSPEND);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Create task 0.  */
        status = NU_Create_Task(&Task_Control_Block[0], "TASK 0", Task_0_Entry, 0,
                                NU_NULL, pointer, 4096, TASK0_PRIORITY,
                                TASK0_TIMESLICE, NU_PREEMPT, NU_START);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Allocate memory for task 1 */
        status = NU_Allocate_Memory(mem_pool, &pointer,
                                    DEMO_STACK_SIZE, NU_NO_SUSPEND);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Create task 1.  */
        status = NU_Create_Task(&Task_Control_Block[1], "TASK 1", Task_1_Entry, 0,
                                NU_NULL, pointer, DEMO_STACK_SIZE, TASK1_PRIORITY,
                                TASK1_TIMESLICE, NU_PREEMPT, NU_START);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Allocate memory for task 2 */
        status = NU_Allocate_Memory(mem_pool, &pointer,
                                    DEMO_STACK_SIZE, NU_NO_SUSPEND);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Create task 2 */
        status = NU_Create_Task(&Task_Control_Block[2], "TASK 2", Task_2_Entry, 0,
                                NU_NULL, pointer, DEMO_STACK_SIZE, TASK2_PRIORITY,
                                TASK2_TIMESLICE, NU_PREEMPT, NU_START);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Allocate memory for task 3 */
        status = NU_Allocate_Memory(mem_pool, &pointer,
                                    DEMO_STACK_SIZE, NU_NO_SUSPEND);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Create task 3.  Note that task 4 uses the same instruction area.  */
        status = NU_Create_Task(&Task_Control_Block[3], "TASK 3", Task_3_and_4_Entry, 0,
                                NU_NULL, pointer, DEMO_STACK_SIZE, TASK3_PRIORITY,
                                TASK3_TIMESLICE, NU_PREEMPT, NU_START);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Allocate memory for task 4 */
        status = NU_Allocate_Memory(mem_pool, &pointer,
                                    DEMO_STACK_SIZE, NU_NO_SUSPEND);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Create task 4.  Note that task 3 uses the same instruction area.  */
        status = NU_Create_Task(&Task_Control_Block[4], "TASK 4", Task_3_and_4_Entry, 0,
                                NU_NULL, pointer, DEMO_STACK_SIZE, TASK4_PRIORITY,
                                TASK4_TIMESLICE, NU_PREEMPT, NU_START);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Allocate memory for task 5 */
        status = NU_Allocate_Memory(mem_pool, &pointer,
                                    DEMO_STACK_SIZE, NU_NO_SUSPEND);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Create task 5.  */
        status = NU_Create_Task(&Task_Control_Block[5], "TASK 5", Task_5_Entry, 0,
                                NU_NULL, pointer, DEMO_STACK_SIZE, TASK5_PRIORITY,
                                TASK5_TIMESLICE, NU_PREEMPT, NU_START);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Allocate memory for communication queue */
        status = NU_Allocate_Memory(mem_pool, &pointer,
                                    DEMO_QUEUE_SIZE * sizeof(UNSIGNED),
                                    NU_NO_SUSPEND);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Create communication queue with a message size of 1.  */
        status = NU_Create_Queue(&Queue_0, "QUEUE 0", pointer,
                                 DEMO_QUEUE_SIZE, NU_FIXED_SIZE,
                                 1, NU_FIFO);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Create synchronization semaphore with an initial count of 1.  */
        status = NU_Create_Semaphore(&Semaphore_0, "SEM 0", 1, NU_FIFO);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Create event flag group.  */
        status = NU_Create_Event_Group(&Event_Group_0, "EVGROUP0");
    }

    /* Check to see if previous operation successful */
    if (status != NU_SUCCESS)
    {
        /* Loop forever */
        while(1);
    }
}


/***********************************************************************
*
*   FUNCTION
*
*       MyTimer_Entry
*
*   DESCRIPTION
*
*       Calculate CPU Usage
*
***********************************************************************/
static VOID    MyTimer_Entry(UNSIGNED argc)
{
    static UINT32 Total_Time;
    static UINT32 Idle_Time;
    static UINT32 Last_Idle_Time;
    static UINT32 Last_Total_Time;
    static INT    Idle_Pct;


    /* Get CPU Count */
    (VOID)NU_PM_Get_CPU_Counters (&Total_Time, &Idle_Time);

    /* Calculate idle % */
    Idle_Pct = (INT)((100 * (Idle_Time - Last_Idle_Time))/(Total_Time - Last_Total_Time));

    /* Calculate Usage % */
    Usage_Pct = (INT)(100 - Idle_Pct);

    /* Save last idle and total times */
    Last_Idle_Time = Idle_Time;
    Last_Total_Time = Total_Time;
}


/***********************************************************************
*
*   FUNCTION
*
*       Task_0_Entry
*
*   DESCRIPTION
*
*       Entry function for Task_0.  This task acts as a system timer
*       task.  Each time the system time is incremented, an event is
*       generated by this task.  More complicated systems might use a
*       routine like this to perform periodic message sending and
*       other time oriented functions.
*
*       Power services are also handled in this task.  Devices are
*       "enabled" for use and operating point is changed after every 10
*       loops.  The voltage and frequency is also retrieved for the given
*       operating point.
*
***********************************************************************/
static VOID    Task_0_Entry(UNSIGNED argc, VOID *argv)
{
    static INT              op_chg_cnt = 0;
    static STATUS           status;
    static UINT8            freq_id;
    static UINT8            volt_id;
    static UINT8            operating_point_count;
    static UINT8            operating_point;


    /* Reference all parameters to ensure no toolset warnings */
    NU_UNUSED_PARAM(argc);
    NU_UNUSED_PARAM(argv);

    /* When Nucleus power services are enabled all the devices
       present in the system are in OFF power state by default.
       The peripheral devices used by the application are turned
       ON here.
       NOTE: This is a local function that uses power services API's. */
    status = Enable_Device_Power_State();

    /* Get the number of operating points */
    status = NU_PM_Get_OP_Count(&operating_point_count);

    /* Subtract 1 from the operating point count to get the highest operating point number */
    operating_point = operating_point_count - 1;
    Current_Operating_Point = operating_point;

    /* Continue this process forever.  */
    while(1)
    {
        /* Change operating point after every 10 loops */
        if (op_chg_cnt == 10)
        {
            /* Go to lower operating point */
            operating_point--;
            op_chg_cnt = 0;

            /* Change operating point using power management service */
            status = NU_PM_Set_Current_OP(operating_point);

            /* Check if operating point changed successfully */
            if (status == NU_SUCCESS)
            {
                /* Set new current operationg point */
                Current_Operating_Point = operating_point;
            }
            else
            {
                /* Print info to indicate failed operating point change */
                printf("Operating point could not go lower than => %u\r\n", Current_Operating_Point);

                /* Wrap back to highest operating point */
                operating_point = operating_point_count - 1;
                Current_Operating_Point = operating_point;

                /* Set the operating point back to the highest operating point */
                status = NU_PM_Set_Current_OP(Current_Operating_Point);
            }
        }

        /* Get the voltage and Frequency IDs for this operating point */
        status = NU_PM_Get_OP_VF(Current_Operating_Point, &freq_id, &volt_id);

        /* Check if previous function call passed */
        if (status == NU_SUCCESS)
        {
            /* Get the Frequency for this ID */
            status = NU_PM_Get_Freq_Info(freq_id, &Frequency);

            /* Check if get frequency information API passed */
            if (status == NU_SUCCESS)
            {
                /* Print the log information */
                Print_Log();
            }
            else
            {
                /* Print error log msg */
                printf("Get frequency information failed");
            }
        }
        else
        {
            /* Print error log msg */
            printf("Get operating point voltage and frequency failed\r\n");
        }

        /* Increment operating point change count value */
        op_chg_cnt++;

        /* Sleep for one second. */
        NU_Sleep(NU_PLUS_TICKS_PER_SEC);

        /* Increment the time.  */
        Task_0_Time++;

        /* Set an event flag to lift the suspension on task 5. */
        (VOID)NU_Set_Events(&Event_Group_0, 1, NU_OR);
    }
}


/***********************************************************************
*
*   FUNCTION
*
*       Task_1_Entry
*
*   DESCRIPTION
*
*       Entry function for Task_1.  This task acts as a queue sending
*       task.  This task will only suspend if the queue is full, its sleep,
*       its time-slice expires, or it is preempted by a higher-priority task
*
***********************************************************************/
static VOID   Task_1_Entry(UNSIGNED argc, VOID *argv)
{
    STATUS         status;
    UNSIGNED       send_message;


    /* Reference all parameters to ensure no toolset warnings */
    NU_UNUSED_PARAM(argc);
    NU_UNUSED_PARAM(argv);

    /* Initialize the message contents.  The receiver will examine the
       message contents for errors.  */
    send_message = 0;

    /* Continue this process forever.  */
    while(1)
    {
        /* Send the message to Queue_0, which task 2 reads from.  Note
           that if the destination queue fills up this task suspends until
           room becomes available.  */
        status =  NU_Send_To_Queue(&Queue_0, &send_message, 1, NU_SUSPEND);

        /* Determine if the message was sent successfully.  */
        if (status == NU_SUCCESS)
        {
            /* Increment the number of messages sent counter */
            Task_1_Messages_Sent++;
        }

        /* Modify the contents of the next message to send.  */
        send_message++;

        /* Sleep for one second. */
        NU_Sleep(10);
    }
}


/***********************************************************************
*
*   FUNCTION
*
*       Task_2_Entry
*
*   DESCRIPTION
*
*       Entry function for Task_2.  This task acts as a queue receiving
*       task.  This task will only suspend if the queue is empty, its sleep,
*       its time-slice expires, or it is preempted by a higher-priority task
*
***********************************************************************/
static VOID   Task_2_Entry(UNSIGNED argc, VOID *argv)
{
    STATUS         status;
    UNSIGNED       receive_message;
    UNSIGNED       received_size;
    UNSIGNED       message_expected;


    /* Reference all parameters to ensure no toolset warnings */
    NU_UNUSED_PARAM(argc);
    NU_UNUSED_PARAM(argv);

    /* Initialize the message contents to expect.  */
    message_expected =  0;

    /* Continue this process forever.  */
    while(1)
    {
        /* Retrieve a message from Queue_0, which task 1 writes to.  Note
           that if the source queue is empty this task suspends until
           something becomes available.  */
        status =  NU_Receive_From_Queue(&Queue_0, &receive_message, 1,
                                        &received_size, NU_SUSPEND);

        /* Determine if the message was received successfully.  */
        if (status == NU_SUCCESS)
        {
            /* Increment number of messages received counter */
            Task_2_Messages_Received++;
        }

        /* Check the contents of the message against what this task
           is expecting.  */
        if ((received_size != 1) ||
            (receive_message != message_expected))
        {
            /* Increment invalid message counter */
            Task_2_Invalid_Messages++;
        }

        /* Modify the expected contents of the next message.  */
        message_expected++;

        /* Sleep for one second. */
        NU_Sleep(10);

    }
}

/***********************************************************************
*
*   FUNCTION
*
*       Task_3_and_4_Entry
*
*   DESCRIPTION
*
*       Entry function for Task_3 and Task_4 (both tasks share a
*       common instruction area but have different stacks).  This
*       function has 2 tasks (Task_3 and Task_4) competing for a single
*       resource (a semaphore).  After obtaining the resource, each
*       task keeps it for a short duration before releasing it.  The
*       task not owning the resource will suspend while waiting for this
*       resource.
*
***********************************************************************/
static VOID  Task_3_and_4_Entry(UNSIGNED argc, VOID *argv)
{
    STATUS  status;


    /* Reference all parameters to ensure no toolset warnings */
    NU_UNUSED_PARAM(argc);
    NU_UNUSED_PARAM(argv);

    /* Loop to allocate and deallocate the resource.  */
    while(1)
    {
        /* Allocate the resource.  Suspend until it becomes available.  */
        status =  NU_Obtain_Semaphore(&Semaphore_0, NU_SUSPEND);

        /* If the status is successful, show that this task owns the
           resource.  */
        if (status ==  NU_SUCCESS)
        {

            /* Find which task (3 or 4) has the semaphore */
            Task_3_4_Resource_Owner =  NU_Current_Task_Pointer();

            /* Release the semaphore.  */
            (VOID)NU_Release_Semaphore(&Semaphore_0);
        }

        /* Sleep for one second to cause the other task to suspend on
           the resource.  */
        NU_Sleep(NU_PLUS_TICKS_PER_SEC);
    }
}


/***********************************************************************
*
*   FUNCTION
*
*       Task_5_Entry
*
*   DESCRIPTION
*
*       Entry function for Task_5.  This function repeatedly awaits for
*       an event to be generated / set by Task_0.
*
***********************************************************************/
static VOID  Task_5_Entry(UNSIGNED argc, VOID *argv)
{
    STATUS          status;
    UNSIGNED        event_group;


    /* Reference all parameters to ensure no toolset warnings */
    NU_UNUSED_PARAM(argc);
    NU_UNUSED_PARAM(argv);

    /* Continue this process forever.  */
    while(1)
    {
        /* Wait for an event and consume it.  */
        status =  NU_Retrieve_Events(&Event_Group_0, 1, NU_OR_CONSUME,
                                     &event_group, NU_SUSPEND);

        /* If the status is okay, increment the counter.  */
        if (status == NU_SUCCESS)
        {
            /* Increment the number of events detected counter */
            Task_5_Event_Detections++;
        }
    }
}


/*************************************************************************
*
*   FUNCTION
*
*       Print_Log
*
*   DESCRIPTION
*
*       Outputs some demo status information
*
*************************************************************************/
static VOID   Print_Log(VOID)
{

    /* Output Release Information */
    printf("\n\n\r************************************");
    printf("*******************************************\r\n");

    /* Build a string with the demo name */
    printf("                       Nucleus Power Demonstration\r\n");

    /* Build a string with the build time stamp */
    printf("                     Build Timestamp - " __DATE__ "/" __TIME__ "\r\n");

    /* Output end of banner header */
    printf("******************************************");
    printf("*************************************\n\r\n");

    /* Build a string with current task 0 time */
    printf("Task 0 Time:                      %u\r\n", Task_0_Time);

    /* Build a string containing the number of timer interrupts that have occurred */
    printf("Timer Interrupts:                 %u\r\n\n", (UINT)NU_Retrieve_Clock());

    /* Build a string with number of messages sent */
    printf("Task 1 Messages Sent:             %u\r\n\n", Task_1_Messages_Sent);

    /* Build a string with number of messages received */
    printf("Task 2 Messages Received:         %u\r\n", Task_2_Messages_Received);

    /* Build a string with number of invalid messages */
    printf("Task 2 Invalid Messages:          %u\r\n\n\n", Task_2_Invalid_Messages);

    /* Build a string for who has the resource */
    printf("Task 3/4 Resource Owner:          ");

    /* Check if task 3 has the resource (semaphore) */
    if (Task_3_4_Resource_Owner == &Task_Control_Block[3])
    {
        /* Build string for task 3 owning resource */
        printf("Task 3\r\n\n\n");
    }
    else if (Task_3_4_Resource_Owner == &Task_Control_Block[4])
    {
        /* Build string for task 4 owning resource */
        printf("Task 4\r\n\n\n");
    }
    else
    {
        /* Build string for nobody owning resource
           NOTE:  This should never occur. */
        printf("Nobody\r\n\n\n");
    }

    /* Build a string with number of task 5 event detections */
    printf("Task 5 Event Detections:          %u\r\n\n", Task_5_Event_Detections);

    /* Print current operating point and Frequency */
    printf("Current Operating Point:          %u\r\n", Current_Operating_Point);
    printf("Current CPU Frequency:            %u Mhz\r\n", (INT)(Frequency / 1000000));
    printf("Current CPU Usage:                %d %%\r\n\n", (INT)(Usage_Pct));
}


/*************************************************************************
*
*   FUNCTION
*
*       Turn_ON_Device
*
*   DESCRIPTION
*
*       Call-back function that uses Power Services API to turn devices
*       on the device ID
*
*************************************************************************/
static STATUS Turn_ON_Device(DV_DEV_ID device, VOID* context)
{
    STATUS  status;

    status = NU_PM_Set_Power_State (device, POWER_ON_STATE);

    return status;
}


/*************************************************************************
*
*   FUNCTION
*
*       Turn_OFF_Device
*
*   DESCRIPTION
*
*       Call-back function to use Power Services to turn device OFF
*       using device ID
*
*************************************************************************/
static STATUS Turn_OFF_Device(DV_DEV_ID device, VOID* context)
{
    STATUS  status;

    status = NU_PM_Set_Power_State (device, POWER_OFF_STATE);

    return status;
}


/*************************************************************************
*
*   FUNCTION
*
*       Enable_Device_Power_State
*
*   DESCRIPTION
*
*       Use power services to change device state to find serial device
*       with power support and register the device with Power services.
*
*************************************************************************/
static STATUS Enable_Device_Power_State(VOID)
{
    STATUS              status;
    DV_LISTENER_HANDLE  listener_handle;
    DV_DEV_LABEL        serial_label[] = {{SERIAL_LABEL}, {POWER_CLASS_LABEL}};


    status = DVC_Reg_Change_Notify(serial_label, 1, &Turn_ON_Device,
                                   &Turn_OFF_Device, NU_NULL, &listener_handle);

    return status;
}

