/***********************************
* Include Files
***********************************/
#include                <stdio.h>
#include                "nucleus.h"
#include                "kernel/nu_kernel.h"
#include                "kernel_demo.h"

/***********************************
* Global Variables
***********************************/
static NU_TASK          Task_Control_Block[7];
static NU_TASK *        Task_3_4_Resource_Owner;
static NU_QUEUE         Queue_0;
static NU_SEMAPHORE     Semaphore_0;
static NU_EVENT_GROUP   Event_Group_0;
static UINT             Task_0_Time;
static UINT             Task_1_Messages_Sent;
static UINT             Task_2_Messages_Received;
static UINT             Task_2_Invalid_Messages;
static UINT             Task_5_Event_Detections;

/***********************************
* Function Prototypes
***********************************/
static VOID             Task_0_Entry(UNSIGNED argc, VOID *argv);
static VOID             Task_1_Entry(UNSIGNED argc, VOID *argv);
static VOID             Task_2_Entry(UNSIGNED argc, VOID *argv);
static VOID             Task_3_and_4_Entry(UNSIGNED argc, VOID *argv);
static VOID             Task_5_Entry(UNSIGNED argc, VOID *argv);
static VOID             Task_6_Entry(UNSIGNED argc, VOID *argv);


/***********************************************************************
*
*   FUNCTION
*
*       Application_Initialize
*
*   DESCRIPTION
*
*       Demo application entry point - initializes Nucleus Plus
*       demonstration application by creating and initializing necessary
*       tasks, memory pools, and communication components.
*
***********************************************************************/
VOID    Application_Initialize(NU_MEMORY_POOL* mem_pool, NU_MEMORY_POOL* uncached_mem_pool)
{
    VOID *         pointer;
    STATUS         status;

    
    /* Reference unused parameters to avoid toolset warnings */
    NU_UNUSED_PARAM(uncached_mem_pool);

    /* Allocate memory for task 0 */
    status = NU_Allocate_Memory(mem_pool, &pointer,
                                DEMO_STACK_SIZE, NU_NO_SUSPEND);

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Create task 0.  */
        status = NU_Create_Task(&Task_Control_Block[0], "TASK 0", Task_0_Entry, 0, NU_NULL, pointer,
                                DEMO_STACK_SIZE, TASK0_PRIORITY, TASK0_TIMESLICE,
                                NU_PREEMPT, NU_START);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Allocate memory for task 1 */
        status = NU_Allocate_Memory(mem_pool, &pointer,
                                    DEMO_STACK_SIZE, NU_NO_SUSPEND);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Create task 1.  */
        status = NU_Create_Task(&Task_Control_Block[1], "TASK 1", Task_1_Entry, 0, NU_NULL, pointer,
                                DEMO_STACK_SIZE, TASK1_PRIORITY, TASK1_TIMESLICE,
                                NU_PREEMPT, NU_START);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Allocate memory for task 2 */
        status = NU_Allocate_Memory(mem_pool, &pointer,
                                    DEMO_STACK_SIZE, NU_NO_SUSPEND);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Create task 2 */
        status = NU_Create_Task(&Task_Control_Block[2], "TASK 2", Task_2_Entry, 0, NU_NULL, pointer,
                                DEMO_STACK_SIZE, TASK2_PRIORITY, TASK2_TIMESLICE,
                                NU_PREEMPT, NU_START);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Allocate memory for task 3 */
        status = NU_Allocate_Memory(mem_pool, &pointer,
                                    DEMO_STACK_SIZE, NU_NO_SUSPEND);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Create task 3.  Note that task 4 uses the same instruction area.  */
        status = NU_Create_Task(&Task_Control_Block[3], "TASK 3", Task_3_and_4_Entry, 0, NU_NULL, pointer,
                                DEMO_STACK_SIZE, TASK3_PRIORITY, TASK3_TIMESLICE,
                                NU_PREEMPT, NU_START);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Allocate memory for task 4 */
        status = NU_Allocate_Memory(mem_pool, &pointer,
                                    DEMO_STACK_SIZE, NU_NO_SUSPEND);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Create task 4.  Note that task 3 uses the same instruction area.  */
        status = NU_Create_Task(&Task_Control_Block[4], "TASK 4", Task_3_and_4_Entry, 0, NU_NULL, pointer,
                                DEMO_STACK_SIZE, TASK4_PRIORITY, TASK4_TIMESLICE,
                                NU_PREEMPT, NU_START);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Allocate memory for task 5 */
        status = NU_Allocate_Memory(mem_pool, &pointer,
                                    DEMO_STACK_SIZE, NU_NO_SUSPEND);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Create task 5.  */
        status = NU_Create_Task(&Task_Control_Block[5], "TASK 5", Task_5_Entry, 0, NU_NULL, pointer,
                                DEMO_STACK_SIZE, TASK5_PRIORITY, TASK5_TIMESLICE,
                                NU_PREEMPT, NU_START);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Allocate memory for task 6 */
        status = NU_Allocate_Memory(mem_pool, &pointer,
                                    4096, NU_NO_SUSPEND);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Create task 6 */
        status = NU_Create_Task(&Task_Control_Block[6], "TASK 6", Task_6_Entry, 0, NU_NULL, pointer,
                                4096, TASK6_PRIORITY, TASK6_TIMESLICE,
                                NU_PREEMPT, NU_START);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Allocate memory for communication queue */
        status = NU_Allocate_Memory(mem_pool, &pointer,
                                    DEMO_QUEUE_SIZE * sizeof(UNSIGNED),
                                    NU_NO_SUSPEND);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Create communication queue with a message size of 1.  */
        status = NU_Create_Queue(&Queue_0, "QUEUE 0", pointer,
                                 DEMO_QUEUE_SIZE, NU_FIXED_SIZE,
                                 1, NU_FIFO);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Create synchronization semaphore with an initial count of 1.  */
        status = NU_Create_Semaphore(&Semaphore_0, "SEM 0", 1, NU_FIFO);
    }

    /* Check to see if previous operation successful */
    if (status == NU_SUCCESS)
    {
        /* Create event flag group.  */
        status = NU_Create_Event_Group(&Event_Group_0, "EVGROUP0");
    }
    
    /* Check to see if previous operation successful */
    if (status != NU_SUCCESS)
    {
        /* Loop forever */
        while(1);
    }
}


/***********************************************************************
*
*   FUNCTION
*
*       Task_0_Entry
*
*   DESCRIPTION
*
*       Entry function for Task_0.  This task acts as a system timer
*       task.  Each time the system time is incremented, an event is
*       generated by this task.  More complicated systems might use a
*       routine like this to perform periodic message sending and
*       other time oriented functions
*
***********************************************************************/
static VOID    Task_0_Entry(UNSIGNED argc, VOID *argv)
{
    /* Reference all parameters to ensure no toolset warnings */
    NU_UNUSED_PARAM(argc);
    NU_UNUSED_PARAM(argv);

    /* Continue this process forever.  */
    while(1)
    {
        /* Sleep for one second. */
        NU_Sleep(NU_PLUS_TICKS_PER_SEC);

        /* Increment the time.  */
        Task_0_Time++;

        /* Set an event flag to lift the suspension on task 5. */
        (VOID)NU_Set_Events(&Event_Group_0, 1, NU_OR);
    }
}


/***********************************************************************
*
*   FUNCTION
*
*       Task_1_Entry
*
*   DESCRIPTION
*
*       Entry function for Task_1.  This task acts as a queue sending
*       task.  This task will only suspend if the queue is full, its
*       time-slice expires, or it is preempted by a higher-priority task
*
***********************************************************************/
static VOID   Task_1_Entry(UNSIGNED argc, VOID *argv)
{
    STATUS         status;
    UNSIGNED       send_message;


    /* Reference all parameters to ensure no toolset warnings */
    NU_UNUSED_PARAM(argc);
    NU_UNUSED_PARAM(argv);

    /* Initialize the message contents.  The receiver will examine the
       message contents for errors.  */
    send_message = 0;

    /* Continue this process forever.  */
    while(1)
    {
        /* Send the message to Queue_0, which task 2 reads from.  Note
           that if the destination queue fills up this task suspends until
           room becomes available.  */
        status =  NU_Send_To_Queue(&Queue_0, &send_message, 1, NU_SUSPEND);

        /* Determine if the message was sent successfully.  */
        if (status == NU_SUCCESS)
        {
            /* Increment the number of messages sent counter */
            Task_1_Messages_Sent++;
        }

        /* Modify the contents of the next message to send.  */
        send_message++;

        /* Suspend for 1/10th of a second */
        NU_Sleep(NU_PLUS_TICKS_PER_SEC/10);
    }
}


/***********************************************************************
*
*   FUNCTION
*
*       Task_2_Entry
*
*   DESCRIPTION
*
*       Entry function for Task_2.  This task acts as a queue receiving
*       task.  This task will only suspend if the queue is empty, its
*       time-slice expires, or it is preempted by a higher-priority task
*
***********************************************************************/
static VOID   Task_2_Entry(UNSIGNED argc, VOID *argv)
{
    STATUS         status;
    UNSIGNED       receive_message;
    UNSIGNED       received_size;
    UNSIGNED       message_expected;


    /* Reference all parameters to ensure no toolset warnings */
    NU_UNUSED_PARAM(argc);
    NU_UNUSED_PARAM(argv);

    /* Initialize the message contents to expect.  */
    message_expected =  0;

    /* Continue this process forever.  */
    while(1)
    {

        /* Retrieve a message from Queue_0, which task 1 writes to.  Note
           that if the source queue is empty this task suspends until
           something becomes available.  */
        status =  NU_Receive_From_Queue(&Queue_0, &receive_message, 1,
                                        &received_size, NU_SUSPEND);

        /* Determine if the message was received successfully.  */
        if (status == NU_SUCCESS)
        {
            /* Increment number of messages received counter */
            Task_2_Messages_Received++;
        }

        /* Check the contents of the message against what this task
           is expecting.  */
        if ((received_size != 1) ||
            (receive_message != message_expected))
        {
            /* Increment invalid message counter */
            Task_2_Invalid_Messages++;
        }

        /* Modify the expected contents of the next message.  */
        message_expected++;
    }
}


/***********************************************************************
*
*   FUNCTION
*
*       Task_3_and_4_Entry
*
*   DESCRIPTION
*
*       Entry function for Task_3 and Task_4 (both tasks share a
*       common instruction area but have different stacks).  This
*       function has 2 tasks (Task_3 and Task_4) competing for a single
*       resource (a semaphore).  After obtaining the resource, each
*       task keeps it for a short duration before releasing it.  The
*       task not owning the resource will suspend while waiting for this
*       resource.
*
***********************************************************************/
static VOID  Task_3_and_4_Entry(UNSIGNED argc, VOID *argv)
{
    STATUS  status;


    /* Reference all parameters to ensure no toolset warnings */
    NU_UNUSED_PARAM(argc);
    NU_UNUSED_PARAM(argv);

    /* Loop to allocate and deallocate the resource.  */
    while(1)
    {
        /* Allocate the resource.  Suspend until it becomes available.  */
        status =  NU_Obtain_Semaphore(&Semaphore_0, NU_SUSPEND);

        /* If the status is successful, show that this task owns the
           resource.  */
        if (status ==  NU_SUCCESS)
        {

            /* Find which task (3 or 4) has the semaphore */
            Task_3_4_Resource_Owner =  NU_Current_Task_Pointer();

            /* Sleep for one second to cause the other task to suspend on
               the resource.  */
            NU_Sleep(NU_PLUS_TICKS_PER_SEC);

            /* Release the semaphore.  */
            (VOID)NU_Release_Semaphore(&Semaphore_0);
        }
    }
}


/***********************************************************************
*
*   FUNCTION
*
*       Task_5_Entry
*
*   DESCRIPTION
*
*       Entry function for Task_5.  This function repeatedly awaits for
*       an event to be generated / set by Task_0.
*
***********************************************************************/
static VOID  Task_5_Entry(UNSIGNED argc, VOID *argv)
{
    STATUS          status;
    UNSIGNED        event_group;


    /* Reference all parameters to ensure no toolset warnings */
    NU_UNUSED_PARAM(argc);
    NU_UNUSED_PARAM(argv);

    /* Continue this process forever.  */
    while(1)
    {
        /* Wait for an event and consume it.  */
        status =  NU_Retrieve_Events(&Event_Group_0, 1, NU_OR_CONSUME,
                                     &event_group, NU_SUSPEND);

        /* If the status is okay, increment the counter.  */
        if (status == NU_SUCCESS)
        {
            /* Increment the number of events detected counter */
            Task_5_Event_Detections++;
        }
    }
}


/*************************************************************************
*
*   FUNCTION
*
*       Task_6_Entry
*
*   DESCRIPTION
*
*       Outputs some demo status information
*
*************************************************************************/
static VOID   Task_6_Entry(UNSIGNED argc, VOID *argv)
{
    /* Loop forever */
    while (1)
    {
        /* Output Release Information */
        printf("\n\n\r************************************");
        printf("*******************************************\r\n");

        /* Build a string with the release information */
        printf("                         ");
        printf(NU_Release_Information());
        printf(" Demonstration\r\n");

        /* Build a string with the build time stamp */
        printf("                     ");
        printf("Build Timestamp - " __DATE__ "/" __TIME__ "\r\n");

        /* Output end of banner header */
        printf("******************************************");
        printf("*************************************\n\r\n");

        /* Build a string with current task 0 time */
        printf("Task 0 Time:                      %u\r\n", Task_0_Time);

        /* Build a string containing the number of timer interrupts that have occurred */
        printf("Timer Interrupts:                 %u\r\n\n", (UINT)NU_Retrieve_Clock());

        /* Build a string with number of messages sent */
        printf("Task 1 Messages Sent:             %u\r\n\n", Task_1_Messages_Sent);

        /* Build a string with number of messages received */
        printf("Task 2 Messages Received:         %u\r\n", Task_2_Messages_Received);

        /* Build a string with number of invalid messages */
        printf("Task 2 Invalid Messages:          %u\r\n\n\n", Task_2_Invalid_Messages);

        /* Build a string for who has the resource */
        printf("Task 3/4 Resource Owner:          ");

        /* Check if task 3 has the resource (semaphore) */
        if (Task_3_4_Resource_Owner == &Task_Control_Block[3])
        {
            /* Build string for task 3 owning resource */
            printf("Task 3\r\n\n\n");
        }
        else if (Task_3_4_Resource_Owner == &Task_Control_Block[4])
        {
            /* Build string for task 4 owning resource */
            printf("Task 4\r\n\n\n");
        }
        else
        {
            /* Build string for nobody owning resource
               NOTE:  This should never occur. */
            printf("Nobody\r\n\n\n");
        }

        /* Build a string with number of task 5 event detections */
        printf("Task 5 Event Detections:          %u\r\n\n\n\n\n", Task_5_Event_Detections);

        /* Await for a keypress to update status information */
        printf("                       -----PRESS ANY KEY TO UPDATE-----");

        /* Get character */
        (VOID)getchar();

    }   /* while (1) */

}
