/***********************************************************************
*
*             Copyright 2011 Mentor Graphics Corporation
*                         All Rights Reserved.
*
* THIS WORK CONTAINS TRADE SECRET AND PROPRIETARY INFORMATION WHICH IS
* THE PROPERTY OF MENTOR GRAPHICS CORPORATION OR ITS LICENSORS AND IS
* SUBJECT TO LICENSE TERMS.
*
************************************************************************

************************************************************************
*
*   FILE NAME
*
*       renesas_sh_defs.h
*
*   DESCRIPTION
*
*       This file contains all definitions, structures, etc for the
*       Renesas SuperH toolset
*
*   DATA STRUCTURES
*
*       None
*
*   DEPENDENCIES
*
*       machine.h
*
***********************************************************************/

#ifndef         RENESAS_SUPERH_DEFS_H
#define         RENESAS_SUPERH_DEFS_H

/* Include machine.h to inclue Renesas C++ Compiler intrinsic functions */
#include        <machine.h>

/* Define required for RTL support. */
#ifndef         ESAL_TS_RTL_SUPPORT
#define         ESAL_TS_RTL_SUPPORT                     NU_TRUE
#endif

/* Define required stack pointer alignment for the given toolset.*/
#define         ESAL_TS_REQ_STK_ALIGNMENT               4

/* Defines if toolset supports 64-bit data types (long long) */
#define         ESAL_TS_64BIT_SUPPORT                   NU_FALSE

/* Defines, in bytes, toolset minimum required alignment for structures */
#define         ESAL_TS_STRUCT_ALIGNMENT                4

/* Size, in bits, of integers for the given toolset / architecture */
#define         ESAL_TS_INTEGER_SIZE                    32

/* Size, in bits, of code pointer for the given toolset / architecture */
#define         ESAL_TS_CODE_PTR_SIZE                   32

/* Size, in bits, of data pointer for the given toolset / architecture */
#define         ESAL_TS_DATA_PTR_SIZE                   32

/* Define if necessary to copy code / data from ROM to RAM */
#define         ESAL_TS_ROM_TO_RAM_COPY_SUPPORT         CFG_NU_OS_KERN_PLUS_ENABLE_ROM_TO_RAM_COPY

/* Defines used to enforce alignment of data structure members (if necessary). */


/* Define tool specific type for HUGE and FAR data pointers - these will usually
   be defined to nothing.  Some 16-bit architectures may require these
   to be defined differently to access data across memory pages */
#define         ESAL_TS_HUGE_PTR_TYPE
#define         ESAL_TS_FAR_PTR_TYPE

/* Define if position-independent code / data (PIC/PID) support (if available)
   is enabled.
   NOTE:  This may be required to be set to NU_TRUE when utilizing any
          OS components requiring position-independent code / data */
#define         ESAL_TS_PIC_PID_SUPPORT                 NU_FALSE



/* Define number of single-precision floating point registers required to
   be saved within a toolset stack frame (FR12-FR15) */
#define         ESAL_TS_STK_NUM_FPU_REGS        4

/* Define stack frame structure for the toolset / architecture supported. */
typedef struct  ESAL_TS_STK_STRUCT
{
    UINT32      stack_type;
    UINT32      r8;
    UINT32      r9;
    UINT32      r10;
    UINT32      r11;
    UINT32      r12;
    UINT32      r13;
    UINT32      r14;
    UINT32      gbr;
    UINT32      mach;
    UINT32      macl;
    UINT32      rtn_address;

#if (ESAL_AR_STK_FPU_SUPPORT == NU_TRUE)

    UINT32      fpr[ESAL_AR_STK_NUM_FPU_REGS];

#endif  /* ESAL_AR_STK_FPU_SUPPORT == NU_TRUE */

} ESAL_TS_STK;

/* Becuase time.h is not present in Renesas Tool Set 
   (time_t is required in plus_core.h) */
typedef unsigned long time_t;

/* Extern function references */
extern VOID     ESAL_TS_RTE_SP_Write(VOID* stack_ptr);
extern VOID *   ESAL_TS_RTE_SP_Read(VOID);
extern VOID     ESAL_TS_RTE_Sim_IO_Write(UINT32 val);


/* This macro writes the stack pointer. */
#define         ESAL_TS_RTE_SP_WRITE(stack_ptr)                                     \
                {                                                                   \
                    /* Set hardware stack pointer to passed in address */           \
                    ESAL_TS_RTE_SP_Write(stack_ptr);                                \
                }

/* This macro reads the stack pointer. */
#define         ESAL_TS_RTE_SP_READ()                                               \
                    ESAL_TS_RTE_SP_Read()

/* This macro reads the status register (SR) */
#define         ESAL_TS_RTE_SR_READ(sr_value_ptr)                                   \
                {                                                                   \
                    /* Get current SR value */                                      \
                    *sr_value_ptr = get_cr();                                       \
                }

/* This macro writes the status register (SR) */
#define         ESAL_TS_RTE_SR_WRITE(sr_value)                                      \
                {                                                                   \
                    /* Write new SR value */                                        \
                    set_cr((INT)sr_value);                                          \
                }

/* This macro executes a NOP instruction */
#define         ESAL_TS_RTE_NOP_EXECUTE()                                           \
                {                                                                   \
                    nop();                                                          \
                }

/* This macro sets the VBR */
#define         ESAL_TS_RTE_VBR_WRITE(vbr_value)                                    \
                {                                                                   \
                    /* Write the new VBR value */                                   \
                    set_vbr((VOID *)vbr_value);                                     \
                }

/* This macro reads the FPSCR */
#define         ESAL_TS_RTE_FPSCR_READ(fpscr_ptr)                                   \
                {                                                                   \
                    /* Read the FPSCR */                                            \
                    *fpscr_ptr = (UINT32)get_fpscr();                               \
                }

/* This macro executes a SLEEP instruction */
#define         ESAL_TS_RTE_SLEEP_EXECUTE()                                         \
                {                                                                   \
                    sleep();                                                        \
                }

/* This macro writes to the sim I/O */
#define         ESAL_TS_RTE_SIM_IO_WRITE(val)                                       \
                {                                                                   \
                    ESAL_TS_RTE_Sim_IO_Write(val);                                  \
                }

#if (ESAL_TS_PIC_PID_SUPPORT == NU_TRUE)



/* This macro sets the PIC/PID base address register */
#define         ESAL_TS_RTE_PIC_PID_BASE_SET(pic_base, pid_base)                    \
                {                                                                   \

                }

/* This macro gets the PIC/PID base address register */
#define         ESAL_TS_RTE_PIC_PID_BASE_GET(pic_base_ptr, pid_base_ptr)            \
                {                                                                   \

                }

#endif  /* ESAL_TS_PIC_PID_SUPPORT == NU_TRUE */

/* Macros for memory definitions */
#define         TOOLSET_BSS_START_ADDR      (VOID *)__sectop("B")
#define         TOOLSET_BSS_END_ADDR        (VOID *)__secend("B")
#define         TOOLSET_BSS_SIZE            (UINT32)__secsize("B")
#define         TOOLSET_DATA_SRC_ADDR       (VOID *)__sectop("D")
#define         TOOLSET_DATA_DST_ADDR       (VOID *)__sectop("D")
#define         TOOLSET_DATA_SIZE           (UINT32)__secsize("D")

/* This macro marks a symbol declaration as weakly linked */
#define         ESAL_TS_WEAK_REF(decl)      decl

/* This macro marks a symbol definition as weakly linked */
#define         ESAL_TS_WEAK_DEF(decl)      decl

/* This macro returns the passed value */
#define         ESAL_TS_NO_RETURN(return_val) return(return_val)

/* This macro places a compiler memory barrier to ensure read / write commands
 * cannot be re-ordered around it */
#define         ESAL_TS_RTE_COMPILE_MEM_BARRIER()                                   \
                {                                                                   \
                    nop();                                                          \
                }

#endif  /* RENESAS_SUPERH_DEFS_H */
