/***********************************************************************
*
*             Copyright 2011 Mentor Graphics Corporation
*                         All Rights Reserved.
*
* THIS WORK CONTAINS TRADE SECRET AND PROPRIETARY INFORMATION WHICH IS
* THE PROPERTY OF MENTOR GRAPHICS CORPORATION OR ITS LICENSORS AND IS
* SUBJECT TO LICENSE TERMS.
*
************************************************************************

************************************************************************
*
*   FILE NAME
*
*       ppc_mmu_e500_defs.h
*
*   DESCRIPTION
*
*       This file contains all definitions, structures, etc for the
*       PPC E500 MMU.
*
*   DATA STRUCTURES
*
*       None
*
*   DEPENDENCIES
*
*       None
*
***********************************************************************/

#ifndef     PPC_MMU_E500_DEFS_H
#define     PPC_MMU_E500_DEFS_H

/*********************************************
* Common definitions
*********************************************/

#define     PPC_MMU_E500_VARIANT                4

/* Sizes, in bytes, of instruction cache for this processor.
   NOTE:  This is the size of the core cache (L1 cache) for this processor. */
#define     ESAL_CO_MEM_ICACHE_SIZE             (UINT)(CFG_NU_OS_ARCH_PPC_COM_ICACHE_SIZE * 1024)
#define     ESAL_CO_MEM_DCACHE_SIZE             (UINT)(CFG_NU_OS_ARCH_PPC_COM_DCACHE_SIZE * 1024)

/* Size of cache block */
#define     ESAL_CO_MEM_BLOCK_SIZE              32

/* PIC Error codes */
#define     MAX_PAGES_PER_MEM_REGION            16

#define     MAX_MEMORY_REGIONS                  16

#define     E500_MAX_TLB1_ENTRIES               16
#define     E500_MAX_SUPPORTED_PAGE_SIZES       11
#define     E500V2_HCD_PAGE_SIZE_IDX            (E500_MAX_SUPPORTED_PAGE_SIZES -1)
#define     E500V2_LCD_PAGE_SIZE_IDX            0

#define     DEFAULT_NUCLEUS_VIR_ADDR_BASE       0

#define     ESAL_MEMORY_REGISTRY_ERROR          -1
#define     ESAL_UNSUPPORTED_MEM_TYPE           -2
#define     ESAL_INVALID_SIZE                   -3
#define     ESAL_INVALID_INPUT                  -4
#define     ESAL_TLB_UNAVAIL                    -5


#define     INVALID_ENTRY                       1
#define     VALID_ENTRY                         2
#define     ADDR_SPA_0                          0
#define     ADDR_SPA_1                          1

#define     CACHED_UNCACHED_RATIO_1_1           2

#define     MIN_PAGE_SIZE_MASK                  0xFFFFE000

enum PAGE_SIZES {
    ESAL_PG_SZ_START_IDX = 0,
    ESAL_CO_MEM_PAGE_SIZE_4KB,
    ESAL_CO_MEM_PAGE_SIZE_16KB,
    ESAL_CO_MEM_PAGE_SIZE_64KB,
    ESAL_CO_MEM_PAGE_SIZE_256KB,
    ESAL_CO_MEM_PAGE_SIZE_1MB,
    ESAL_CO_MEM_PAGE_SIZE_4MB,
    ESAL_CO_MEM_PAGE_SIZE_16MB,
    ESAL_CO_MEM_PAGE_SIZE_64MB,
    ESAL_CO_MEM_PAGE_SIZE_256MB,
    ESAL_CO_MEM_PAGE_SIZE_1GB,
    ESAL_CO_MEM_PAGE_SIZE_4GB,
    ESAL_PG_SZ_END_IDX,
};

/*********************************************
* MMU definitions
*********************************************/

#define     L_WORD_IDX(x)                       (x-32)
#define     PAGE_MASK                           0xFFFFF000

/* Define SPR registers involved with TLB control */
#define     ESAL_CO_MEM_TLB1CFG_REG_ID          689

/* TLB1CFG bit definitions */
#define     ESAL_CO_MEM_TLB1CFG_NENTRY          0x00000FFF

/* Total number of L2_CAM entries */
#define     L2CAM_TLB1_MAX_ENTRIES              16

/* MAS0 bit definitions */
#define     ESAL_CO_MEM_MAS0_TLBSEL_TLB1        ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(35)))
#define     ESAL_CO_MEM_MAS0_ESEL(x)            (x << (31- L_WORD_IDX(47)))
#define     ESAL_CO_MEM_MAS0_TLBSEL_TLB0        ESAL_GE_MEM_32BIT_CLEAR((L_WORD_IDX(35)))
/* MAS1 bit definitions */
#define     ESAL_CO_MEM_MAS1_V_BIT_VALID        ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(32)))
#define     ESAL_CO_MEM_MAS1_V_BIT_INVALID      ESAL_GE_MEM_32BIT_CLEAR((L_WORD_IDX(32)))
#define     ESAL_CO_MEM_MAS1_IPROT_PROTECTED    ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(33)))
#define     ESAL_CO_MEM_MAS1_TID(x)             (x << (31 - L_WORD_IDX(47)))
#define     ESAL_CO_MEM_MAS1_TS_1               ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(51)))
#define     ESAL_CO_MEM_MAS1_TSIZE(x)           (x << (31 - L_WORD_IDX(55)))
#define     ESAL_CO_MEM_MAS1_TSIZE_TO_BYTES(x)  (0x1 << ((2 * x) + 10))

/* MAS2 bit definitions */
#define     ESAL_CO_MEM_GET_EPN(x)              ((x & PAGE_MASK) >> 12)
#define     ESAL_CO_MEM_MAS2_EPN(x)             (x << (31 - L_WORD_IDX(51)))
#define     ESAL_CO_MEM_MAS2_X0                 ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(57)))
#define     ESAL_CO_MEM_MAS2_X1                 ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(58)))
#define     ESAL_CO_MEM_MAS2_W                  ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(59)))
#define     ESAL_CO_MEM_MAS2_I                  ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(60)))
#define     ESAL_CO_MEM_MAS2_M                  ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(61)))
#define     ESAL_CO_MEM_MAS2_G                  ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(62)))
#define     ESAL_CO_MEM_MAS2_E                  ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(63)))

/* MAS3 bit definitions */
#define     ESAL_CO_MEM_GET_RPN(x)              ((x & PAGE_MASK) >> 12)
#define     ESAL_CO_MEM_MAS3_RPN(x)             (x << (31 - L_WORD_IDX(51)))
#define     ESAL_CO_MEM_MAS3_U0_U3(x)           (x << (31 - L_WORD_IDX(57)))
#define     ESAL_CO_MEM_MAS3_UX                 ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(58)))
#define     ESAL_CO_MEM_MAS3_SX                 ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(59)))
#define     ESAL_CO_MEM_MAS3_UW                 ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(60)))
#define     ESAL_CO_MEM_MAS3_SW                 ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(61)))
#define     ESAL_CO_MEM_MAS3_UR                 ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(62)))
#define     ESAL_CO_MEM_MAS3_SR                 ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(63)))
#define     ESAL_KERNEL_SUP_PERMISSIONS         0x15

/* MAS4 bit definitions */
#define     ESAL_CO_MEM_MAS4_TLBSELD            ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(35)))
#define     ESAL_CO_MEM_MAS4_TIDSELD            (x << (L_WORD_IDX(46)-1))
#define     ESAL_CO_MEM_MAS4_TSIZED             (x << (L_WORD_IDX(52)-1))
#define     ESAL_CO_MEM_MAS4_X0D                ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(57)))
#define     ESAL_CO_MEM_MAS4_X1D                ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(58)))
#define     ESAL_CO_MEM_MAS4_WD                 ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(59)))
#define     ESAL_CO_MEM_MAS4_ID                 ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(60)))
#define     ESAL_CO_MEM_MAS4_MD                 ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(61)))
#define     ESAL_CO_MEM_MAS4_GD                 ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(62)))
#define     ESAL_CO_MEM_MAS4_ED                 ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(63)))

/* MAS5 not implemented in E500 core complex */

/* MAS6 bit definitions */
#define     ESAL_CO_MEM_MAS6_SPID0(x)           (x << (L_WORD_IDX(40)-1))
#define     ESAL_CO_MEM_MAS6_SAS                ESAL_GE_MEM_32BIT_SET((L_WORD_IDX(63)))

/* MAS7 bit definitions */
#define     ESAL_CO_MEM_MAS7_RPN(x)             (x << (L_WORD_IDX(60)-1))

/*********************************************
* Cache definitions
*********************************************/

/* Define core cache availability
   NOTE:  A differentiation is made in ESAL between cache that
          is contained on a processor and cache that is
          inherent as part of a core (L2 vs L1 cache). */
#define     ESAL_CO_CACHE_AVAILABLE             NU_TRUE

/*********************************************
* Common definitions
*********************************************/

/* Define SPR registers involved with cache control */
#define     ESAL_CO_MEM_HID1_REG_ID             1009
#define     ESAL_CO_MEM_L1CSR0_REG_ID           1010
#define     ESAL_CO_MEM_L1CSR1_REG_ID           1011
#define     ESAL_CO_MEM_L1CFG0_REG_ID           515
#define     ESAL_CO_MEM_L1CFG1_REG_ID           516

/* Define the Instruction Address Space Enable (IS) and Data Address Space (DS) Enable bits */
#define     ESAL_CO_MEM_MSR_IS_BIT              ESAL_GE_MEM_32BIT_SET(26)
#define     ESAL_CO_MEM_MSR_DS_BIT              ESAL_GE_MEM_32BIT_SET(27)

/* Define Address Broadcast Enable */
#define     ESAL_CO_MEM_HID1_ABE_BIT            ESAL_GE_MEM_32BIT_SET(19)

/* Define Level 1 Control and Status Register 0 (L1CSR0) bits */
#define     ESAL_CO_MEM_L1CSR0_CPE_BIT          ESAL_GE_MEM_32BIT_SET(15) /* Cache Parity Checking Enable */
#define     ESAL_CO_MEM_L1CSR0_CPI_BIT          ESAL_GE_MEM_32BIT_SET(16) /* Party Error Injection Enable */
#define     ESAL_CO_MEM_L1CSR0_CSLC_BIT         ESAL_GE_MEM_32BIT_SET(20) /* Cache Snoop Lock Clear       */
#define     ESAL_CO_MEM_L1CSR0_CUL_BIT          ESAL_GE_MEM_32BIT_SET(21) /* Cache Unable to Lock         */
#define     ESAL_CO_MEM_L1CSR0_CL0_BIT          ESAL_GE_MEM_32BIT_SET(22) /* Cache Lock Overflow          */
#define     ESAL_CO_MEM_L1CSR0_CLFR_BIT         ESAL_GE_MEM_32BIT_SET(23) /* Cache Lock bits Flash Reset  */
#define     ESAL_CO_MEM_L1CSR0_CFI_BIT          ESAL_GE_MEM_32BIT_SET(30) /* Cache Flash Invalidate       */  
#define     ESAL_CO_MEM_L1CSR0_CE_BIT           ESAL_GE_MEM_32BIT_SET(31) /* Cache Enable                 */

/* Define Level 1 Control and Status Register 1 (L1CSR1) bits */
#define     ESAL_CO_MEM_L1CSR1_ICPE_BIT         ESAL_GE_MEM_32BIT_SET(15) /* Cache Parity Checking Enable */
#define     ESAL_CO_MEM_L1CSR1_ICPI_BIT         ESAL_GE_MEM_32BIT_SET(16) /* Party Error Injection Enable */
#define     ESAL_CO_MEM_L1CSR1_ICSLC_BIT        ESAL_GE_MEM_32BIT_SET(20) /* Cache Snoop Lock Clear       */
#define     ESAL_CO_MEM_L1CSR1_ICUL_BIT         ESAL_GE_MEM_32BIT_SET(21) /* Cache Unable to Lock         */
#define     ESAL_CO_MEM_L1CSR1_ICL0_BIT         ESAL_GE_MEM_32BIT_SET(22) /* Cache Lock Overflow          */
#define     ESAL_CO_MEM_L1CSR1_ICLFR_BIT        ESAL_GE_MEM_32BIT_SET(23) /* Cache Lock bits Flash Reset  */
#define     ESAL_CO_MEM_L1CSR1_ICFI_BIT         ESAL_GE_MEM_32BIT_SET(30) /* Cache Flash Invalidate       */  
#define     ESAL_CO_MEM_L1CSR1_ICE_BIT          ESAL_GE_MEM_32BIT_SET(31) /* Cache Enable                 */

/* Define Level 1 Cache Configuration Register 0 (L1CCFG0) bits */

/* Cache Architecture: Default 00 = Harvard */
#define     ESAL_CO_MEM_L1CFG0_CARCH            (ESAL_GE_MEM_32BIT_SET(0) | ESAL_GE_MEM_32BIT_SET(1)) 

/* Cache Line Size: Default 00 = 32 bytes */   
#define     ESAL_CO_MEM_L1CFG0_CBSIZE           (ESAL_GE_MEM_32BIT_SET(7) | ESAL_GE_MEM_32BIT_SET(8))

/* Cache Replacement Policy: Default 01 = Pseudo LRU */   
#define     ESAL_CO_MEM_L1CFG0_CREPL            (ESAL_GE_MEM_32BIT_SET(9) | ESAL_GE_MEM_32BIT_SET(10))

/* Cache Locking Available: Default 1 = Available */
#define     ESAL_CO_MEM_L1CFG0_CLA              ESAL_GE_MEM_32BIT_SET(11)

/* Cache Parity Available: Default 1 = Available */
#define     ESAL_CO_MEM_L1CFG0_CPA              ESAL_GE_MEM_32BIT_SET(12)

/* Cache number of ways (111=8) */
#define     ESAL_CO_MEM_L1CFG0_CNWAY            (ESAL_GE_MEM_32BIT_SET(18) |    \
                                                 ESAL_GE_MEM_32BIT_SET(19) |    \
                                                 ESAL_GE_MEM_32BIT_SET(20)) 

#define     ESAL_CO_MEM_L1CFG0_CSIZE_MASK       0x000000FF      /* Cache Size mask (32 KBytes=0x20) */

/* Define Level 1 Cache Configuration Register 1 (L1CCFG1) bits */

/* Cache Line Size: Default 00 = 32 bytes */   
#define     ESAL_CO_MEM_L1CFG0_ICBSIZE          (ESAL_GE_MEM_32BIT_SET(7) | ESAL_GE_MEM_32BIT_SET(8))

/* Cache Replacement Policy: Default 01 = Pseudo LRU */   
#define     ESAL_CO_MEM_L1CFG0_ICREPL           (ESAL_GE_MEM_32BIT_SET(9) | ESAL_GE_MEM_32BIT_SET(10))

/* Cache Locking Available: Default 1 = Available */
#define     ESAL_CO_MEM_L1CFG0_ICLA             ESAL_GE_MEM_32BIT_SET(11)

/* Cache Parity Available: Default 1 = Available */
#define     ESAL_CO_MEM_L1CFG0_ICPA             ESAL_GE_MEM_32BIT_SET(12)

/* Cache number of ways (111=8) */   
#define     ESAL_CO_MEM_L1CFG0_ICNWAY_MASK      0x0007F800

/* Cache Size mask (32 KBytes=0x20) */         
#define     ESAL_CO_MEM_L1CFG0_ICSIZE_MASK      0x000007FF

#if (ESAL_CO_CACHE_AVAILABLE == NU_TRUE)

/* This macro invalidates all of the cache at the core level. */
#define         ESAL_CO_MEM_CACHE_ALL_INVALIDATE()                          \
                {                                                           \
                    INT     l1csr_val;                                      \
                                                                            \
                    /* Invalidate all DCache on the core */                 \
                    ESAL_TS_RTE_SPR_READ(ESAL_CO_MEM_L1CSR0_REG_ID,         \
                                         &l1csr_val);                       \
                    l1csr_val |= ESAL_CO_MEM_L1CSR0_CFI_BIT;                \
                    ESAL_TS_RTE_MSYNC_EXECUTE();                            \
                    ESAL_TS_RTE_ISYNC_EXECUTE();                            \
                    ESAL_TS_RTE_SPR_WRITE(ESAL_CO_MEM_L1CSR0_REG_ID,        \
                                          l1csr_val);                       \
                    ESAL_TS_RTE_MSYNC_EXECUTE();                            \
                    ESAL_TS_RTE_ISYNC_EXECUTE();                            \
                                                                            \
                    /* Invalidate all ICache on the core */                 \
                    ESAL_TS_RTE_SPR_READ(ESAL_CO_MEM_L1CSR1_REG_ID,         \
                                         &l1csr_val);                       \
                    l1csr_val |= ESAL_CO_MEM_L1CSR1_ICFI_BIT;               \
                    ESAL_TS_RTE_SPR_WRITE(ESAL_CO_MEM_L1CSR1_REG_ID,        \
                                          l1csr_val);                       \
                                                                            \
                    /* Execute sync instructions */                         \
                    ESAL_TS_RTE_MSYNC_EXECUTE();                            \
                    ESAL_TS_RTE_ISYNC_EXECUTE();                            \
                }

/* This macro invalidates all of the instruction cache at the core level. */
#define         ESAL_CO_MEM_ICACHE_ALL_INVALIDATE()                         \
                {                                                           \
                    INT     l1csr_val;                                      \
                                                                            \
                    /* Invalidate all ICache on the core */                 \
                    ESAL_TS_RTE_SPR_READ(ESAL_CO_MEM_L1CSR1_REG_ID,         \
                                         &l1csr_val);                       \
                    l1csr_val |= ESAL_CO_MEM_L1CSR1_ICFI_BIT;               \
                    ESAL_TS_RTE_SPR_WRITE(ESAL_CO_MEM_L1CSR1_REG_ID,        \
                                          l1csr_val);                       \
                                                                            \
                    /* Execute sync instructions */                         \
                    ESAL_TS_RTE_MSYNC_EXECUTE();                            \
                    ESAL_TS_RTE_ISYNC_EXECUTE();                            \
                }

/* This macro invalidates all of the data cache at the core level. */
#define         ESAL_CO_MEM_DCACHE_ALL_INVALIDATE()                         \
                {                                                           \
                    INT     l1csr_val;                                      \
                                                                            \
                    /* Invalidate all DCache on the core */                 \
                    ESAL_TS_RTE_SPR_READ(ESAL_CO_MEM_L1CSR0_REG_ID,         \
                                         &l1csr_val);                       \
                    l1csr_val |= ESAL_CO_MEM_L1CSR0_CFI_BIT;                \
                    ESAL_TS_RTE_MSYNC_EXECUTE();                            \
                    ESAL_TS_RTE_ISYNC_EXECUTE();                            \
                    ESAL_TS_RTE_SPR_WRITE(ESAL_CO_MEM_L1CSR0_REG_ID,        \
                                          l1csr_val);                       \
                                                                            \
                    /* Execute sync instructions */                         \
                    ESAL_TS_RTE_MSYNC_EXECUTE();                            \
                    ESAL_TS_RTE_ISYNC_EXECUTE();                            \
                }

/* This macro invalidates all instruction cache for the specified address
   range at the core level. */
#define         ESAL_CO_MEM_ICACHE_INVALIDATE(addr, size)                   \
                {                                                           \
                    INT     cache_block = 0;                                \
                                                                            \
                    do                                                      \
                    {                                                       \
                        /* Execute data block invalidate. */                \
                        ESAL_TS_RTE_ICBI_EXECUTE(addr, cache_block);        \
                                                                            \
                        /* Execute sync instructions */                     \
                        ESAL_TS_RTE_MSYNC_EXECUTE();                        \
                        ESAL_TS_RTE_ISYNC_EXECUTE();                        \
                                                                            \
                        /* Go to next cache block. */                       \
                        cache_block += ESAL_CO_MEM_BLOCK_SIZE;              \
                                                                            \
                    } while(cache_block < size);                            \
                }

/* This macro invalidates all data cache for the specified address
   range at the core level. */
#define         ESAL_CO_MEM_DCACHE_INVALIDATE(addr, size)                   \
                {                                                           \
                    INT     cache_block = 0;                                \
                                                                            \
                    do                                                      \
                    {                                                       \
                        /* Execute data block invalidate. */                \
                        ESAL_TS_RTE_DCBI_EXECUTE(addr, cache_block);        \
                                                                            \
                        /* Execute sync instructions */                     \
                        ESAL_TS_RTE_MSYNC_EXECUTE();                        \
                        ESAL_TS_RTE_ISYNC_EXECUTE();                        \
                                                                            \
                        /* Go to next cache block. */                       \
                        cache_block += ESAL_CO_MEM_BLOCK_SIZE;              \
                                                                            \
                    } while(cache_block < size);                            \
                }

/* This macro flushes all data cache to physical memory (writeback cache)
   and invalidates all data cache entries at the core level. */
#define         ESAL_CO_MEM_DCACHE_ALL_FLUSH_INVAL()                            \
                {                                                               \
                    /* Flush and invalidate entire data cache */                \
                    ESAL_CO_MEM_DCACHE_FLUSH_INVAL(0, ESAL_CO_MEM_DCACHE_SIZE); \
                }

/* This macro flushes all data cache to physical memory (writeback cache)
   for the given address range, then invalidates all data cache entries
   at the core level. */
#define         ESAL_CO_MEM_DCACHE_FLUSH_INVAL(addr, size)                  \
                {                                                           \
                    INT     cache_block = 0;                                \
                                                                            \
                    do                                                      \
                    {                                                       \
                        /* Execute data block flush. */                     \
                        ESAL_TS_RTE_DCBF_EXECUTE(addr, cache_block);        \
                                                                            \
                        /* Execute sync instructions */                     \
                        ESAL_TS_RTE_MSYNC_EXECUTE();                        \
                        ESAL_TS_RTE_ISYNC_EXECUTE();                        \
                                                                            \
                        /* Execute data block invalidate. */                \
                        ESAL_TS_RTE_DCBI_EXECUTE(addr, cache_block);        \
                                                                            \
                        /* Execute sync instructions */                     \
                        ESAL_TS_RTE_MSYNC_EXECUTE();                        \
                        ESAL_TS_RTE_ISYNC_EXECUTE();                        \
                                                                            \
                        /* Go to next cache block. */                       \
                        cache_block += ESAL_CO_MEM_BLOCK_SIZE;              \
                                                                            \
                    } while(cache_block < size);                            \
                }

/* This macro disables instruction and data cache */
#define         ESAL_CO_MEM_CACHE_DISABLE()                                                     \
                {                                                                               \
                    INT     l1csr_val;                                                          \
                                                                                                \
                    /* Disable both I & D cache. Read the DCache and ICache Control and Status  \
                       registers and AND out the Cache Enable and Parity enable bits then       \
                       write the updated value back */                                          \
                    ESAL_TS_RTE_SPR_READ(ESAL_CO_MEM_L1CSR0_REG_ID, &l1csr_val);                \
                    l1csr_val &= ~(ESAL_CO_MEM_L1CSR0_CE_BIT | ESAL_CO_MEM_L1CSR0_CPE_BIT);     \
                    ESAL_TS_RTE_MSYNC_EXECUTE();                                                \
                    ESAL_TS_RTE_ISYNC_EXECUTE();                                                \
                    ESAL_TS_RTE_SPR_WRITE(ESAL_CO_MEM_L1CSR0_REG_ID, l1csr_val);                \
                    ESAL_TS_RTE_ISYNC_EXECUTE();                                                \
                                                                                                \
                    ESAL_TS_RTE_SPR_READ(ESAL_CO_MEM_L1CSR1_REG_ID, &l1csr_val);                \
                    l1csr_val &= ~(ESAL_CO_MEM_L1CSR1_ICE_BIT | ESAL_CO_MEM_L1CSR1_ICPE_BIT);   \
                    ESAL_TS_RTE_SPR_WRITE(ESAL_CO_MEM_L1CSR1_REG_ID, l1csr_val);                \
                    ESAL_TS_RTE_ISYNC_EXECUTE();                                                \
                }

#endif  /* ESAL_CO_CACHE_AVAILABLE == NU_TRUE */

/*********************************************
* PPC E500 Architecture MMU definitions
*********************************************/

/* This macro disables the MMU */
#define ESAL_AR_MMU_DISABLE()                       \
    {                                               \
        /* Not available in this architecture */	\
    }

#endif  /* PPC_MMU_E500_DEFS_H */
