/**************************************************************************
*
*               Copyright Mentor Graphics Corporation 2005
*                         All Rights Reserved.
*
* THIS WORK CONTAINS TRADE SECRET AND PROPRIETARY INFORMATION WHICH IS
* THE PROPERTY OF MENTOR GRAPHICS CORPORATION OR ITS LICENSORS AND IS
* SUBJECT TO LICENSE TERMS.
*
***************************************************************************

***************************************************************************
*
* FILE NAME
*
*   usb_input.c
*
* DESCRIPTION
*
*   This file contains the usb input grafix driver functions for mouse 
*   and keyboard.
*
* DATA STRUCTURES
*
*   None
*
* FUNCTIONS
*
*   USB_Mouse_Event_Handler     This function handles the events generated 
*                               by the mouse.
*   USB_Mouse_Wakeup            This function is the mouse's main device 
*                               driver.
*   USB_Mouse_Pos               This function updates the position of the 
*                               coordinates.
*   USB_Keyboard_Wakeup         This function is the keyboard's main device
*                               driver. 
*
* DEPENDENCIES
*
*   usb_input.h 
*
**************************************************************************/
#include    "drivers/usb_input.h"

/*=============== Globals ==============================*/
static mouseRcd         *USB_Mouse_Rcd;
extern struct _mouseRcd *curInput;


/**************************************************************************
* FUNCTION
*
*    USB_Mouse_Event_Handler
*
* DESCRIPTION
*
*    This function handles the events generated by the mouse.
*
* INPUTS
*
*    cb     Pointer to the mouse control block.
*    handle Handle to the device pointer.
*    event  Pointer to the mouse event.
*
* OUTPUTS
*
*    NU_SUCCESS                 Success.
*
**************************************************************************/
STATUS USB_Mouse_Event_Handler (NU_USBH_MOUSE * cb, VOID *handle,
                                        const NU_USBH_MOUSE_EVENT *event)
{
    INT32 xpos, ypos;
    INT32 x;
    INT32 y;
    INT32 buttons;

    /* Get Mouse coordinates and state. */
    PD_ReadMouse( &x, &y, &buttons);

    /* Scale x and y coordinates. */
    xpos = x + (signed char)(event->x);
    ypos = y  + (signed char)(event->y);

    /* Limit x and y coordinates within the limits. */
    if( xpos < USB_Mouse_Rcd->mrLimits.Xmin)
    {
        xpos = USB_Mouse_Rcd->mrLimits.Xmin;
    }
    else if( xpos > USB_Mouse_Rcd->mrLimits.Xmax)
    {
        xpos = USB_Mouse_Rcd->mrLimits.Xmax;
    }

    if( ypos < USB_Mouse_Rcd->mrLimits.Ymin)
    {
        ypos = USB_Mouse_Rcd->mrLimits.Ymin;
    }
    else if( ypos > USB_Mouse_Rcd->mrLimits.Ymax)
    {
        ypos = USB_Mouse_Rcd->mrLimits.Ymax;
    }

    /*  Check Left Mouse Button */
    if(NU_USBH_LEFT_BUTTON_STATE(event->buttons))
    {
        /* If Left button is pressed, indicate the event in the mouse
         * record structure as button pressed.
         */
        if(!buttons)
        {
            USB_Mouse_Rcd->mrEvent.eventType = mPRESS;
            USB_Mouse_Rcd->mrEvent.eventButtons = 1;

            /* Call the call back routine. */
    	    curInput->mrCallBack(USB_Mouse_Rcd);
        }
    }
    else if(buttons)
    {
        USB_Mouse_Rcd->mrEvent.eventType = mREL;
        USB_Mouse_Rcd->mrEvent.eventButtons = 0;

        /* Call the call back routine. */
    	curInput->mrCallBack(USB_Mouse_Rcd);
    }
    else
    {
        USB_Mouse_Rcd->mrEvent.eventType = 0;
        USB_Mouse_Rcd->mrEvent.eventButtons = 0;
    }

    /* Have we really moved? */
    if((USB_Mouse_Rcd->mrEvent.eventX != xpos) ||
       (USB_Mouse_Rcd->mrEvent.eventY != ypos))
    {
        /* Update position of mouse pointer.*/
        USB_Mouse_Rcd->mrEvent.eventType = mPOS;
        USB_Mouse_Rcd->mrEvent.eventX = xpos;
        USB_Mouse_Rcd->mrEvent.eventY = ypos;

        /* Call the call back routine. */
    	curInput->mrCallBack(USB_Mouse_Rcd);
    }

    return NU_SUCCESS;
}

/**************************************************************************
* FUNCTION
*
*   USB_Mouse_Wakeup
*
* DESCRIPTION
*
*   This function is the mouse's main device driver.
*
* INPUTS
*
*   rcd        Mouse record.
*   md         Command: IMOPEN, IMCLOSE.
*
* OUTPUTS
*
*   ret_val    Return value of the open, close, and pos functions.
*
**************************************************************************/
INT32 USB_Mouse_Wakeup(mouseRcd *rcd, INT32 md)
{
    /* Save the mouse input device record in the local variable. */ 
    USB_Mouse_Rcd = rcd;

	/* Save the mouse default position.*/
    USB_Mouse_Pos(rcd);

#if (CFG_NU_OS_UI_GRAFIXRS_CURSOR_SUPPORT == NU_TRUE)
    ShowCursor();
#endif

    return (NU_SUCCESS);
}

/**************************************************************************
* FUNCTION
*
*    USB_Mouse_Pos
*
* DESCRIPTION
*
*   This function updates the position of the mouse
*   co-ordinates.
*
* INPUTS
*
*    rcd    Pointer to the mouse record.
*
* OUTPUTS
*
*    NU_SUCCESS
*
**************************************************************************/
INT32 USB_Mouse_Pos(mouseRcd *rcd)
{
    UINT16  xpos,ypos;

    /* Read current X and Y position of mouse. */
    xpos = rcd->mrEvent.eventX;
    ypos = rcd->mrEvent.eventY;

    /* Limit the coordinates within the limits. */
    if( xpos < rcd->mrLimits.Xmin)
    {
        xpos = rcd->mrLimits.Xmin;
    }
    if(xpos > rcd->mrLimits.Xmax)
    {
        xpos = rcd->mrLimits.Xmax;
    }
    if(ypos < rcd->mrLimits.Ymin)
    {
        ypos = rcd->mrLimits.Ymin;
    }
    if(ypos > rcd->mrLimits.Ymax)
    {
        ypos = rcd->mrLimits.Ymax;
    }

    return ( NU_SUCCESS );
}

/**************************************************************************
* FUNCTION
*
*    USB_Keyboard_Wakeup
*
* DESCRIPTION
*
*    This function is the keyboard's main device driver.
*
* INPUTS
*
*    None
*
* OUTPUTS
*
*    ret_val - Return value of the open and close functions.
*
**************************************************************************/
INT32 USB_Keyboard_Wakeup(VOID)
{
    return (NU_SUCCESS);
}

/*======================= File End ====================================*/

