/*=============================================================================
 *     Copyright Texas Instruments, Inc., 2002.  All Rights Reserved.
 */

/**
  Simple but often used functions

 */



/**

 This function is a simple way to execute a particular operation one
 time for every element or every combination of elements in one or
 more arrays.


 A simple example: the following code...
 forEach( [ "FIRST", "SECOND" ], function ( value )
 {
    pkg.output.writeln(" Saw value = " + value );
 });

 ...produces output:
 Saw value = FIRST
 Saw value = SECOND


 A more complex example: the following code...
 forEach( [ "GOOD", "EVIL" ], [ "DEMON", "DEITY" ], function ( polarity, figure )
 {
    pkg.output.writeln(" Saw a " + polarity + " " + figure );
 });

 ...produces output:
 Saw a GOOD DEMON
 Saw a GOOD DEITY
 Saw a EVIL DEMON
 Saw a EVIL DEITY


 You can therefore provide as many combinations of arrays as you
 want.  However, this operation will fail if the function you provide
 does not accept one value per array of values that you provide.

 */
public function forEach( args... )
{

  /**
      A test operation to verify that this works correctly.  To run
      the test, type at an os prompt:

      (new pkg.util.forEach()).test()
   */
  public function test()
  {
    pkg.output.writeln("__EXPLAIN__   You should see the following:");
    pkg.output.writeln("Saw value = FIRST");
    pkg.output.writeln("Saw value = SECOND");
    pkg.output.writeln("__TEST__     Running test");
    forEach( [ "FIRST", "SECOND" ], function ( value )
    {
       pkg.output.writeln("Saw value = " + value );
    });

    pkg.output.writeln("__EXPLAIN__   You should see the following:");
    pkg.output.writeln("Saw a GOOD DEMON");
    pkg.output.writeln("Saw a GOOD DEITY");
    pkg.output.writeln("Saw a EVIL DEMON");
    pkg.output.writeln("Saw a EVIL DEITY");
    pkg.output.writeln("__TEST__     Running test");
    forEach( [ "GOOD", "EVIL" ], [ "DEMON", "DEITY" ], function ( polarity, figure )
    {
       pkg.output.writeln("Saw a " + polarity + " " + figure );
    });

    pkg.output.writeln("__EXPLAIN__   You should see the following:");
    pkg.output.writeln("Look! a HAIRY PRETTY MAN");
    pkg.output.writeln("Look! a HAIRY PRETTY WOMAN");
    pkg.output.writeln("Look! a HAIRY UGLY MAN");
    pkg.output.writeln("Look! a HAIRY UGLY WOMAN");
    pkg.output.writeln("Look! a BALD PRETTY MAN");
    pkg.output.writeln("Look! a BALD PRETTY WOMAN");
    pkg.output.writeln("Look! a BALD UGLY MAN");
    pkg.output.writeln("Look! a BALD UGLY WOMAN");
    pkg.output.writeln("__TEST__     Running test");
    forEach(
        [ "HAIRY", "BALD" ]
      , [ "PRETTY", "UGLY" ]
      , [ "MAN", "WOMAN" ]
      , function ( hairState, beauty, gender )
    {
       pkg.output.writeln("Look! a " + hairState + " " + beauty + " " + gender );
    });

    pkg.output.writeln("__FINISH__     Did it work?");

  }

  if (args.length() < 1) return;

  const var arrayOfValueSets = args[0..args.length() - 2];
  const var theFunc = args[ args.length() - 1 ];

  const var retval = forEachHelper( [], arrayOfValueSets, theFunc );

  if (retval == false)
  {
    pkg.output.logWarning(" Early return from forEach ");
  }

  return retval;
}


/**
  A helper function that does the recursion to support the
  complex form forEach command.

  This function returns false if the test is to be terminated
  early- the user provided function determines this by
  returning false.
 */
function forEachHelper(
    preparedArgSequence  // First n arguments to the function
  , arrayOfValueSets     // array of arrays of values for other function parms
  , theFunc              // the function to call
  )
{
  if (arrayOfValueSets.length() <= 0)
  {
    const var retval = theFunc.callAsFunction( preparedArgSequence );
    if (retval==false)
    {
      pkg.output.logWarning( "Early exit from foreach with params " + preparedArgSequence );
      return false;
    }
    return true;
  }
  else
  {
    for (var i = 0; i < arrayOfValueSets[0].length(); ++i)
    {
      const var newArgSequence = [];

      // Create new copy of arg sequence
      for (var j = 0; j < preparedArgSequence.length(); ++j)
      {
        newArgSequence[ j ] = preparedArgSequence[ j ];
      }

      // Add the current element
      newArgSequence[ newArgSequence.length() ] = arrayOfValueSets[0][i];

      // Create the new value set
      const var newValueSetArray =
          (arrayOfValueSets.length() < 2)
        ? []
        : arrayOfValueSets[ 1 .. arrayOfValueSets.length() - 1 ];

      const var result = forEachHelper(
          newArgSequence
        , newValueSetArray
        , theFunc
        );

      if (result == false)
      {
        return false;
      }
    }
  }

  return true;
}


