/*=============================================================================
 *     Copyright Texas Instruments, Inc., 2002.  All Rights Reserved.
 */

/**
  A library for managing comparing lists of expected results and actual
  results that may be generated at different times.

  In the course of test script development, I noticed a common pattern
  to my coding- I would often type the same code multiple times in order
  to first run a specific test and then to display specifically how it
  failed.  This is a simple library for managing these expectations.

  @author Brad Hyslop

  @version 1
 */


public var expect = new (function() {


  /**
      A unit test function and an example of how to use this comparison mechanism.
   */
  public function test()
  {
    var elist = new ExpectationList();
    var rlist = new java.util.LinkedList();

    /* Add a boolean true result and tests depending upon that */
    rlist.add( new java.lang.Boolean( true ) );
    elist.addExpectation(
        new Expectation(
            new ExpectationClause( "result", "==true", "PASS Expected" )
          , new ExpectationClause( "!result","", "FAIL Expected" )
          , new ExpectationClause( "!result == false","","PASS Expected" )
          , new ExpectationClause( "result == true" )
          , new ExpectationClause( "!result == false" )
          , new ExpectationClause( "result"," == false", "FAIL Expected " )
        )
      );

    /* Add a numerical result and tests depending upon that */
    rlist.add( 5 );
    elist.addExpectation(
        new Expectation(
            new ExpectationClause( "result", " == 5", "PASS Expected" )
          , new ExpectationClause( "result", " > 4", "PASS Expected" )
          , new ExpectationClause( "result", " == 7", "FAIL Expected" )
        )
      );

    /* Use the advanced interface to add a bool result and test it */
    rlist.add( new java.lang.Boolean( true ) );
    elist.add(
        ["result" ]
      , ["result", "==true", " PASS Expected" ]
      , ["!result", "==false", "PASS Expected" ]
      );

    /* Use the advanced interface to add a bool result and test it */
    rlist.add( 123 );
    elist.add(
        ["result", "==123"]
      , ["result", ">122"]
      , ["result*5", "==615"]
      );

    elist.evaluate( rlist, pkg.output.logError, pkg.output.logInfoMsg );

    return true;

  }


  /**
      A class that holds a single expectation clause.  An expectation clause
      is specified by 1, 2, or 3 strings passed as var args to this object.

      @param   args[0]    This is the clause which is evaluated to produce a
                          partial result. The magic word 'result' will be
                          effectively substituted by the particular object
                          that is involved in the compare.  Once evaluated,
                          this value is stored.

      @param   args[1]    If the preceeding clause does not return a boolean
                          result, the caller will have to provide this string.
                          This string is appended to the partial result in
                          order to create a boolean result.

      @param   args[2]    This is a comment string which is appended to the
                          string sent to the appropriate writer.  This is an
                          optional param.

   */
  public function ExpectationClause( args... )
  {
    const var evaluatable        = args[0]==undefined || args[0]==null ? "false" : args[0];
    const var appendedComparison = args[1]==undefined || args[1]==null ? ""      : args[1];
    const var comment            = args[2]==undefined || args[2]==null ? ""      : args[2];


    /**
        toString method used mostly for debugging this service.
     */
    public function toString()
    {
      return
          "ExpectationClause {"
        + "comparison=" + evaluatable
        + appendedComparison
        + " ces=" + commentEvalString
        + "}\n";
    }


    /**
        A method that evaluates this ExpectationClause given a particular
        result object.  In addition to returning true or false, this method
        also displays output to the appropriate stream depending upon the
        result.

        @param  resultObject    The object to substitute for 'result' in
                                the comparison clause during the evaluation

        @param  errorWriter     A writer to which a failing result is sent.
                                This can be null to suppress output.

        @param  infoWriter      A writer to which a passing result is sent.
                                This can be null to suppress output.
     */
    public function evaluate( resultObject, errorWriter, infoWriter )
    {
      const var result = resultObject;
      const var partialEvaluation;
      eval( "partialEvaluation = " + evaluatable + ";" );
      var testSuccess;

      if (partialEvaluation == undefined)
      {
        errorWriter( "ERROR: Unevaluatable result '" + evaluatable );
        testSuccess = false;
      }
      else
      {
        eval( "testSuccess = partialEvaluation" + appendedComparison + ";" );
      }

      const var logString =
          "evaluating condition {" + evaluatable + appendedComparison
        + "} or equivalently {" + partialEvaluation + appendedComparison + "}   "
        + comment;

      if (testSuccess == null)
      {
        errorWriter( "ERROR: Unevaluatable expression '" + evaluatable + testSuccess );
      }
      else
      {
        if ((!testSuccess) && (errorWriter != null))
        {
          errorWriter( "FAIL: " + logString );
        }

        if (testSuccess && (infoWriter != null))
        {
          infoWriter( "PASS: " + logString );
        }
      }


      return testSuccess==true?true:false;
    }

  }


  /**
      An object that contains a set of ExpectationClauses that are to all
      be evaluated against a particular result object.  The constructor
      takes a parameter list of zero or more parameters which are
      ExpectationClause objects.
   */
  public function Expectation( ear... )
  {

    var expectationClauseArray = [];
    for (var i = 0; i < ear.length(); ++i)
    {
      expectationClauseArray[ i ] = ear[ i ];
    }


    /**
        toString method used mostly for debugging this service.
     */
    public function toString()
    {
      var retstring = "Expectation {\n";
      for (var i = 0; i < expectationClauseArray.length(); ++i)
      {
        retstring += expectationClauseArray.toString();
      }
      retstring += "}\n";

      return retstring;
    }


    /**
      Add an ExpectationClause instance to the list which is managed
      by this Expectation.
     */
    public function addClause( expectationClause )
    {
      expectationClauseArray[ expectationClauseArray.length() ] = expectationClause;
    }


    /**
      Cause the set of ExpecationClauses managed by this Expectation
      to be evaluated against the designated object.

      @param  resultObject    The object to substitute for 'result' in
                              the appropriate strings in the ExpectationClause

      @param  errorWriter     A writer to which a failing result is sent.
                              This can be null to suppress output.

      @param  infoWriter      A writer to which a passing result is sent.
                              This can be null to suppress output.
     */
    public function evaluate( resultObject, errorWriter, infoWriter )
    {
      var retval = true;
      for (var i = 0; i < expectationClauseArray.length(); ++i)
      {
        retval = expectationClauseArray[i].evaluate( resultObject, errorWriter, infoWriter )
            && retval;
      }

      infoWriter( "--" );

      return retval;
    }
  }


  /**
    Manages a list of Expectations.  Given a matching list of
    arbitrary result objects, instances of this class can evaluate
    many results in one shot.
   */
  public function ExpectationList()
  {
    var expectationList =
      java.util.Collections.synchronizedList( new java.util.LinkedList() );


    /**
        Append an expectation to the list maintained by this object.
     */
    public function addExpectation( expectation )
    {
      expectationList.add( expectation );
    }


    /**
        Append an expectation to the list maintained by this object.
     */
    public function clear()
    {
      expectationList.clear();
    }


    /**
        Given a list of result objects, this method evaluates all
        of those results against the list it manages of Expectations.
     */
    public function evaluate( resultList, errorWriter, infoWriter )
    {
      var retval = true;

      /* A hack since Collections.SynchronizedList doesn't clone... */
      const var safeResultList = new java.util.LinkedList();
      while (safeResultList.size() < resultList.size())
      {
        safeResultList.addLast( resultList.get( safeResultList.size() ) );
      }

      var resultIterator = safeResultList.iterator();
      var expectIterator = expectationList.iterator();
      while (resultIterator.hasNext() && expectIterator.hasNext())
      {
        const var expectItem = expectIterator.next();
        const var resultItem = resultIterator.next();

        retval = expectItem.evaluate(
            resultItem
          , errorWriter
          , infoWriter
          ) && retval;
      }

      if (resultIterator.hasNext() || expectIterator.hasNext())
      {
        retval = false;
        errorWriter(
            " Mismatch between expectation list length ("
          + expectationList.size() + ") and result list length ("
          + resultList.size() + ")"
          );
      }

      return retval;
    }


    /**
      This is a convenience operation for creating an Expectation
      to be appended to this list.  I recommend looking at the
      test case and understanding the regular 'add' method and
      then looking at the examples in the test case to see how
      this method is used.

      @param stringArray  a vararg where each element is an
                          array of strings where each of those
                          sub arrays' member sets is a valid
                          parameter set for creating an
                          ExpectationClause.  (Just see the
                          example, it makes more sense that
                          way.)
     */
    public function add( stringArray... )
    {
      var expectation = new Expectation( );
      for (var i = 0; i < stringArray.length(); ++i)
      {
	var args = [];
	for (var j = 0; j < 3; ++j)
	{
	  args[ j ] =
               (j < stringArray[ i ].length()) 
             ? stringArray[ i ][ j ]
             : null;
	}

        var ec = new ExpectationClause( args[0], args[1], args[2] );

        expectation.addClause( ec );
      }

      addExpectation( expectation );
    }

  }

}) ();



