/*=============================================================================
 *     Copyright Texas Instruments, Inc., 2004.  All Rights Reserved.
 */


/**
 * A simple utility that subscribes to the registry node /ESF/Connections
 * to be notified when a connection to the target is established/broken.  It
 * sends email to someone when the connection is broken.  This might be handy
 * for unattended (non-)regression tests, because it will let someone know
 * that the test station needs attention.  Unfortunately this simple bit of
 * script isn't really smart enough to know the difference between a disconnect
 * caused by a target crash, vs. a disconnect when the tests are completed,
 * so you need to tell it to stop monitoring the connection by calling
 * {@link #stop} when your test is completed.  For example:
 * <pre>
 *    public function runTest()
 *    {
 *      try
 *      {
 *        pkg.util.connectionMonitor.start("joe.user@ti.com");
 *        ... test body ...
 *      }
 *      finally
 *      {
 *        pkg.util.connectionMonitor.stop();
 *      }
 *    } 
 * </pre>
 * <p>
 * For the curious, there is some documentation of the registry in the
 * <a href="http://objectscript.sourceforge.net/?docs/cdg.html">Chimera
 * Developer's Guide</a>, but unfortunately at this point it is not quite
 * complete.
 */
public var connectionMonitor = new function(){
  
  private var subscriber = null;
  
  /**
   * Start monitoring the connection, sending email to the specified address
   * if the connection is broken.
   * 
   * @param emailAddr   the email address to send to when connection is broken
   */
  public function start(emailAddr)
  {
    if( subscriber != null )
      stopMonitoringConnection();
    
    /**
     * The data associated with a "directory node" in the registry is called a
     * directory-table.  Keep track of the last directory-table, so when it
     * changes (either due to addition or deletion of a child), we can compare
     * the new dt to the old dt.
     */
    var lastDt = null;
    
    /**
     * Subscribe to node in registry... provided callback function is called
     * whenever this node changes.  The second arg to the function is the
     * new value, in this case a new directory-table
     */
    registry.subscribeToValue( "/ESF/Connections", null, subscriber = function( node, newDt ) {
      
      // check for removed connections:
      if( lastDt != null )
      {
        for( var itr=lastDt.notIn(newDt); itr.hasNext(); )
        {
          var connectionName = itr.next();
          
          // connectionName will be something like "USB".. this is the name you
          // see in the "Connection Chooser" view
          
          var email = new pkg.email.EmailMessage();
          
          email.setSender(emailAddr);  // use destination address as "From" address too
          email.setSubject(connectionName + " has disconnected");
          email.setBody("This is a message to inform you that the Test Harness connection " + connectionName + " has disconnected, perhaps due to a target crash?");
          email.addRecipient(emailAddr);
          
          try
          {
            var server = registry.resolve("/Preferences/Talkback/SMTP Server").getValue();
            if( (server != null) && (server != "") )
              server = "smtp.sand.design.ti.com";
            (new pkg.email.Emailer( server, 25 )).sendMessage(email);
          }
          catch(e)
          {
            e = unwrapException(e);
            return services["prompt"].showErrorMessage("error mailing report:\n" + e.getMessage());
          }
          
        }
      }
      
      // check for new connections:
      for( var itr=newDt.notIn(lastDt); itr.hasNext(); )
      {
        var connectionName = itr.next();
        
        // well, we don't really care about new connections, so this is a no-op
      }
      
      lastDt = newDt;
      
    } );
    
  }  
  
  /**
   * Stop monitoring the connection.
   */
  public function stop()
  {
    if( subscriber != null )
    {
      registry.unsubscribeFromValue(subscriber);
      subscriber = null;
    }
  }
  
  
}();