/*=============================================================================
 *     Copyright Texas Instruments, Inc., 2004.  All Rights Reserved.
 */


/**
 * A class extended from TestCase base class to be used
 * as a base class. This has additional functionality
 * - testDescMap variable handling, 
 *   
 * @param name           This is the name of the test case
 * @param args           zero or more additional args, for backwards 
 *    compatibility with base classes that try to use the old constructor
 */
public function TestSuite( name )
        extends pkg.tf.TestCase(name)
{
   /**
    * testDesc variable which holds the test case id's
    * It is of type HashMap, the keys are the test case id's
    * and the values are string which is a short description 
    * of the test case
    */
   
   public var testDescMap = new java.util.TreeMap();
    
   /**
    * This defines FAILURE status
    */
   public var STATUS_FAILURE = 0x01;
   
   /**
    * This defines SUCCESS status
    */
   public var STATUS_SUCCESS = 0x00;
   
   /**
    * This defines EXCEPTION status
   */
   public var STATUS_EXCEPTION = -1;  
  
  /*=======================================================================*/
  /**
   * Subclasses of test case should override this method to implement the
   * actual test code.
   *
   */
  public function runTest(id, iterations)
  {
    // Init method to do any one time processing before the test case iterations start
    initHook(id, iterations);
    var parsedId = parseId(id);

    for (var i = 0; i < iterations; i++)
    {

      // method to do any pre processing before the test case script is called
      startHook(id);

      // use reflection here to call the test script method
      var result = this.getMember(parsedId)();

      // method to do any post processing after the test case script is called
      stopHook(id, result);

    }

    // Cleanup method to do any one time post processing after test case iterations are done
    exitHook(id);
  }

  /**
    * Init method to do any one time processing before the test case iterations start
    * This method takes care of any pre-processing
    * to be done before invoking the test case method.
    * This is a dummy method in TestSuite base class. Which needs to be
    * over ridden in the derived class
    * @param id           - This is the testCaseId of the test case to be executed
    * @param iterations   - number of iterations need to be run. 
    */ 
  protected function initHook(id, iterations)
  {
  }

  /**
    * Cleanup method to do any one time post processing after test case iterations are done
    * This is a dummy method in TestSuite base class. Which needs to be
    * over ridden in the derived class
    * @param id - This is the testCaseId of the test case to be executed
    */ 
  protected function exitHook(id)
  {
  }
    
  /**
  * This method takes care of any pre-processing
  * to be done before invoking the test case method.
  * This is a dummy method in TestSuite base class. Which needs to be
  * over ridden in the derived class
  * @param testCaseId - This is the testCaseId of the test case to be
  * executed
  */ 
  protected function startHook(testCaseId)
  {
    
  } 
  
  /**
  * This method takes care of any post-processing
  * to be done before invoking the test case method.
  * This is a dummy method in TestSuite base class. Which needs to be
  * over ridden in the derived class
  * @param testCaseId - This is the testCaseId of the test case to be
  * executed
  * @param result - The result of the test case is passed here. Which can
  * be one of the three: STATUS_FAILURE, STATUS_SUCCESS, or STAUS_EXCEPTION
  */ 
  protected function stopHook(testCaseId, result...)
  {
    
  }

  /**
  * This method returns the key set of TestDesc variable
  * Mainly used to fill up the Regression PlugIn
  */
  public function getTestIds()
  {
    return testDescMap.keySet();
    
  }

  /**
  * This method returns the value which maps to the testCaseId key 
  * in TestDesc Hash Map
  * Mainly used to fill up the Regression PlugIn when the Sequencing 
  * file is dragged and dropped
  */
  public function getTestDesc(testCaseId)
  {
    return testDescMap.get(testCaseId);
  }

  /**
   * This method is a wrapper to populate the testDesc variable
   */
  protected function setTestDesc(key, value)
  {
    if (!testDescMap.containsKey(key))
    {
      testDescMap.put(key,value);
    }
    else
      throw new IllegalArgumentException("duplicate key");
  }
  
  /**
  * This method provides the means to run test in 'manual mode'
  * The list of values in testDesc HashMap are displayed here
  * The user is prompted to select one test case. Then the runTest
  * method is called using the testCaseId which was selected by the user
  */
  public function manualMode()
  {
    var choice =  displayTestChoices();

    if (choice[0] != -1)
      runTest(choice[0],choice[1]);
    else
      logError("Error getting the testCaseId");
    
  }

  /** 
  * Following are some utility functions to aid the above methods
  */
  private function parseId(testCaseId)
  {
    var parsedId = replaceInString(testCaseId,".", "_");
    return parsedId;
  }

  public function replaceInString(inputString, inchar, outchar)
  {
    var incharIdxPrev = inputString.indexOf(inchar);
    var incharIdxNext= incharIdxPrev;
    var outputString = "";
    var firstrun = true;
    
    // while loop to end when there are no more 'inchar' type of
    // characters
    while (incharIdxNext != -1)
    {
      // if we have entered this loop for the first time
      // this means incharIdxPrev points to the first occurence
      // of 'inchar' and incharIdxNext is same as incharIdxPrev
      if (firstrun)
      {
        outputString = inputString.substring(0, incharIdxPrev);
        firstrun = false;
      }
      else
      {
        outputString += outchar + inputString.substring(incharIdxPrev+1, incharIdxNext);
        incharIdxPrev = incharIdxNext;
      }
      // a.indexOf(char, idx) throws an exception if idx == a.length()
      // Need to ensure that error scenario is handled
      if (incharIdxPrev == inputString.length()-1)
      {
        outputString += outchar;
        break;
      }
      else
      {
        incharIdxNext = inputString.indexOf(inchar, incharIdxPrev+1);
      }
      
      if (incharIdxNext == -1)
      {
        outputString += outchar + inputString.substring(incharIdxPrev+1);
        break;
      }
    }
    
    // If the inputstring didn't have any 'inchar' type of characters
    if (firstrun)
    {
      return inputString;
    }
    else
    {
      return outputString;
    }
  }
  protected function displayTestChoices()
  {
    var choices = [];
    var choice;
    var size;
    var keySet;
    var keySetIter;

    size = testDescMap.size();
    keySet = testDescMap.keySet();
    keySetIter = keySet.iterator();

    for (var idx = 0; idx< size; idx++)
    {
      var temp = keySetIter.next();
      choices[idx] = temp;

      pkg.output.writeln("Test Choice " + idx + ": " + temp + ":" + testDescMap.get(temp));
    }
    
    // An alternate way to do the above
    /*    
    for( var temp : testDescMap.keySet() )
    {
      pkg.output.writeln("Test Choice " + choices.length() + ": " + temp + ":" +testDescMap.get(temp));
      choices.push(temp);
    } 
    */
   
    var i = 0;
    choice = 0;

    // Give tester three chances to select a test id within a valid range

    while ( ( (choice > size) || (choice < 1) ) && (i <=3) )
    {

      choice = services["prompt"].promptForString( "Please enter the choice of test case you would like to run (1 - "+ size, "1","*" );
      i ++;

      if ( (choice > size ) || (choice < 1) )
        continue;
      else
      {
        var iter = services["prompt"].promptForString( "Please enter the number of iterations", "1","*" );
        var userInput = [];
        userInput[0] = choices[choice];
        userInput[1] = iter;
        return (userInput);
      }
    }

    logError("Error getting the correct testCaseId:"+ choice );
    logError("The test case Id should be between 1 and "+ size);
    return -1;
    
  }
}

/*
 *   Local Variables:
 *   tab-width: 2
 *   indent-tabs-mode: nil
 *   mode: java
 *   c-indentation-style: java
 *   c-basic-offset: 2
 *   eval: (c-set-offset 'statement-cont '0)
 *   eval: (c-set-offset 'substatement-open '0)
 *   eval: (c-set-offset 'case-label '+)
 *   eval: (c-set-offset 'inclass '+)
 *   eval: (c-set-offset 'inline-open '0)
 *   End:
 */
