/*=============================================================================
 *     Copyright Texas Instruments, Inc., 2003.  All Rights Reserved.
 */

/**
 * Base class for test that establishes a connection to the SUT (host/target
 * build).  If the requested connection type does not exist, this test will
 * handle creating it, prompting the user to fill in any required parameters.
 * 
 * @param name   the name of the connection to activate
 * @param type   the SUTConnection type
 */
public function ConnectTo( protected name, protected type )
  extends pkg.tf.TestCase("Connect to " + name + " SUT")
{
  /**
   * entry point to test
   */
  public function runTest()
  {
    var cm = services["esf connection mgr"];
    
    // first ensure that the connection we want to make exists, and is the
    // the correct type:
    if( cm.getConnectionType(name) == null )
    {
      pkg.output.logInfoMsg("connection definition does not exist, so I will create it");
      createConnection();
    }
    else if( cm.getConnectionType(name) != type )
    {
      pkg.output.logInfoMsg("connection exists, but is of the wrong type");
      if( cm.isActive(name) )
      {
        pkg.output.logInfoMsg("deactivating " + name);
        cm.deactivateConnection(name);
      }
      pkg.output.logInfoMsg("removing " + name);
      cm.removeConnection(name);
      createConnection();
    }
    
    // once we get here, the connection exists and is the correct type:
    if( !cm.isActive(name) )
      preConnectHook();
    
    // XXX might need a short delay here to let USB auto-connect?
    
    if( !cm.isActive(name) )
    {
      pkg.output.logInfoMsg("connecting to " + name);
      cm.activateConnection(name);
      java.lang.Thread.sleep(1000);  // XXX ??? 
    }
    
    postConnectHook();
  }
  
  /**
   * A connection of the requested type does not exist, so create it
   */
  private function createConnection()
  {
    services["esf connection mgr"].addConnection( name, type );
    services["registry browser"].showBrowser("/Preferences/ESF/Connections/" + name);
    services["prompt"].showInfoMessage("Please configure the connection " + name + ", and then press \"Ok\" when done");
  }
  
  /**
   * Get the instructions to display to the user to connect to the SUT.  The
   * derived class can override this method to give specific instructions on
   * how to start the SUT.
   * 
   * @return a string, potentially containing HTML markup tags
   */
  protected function getPreConnectInstructions()
  {
    return "start " + name + " SUT, and then press \"Ok\"";
  }
  
  /**
   * Hook to be executed before connecting to SUT.  By default this just
   * shows a message to the user to get them to start the SUT, and then
   * press "Ok" once everything is running, but this could hypothetically
   * be overriden to actually start the SUT for the user
   */
  protected function preConnectHook()
  {
    services["prompt"].showInfoMessage( getPreConnectInstructions() );
  }
  
  /**
   * Hook to be excuted after connection to the SUT.  This should ensure
   * that the target is running.  By default this just sends the START
   * command (if supported by the SUTConnection).
   */
  protected function postConnectHook()
  {
    var cm  = services["esf connection mgr"];
    var sut = cm.getConnection( cm.nodeNameToId(name) );
    if( sut instanceof ti.chimera.esf.sut.EventHandlerControl )
    {
      pkg.output.logInfoMsg("sending START command");
      sut.sendStart();
    }
  }
  
}

