/*=============================================================================
 *     Copyright Texas Instruments 2002. All Rights Reserved.
 */


/*=============================================================================
    Description:
    This test is derived from the TestCase base class.

    Args:
    logWriter        -  A writer object that is used to log any messages.
                        User can either pass in a writer object or null.

    warningWriter    -  A writer object that is used to log any warning messages.
                        User can either pass in a writer object or null.

    errorWriter      -  A writer object that is used to log any error messages
                        User can either pass in a writer object or null.

    Returns:
    n/a

 */
public function ObsUnitTest( logWriter, warningWriter, errorWriter )
  extends pkg.tf.TestCase( "ObsUnitTest", logWriter, warningWriter, errorWriter)
{
  /****************************************************************
   * Global variable initionalizations.
   ****************************************************************/
  var scriptEhId = 0x101;

  var rctx = services["esf routing"].getRoutingContext();


  //====================================================================================
  /**
   * Overrides the runTest() method in TestCase base class.
   *
   */
  public function runTest()
  {
    var prim;

    logInfo( "Beginning Obs Event Handler Unit Test" );
//    logError( "A Test of Error logging" );
//    logWarning( "A Test of Warning logging2" );

    // Make the system quiet
    logInfo( "Resetting..." );
    if ( ! pkg.esf.obs.util.reset( rctx, scriptEhId ) ) return;

    logInfo( "Disabling polling..." );
    prim = rctx.sendAndWaitForPrim(
                new OBS_SET_PULL_PERIOD_REQ(
                  /* source ehId */  scriptEhId,
                  /* pullPeriod */   0
                )
              , rctx.type(OBS_PULL_PERIOD_CNF)
              );
    if (! (prim instanceof OBS_PULL_PERIOD_CNF) )
    {
      /* Dumb check... */
      logError( "Poll disable did not succeed: type is " + prim );
      return;
    }
    if (prim.getPullPeriod() != 0) {
      logError( "Poll value should be 0 but is " + prim.getPullPeriod() );
      return;
    }

    logInfo( "Polling disabled." );

    // Figure out the max observables are allowed in the system
    var statusStruct = pkg.esf.obs.util.getStatusStruct( rctx, scriptEhId );
    const var MAX_OBSERVABLES_ALLOWED  = statusStruct.getObservableMax();
    const var CURRENT_OBSERVABLE_COUNT = statusStruct.getObservableMaxCurrent();

    logInfo( "  Detected " + MAX_OBSERVABLES_ALLOWED + " observables possible" );
    logInfo( "  Current count is " + CURRENT_OBSERVABLE_COUNT + " observables" );

    // Setting and querying tests: obs
    { // verify all disabled
      logInfo( "Disabling all Observation streams..." );
      var i;
      for (i = 1; i < CURRENT_OBSERVABLE_COUNT; ++i)
      {
        if ( ! pkg.esf.obs.util.configureStreamExpecting(
                  rctx
                , scriptEhId
                , i
                , 0
                , 0
                , 0
                , 0
                )
           ) return;
      }
      logInfo( "All Observation streams disabled." );
    }


    {/// enable pull for each, verify, disable
      logInfo( "Enabling push then pull for each and verifying..." );
      var i;
      for (i = 1; i < CURRENT_OBSERVABLE_COUNT; ++i)
      {
        logInfo( "Is observable id " + i + " notify suppressed?" );

        const var observableDef = pkg.esf.obs.util.discoverObservable( rctx, scriptEhId, i );

        const var isNotifySuppressed = observableDef.getNotifySuppressed()>0;

        logInfo( "isNotifySuppressed=" + isNotifySuppressed );

        logInfo( "...disabling all..." );
        if ( ! pkg.esf.obs.util.configureStreamExpecting(
                  rctx
                , scriptEhId
                , i
                , 0
                , 0
                , 0
                , 0
                )
             ) return;

        logInfo( "...enabling push..." );
        if ( ! pkg.esf.obs.util.configureStreamExpecting(
                  rctx
                , scriptEhId
                , i
                , 1
                , 0
                , isNotifySuppressed?0:1
                , 0
                )
           ) return;

        logInfo( "...disabling push..." );
        if ( ! pkg.esf.obs.util.configureStreamExpecting(
                  rctx
                , scriptEhId
                , i
                , 0
                , 0
                , 0
                , 0
                )
           ) return;

        logInfo( "...enabling push2..." );
        if ( ! pkg.esf.obs.util.configureStreamExpecting(
                  rctx
                , scriptEhId
                , i
                , 1
                , 0
                , isNotifySuppressed?0:1
                , 0
                )
           ) return;

        logInfo( "...disabling push and enabling pull..." );
        if ( ! pkg.esf.obs.util.configureStreamExpecting(
                  rctx
                , scriptEhId
                , i
                , 0
                , 1
                , 0
                , 1
                )
           ) return;

        logInfo( "...enabling push and disabling pull..." );
        if ( ! pkg.esf.obs.util.configureStreamExpecting(
                  rctx
                , scriptEhId
                , i
                , 1
                , 0
                , isNotifySuppressed?0:1
                , 0
                )
           ) return;

        logInfo( "...disabling all..." );
        if ( ! pkg.esf.obs.util.configureStreamExpecting(
                  rctx
                , scriptEhId
                , i
                , 0
                , 0
                , 0
                , 0
                )
           ) return;
      }
      logInfo( "Verified toggling of push then pull" );
    }

    { // verify unused observables unconfigurable
      logInfo( "Verify that unused observables are not configurable..." );
      var i;
      for (i = CURRENT_OBSERVABLE_COUNT+1; i < MAX_OBSERVABLES_ALLOWED-1; ++i)
      {
        logInfo( "Checking number " + i );
        if ( ! pkg.esf.obs.util.configureStreamExpecting(
                  rctx
                , scriptEhId
                , i
                , 1
                , 1
                , 0
                , 0
                )
           ) return;
      }
      logInfo( "Unused observables are indeed not configurable..." );
    }
    /// enable zero to maxval time references for each

    // Poll Period test
    /// set poll period, query it, set back to zero (nothing should happen!)
    {
      logInfo( "Checking Poll Period control..." );
      var newPollValue = 12345;
      prim = rctx.sendAndWaitForPrim(
                  new OBS_SET_PULL_PERIOD_REQ(
                    /* source ehId */          scriptEhId,
                    /* pullPeriod */           newPollValue
                  )
                , rctx.type(OBS_PULL_PERIOD_CNF)
                );

      if (newPollValue != prim.getPullPeriod()) {
        logError( "  Expected pullPeriod to be " + newPollValue + " but got " + prim.getPullPeriod() );
        return;
      }

      prim = rctx.sendAndWaitForPrim(
                  new OBS_QUERY_PULL_PERIOD_REQ(
                    /* source ehId */          scriptEhId
                  )
                , rctx.type(OBS_PULL_PERIOD_CNF)
                );

      if (newPollValue != prim.getPullPeriod()) {
        logError( "  Expected pullPeriod to be " + newPollValue + " but got " + prim.getPullPeriod() );
        return;
      }

      newPollValue = 0;
      prim = rctx.sendAndWaitForPrim(
                  new OBS_SET_PULL_PERIOD_REQ(
                    /* source ehId */          scriptEhId,
                    /* pullPeriod */           newPollValue
                  )
                , rctx.type(OBS_PULL_PERIOD_CNF)
                );

      if (newPollValue != prim.getPullPeriod()) {
        logError( "  Expected pullPeriod to be " + newPollValue + " but got " + prim.getPullPeriod() );
        return;
      }

      prim = rctx.sendAndWaitForPrim(
                  new OBS_QUERY_PULL_PERIOD_REQ(
                    /* source ehId */          scriptEhId
                  )
                , rctx.type(OBS_PULL_PERIOD_CNF)
                );

      if (newPollValue != prim.getPullPeriod()) {
        logError( "  Expected pullPeriod to be " + newPollValue + " but got " + prim.getPullPeriod() );
        return;
      }
      logInfo( "Poll Period control validated." );
    }

    {    /// human readable info on observables in the system (for sanity)
      logInfo( "Listing Observable Instances..." );
      var observableInstances;
      prim = rctx.sendAndWaitForPrim(
                  new OBS_QUERY_OBSERVABLE_REQ(
                    /* source ehId */  scriptEhId,
                    /* observableId */ 0
                  )
                , rctx.type(OBS_QUERY_OBSERVABLE_CNF)
                );
      var i;
      for (i = 1; i <= CURRENT_OBSERVABLE_COUNT; ++i) {
        prim = rctx.sendAndWaitForPrim(
                    new OBS_QUERY_OBSERVABLE_REQ(
                      /* source ehId */  scriptEhId,
                      /* observableId */ i
                    )
                  , rctx.type(OBS_QUERY_OBSERVABLE_CNF)
                  );
        var j;
        var bos = new java.io.ByteArrayOutputStream();
        for (j = 0; j < prim.getObservableName().length(); ++j)
        {
          if (0==prim.getObservableName()[j]) break;
          bos.write( prim.getObservableName()[j] );
        }

        logInfo( "  Observable " + i + " is " + bos
                  + " Stamp size is " + prim.getObservableMaxSizeInBytes() );

      }

      logInfo( "Observable Instances listed." );

    }

  }

}




