/*=============================================================================
 *     Copyright Texas Instruments 2001. All Rights Reserved.
 */


/**
 * This is a memory handling class. The class implements the VirtualMemoryHandler
 * methods for reading and writing to memory. This class 
 */
public function HostIntC( regBlock, intTriggerCallback )
{
  var intTrigger = intTriggerCallback;
  
  public function getStartAddr()
  {
    return( regBlock.getStartAddr() );
  }
  
  public function getName()
  {
    return( regBlock.getName() );
  }
  
  public function getStartAddr()
  {
    return( regBlock.getStartAddr() );
  }
  
  public function getEndAddr()
  {
    return( regBlock.getEndAddr() );
  }
  
  /**
   * This function clears an input interrupt
   */
  public function clearInterrupt( sourceId )
  {
    var buf = new Array();
    var reg = regBlock.getRegister("STATUS_0");
    
    /*
     * setup the intC registers to indicate that the source has interrupted
     */
    buf = regBlock.read( ( reg.getAddr() + ( 4 * ( sourceId >> 5 ) ) ),
                         pkg.esf.dev.Register.MEMORY_TYPE_32,
                         1,
                         null );
    
    buf[0] = ( buf[0] & ~( 0x00000001 << ( sourceId & 0x1F ) ) );
    
    regBlock.write( ( reg.getAddr() + ( 4 * ( sourceId >> 5 ) ) ),
                    pkg.esf.dev.Register.MEMORY_TYPE_32,
                    1,
                    buf,
                    null );
    return;
  }
  
  /**
   * This function fires off an IRQ interrupt
   */
  public function dispatchIrqInterrupt( sourceId )
  {
    var buf = new Array();
    var reg = regBlock.getRegister("STATUS_0");
    
    /*
     * setup the intC registers to indicate that the source has interrupted
     */
    buf = regBlock.read( ( reg.getAddr() + ( 4 * ( sourceId >> 5 ) ) ),
                         pkg.esf.dev.Register.MEMORY_TYPE_32,
                         1,
                         null );
    
    buf[0] = ( buf[0] | ( 0x00000001 << ( sourceId & 0x1F ) ) );
    
    regBlock.write( ( reg.getAddr() + ( 4 * ( sourceId >> 5 ) ) ),
                    pkg.esf.dev.Register.MEMORY_TYPE_32,
                    1,
                    buf,
                    null );
    
    /*
     * Inform the target of the interrupt
     */
    intTriggerCallback( 0 );
    return;
  }
  
  /**
   * This function fires off an FIQ interrupt
   */
  public function dispatchFiqInterrupt( sourceId )
  {
    /*
     * setup the intC registers to indicate that the source has interrupted
     */
    
    /*
     * Inform the target of the interrupt
     */
    intTriggerCallback( 1 );
    return;
  }
  
  /*
   * This is where the meat of the Host IntC block emulation stems from.
   * This function will be called with the register name every time
   * a memory location within the register is read. The memHandler
   * will return the value requested, but it hasn't read the memory
   * yet, so it's possible to change the value currently stored in
   * the register before it is read and sent back to the target.
   * 
   * This method is passed as a parameter into the Registers read
   * handling function. It is called from within the RegisterBlock object.
   */
  regBlock.addReadListener( function(reg) {
    
    var regName = reg.getName();
    
    pkg.output.writeln( "Reading " + regName );
    
    /*
     * Reads of a status register is that it clears the status.
     */
    if( regName.indexOf( "STATUS" ) >= 0 )
    {
      pkg.output.writeln( "Clearing "+ regName );
      /*
       * The name of the register contains the string "STATUS"
       */
      reg.setValue( 0 );
    }
    
  } );
  
  /*
   * This is where the meat of the HostIntC block emulation stems from.
   * This function will be called with the register name every time
   * a memory location within the register is written to. The memHandler
   * will write the value requested, but it hasn't written to the memory
   * yet, so it's possible to act on what's about to be written before
   * it is actulally written.
   * 
   * This method is passed as a parameter into the Registers write
   * handling function. It is called from within the RegisterBlock object.
   */
  regBlock.addWriteListener( function( reg, newValue ) {
    
    pkg.output.writeln( "Writing " + newValue + " to "+ reg.getName() );
    
  } );
  
}

/*
 *   Local Variables:
 *   tab-width: 2
 *   indent-tabs-mode: nil
 *   mode: java
 *   c-indentation-style: java
 *   c-basic-offset: 2
 *   eval: (c-set-offset 'statement-cont '0)
 *   eval: (c-set-offset 'substatement-open '0)
 *   eval: (c-set-offset 'case-label '+)
 *   eval: (c-set-offset 'inclass '+)
 *   eval: (c-set-offset 'inline-open '0)
 *   End:
 */
